/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_dynamic_resource.c                                  */
/*  DESCRIPTION:  Dynamic resource solving                                   */
/*                                                                           */
/*  This file defines many types with no obvious best ordering, so we've     */
/*  followed the traditional XHSTT ordering:  times, resources, events       */
/*  (just tasks here), constraints (called expressions here) and solutions;  */
/*  and after those, sets of solutions and solvers.  It is all fully         */
/*  documented in the Appendix to the User's Guide, in the same order.       */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include "khe_priqueue.h"
#include "howard_p.h"
#include <limits.h>
#include <float.h>
#include <math.h>
#include <setjmp.h>

#define TESTING 1
#define COL_WIDTH  12		/* when debugging solutions */
#define COL_COUNT   7		/* when debugging solutions */

#define bool_show(x) ((x) ? "true" : "false")

#define UNDEF_INT -1
#define UNDEF_FLOAT -1.0

/* ***
#define DEBUG_MONITOR_ID "ARC:NA=s1000:1/1Thu:L/NA=s1000:1"
#define DEBUG_MONITOR(m) (false &&					\
  strcmp(KheMonitorId((KHE_MONITOR) m), DEBUG_MONITOR_ID) == 0)
*** */

#define	RERUN  1		/* rerun                        */

#define RERUN_DEBUG(drs)  (false && (drs)->rerun_soln != NULL)
#define RERUN_MONITOR_ID "PreferResourcesRedundancy/1Tue:Night/A=s0+NWTrainee=h1:5"
#define RERUN_PRINT(drs, dm) (RERUN_DEBUG(drs) && (dm) != NULL &&	\
  strcmp(KheDrsMonitorId(dm), RERUN_MONITOR_ID) == 0)
#define ALL_PRINT(drs, dm) (false && (dm) != NULL &&	\
  strcmp(KheDrsMonitorId(dm), RERUN_MONITOR_ID) == 0)

#define DEBUG01 0		/* KheDynamicResourceSolverDoSolve */
#define DEBUG03 0		/* mtasks                       */
#define DEBUG04 0		/* KheDynamicResourceSolverMake */
#define DEBUG07 0		/* ResourceMake                 */
#define DEBUG08 0		/* expression construction      */
#define DEBUG09 0		/* KheDrsSolveOpen (costs)      */
#define DEBUG10 0		/* KheDrsExprOpen               */
#define DEBUG11 0		/* KheDrsResourceOpen           */
#define DEBUG12 0		/* KheDrsSolnExpandMakeAndMeld  */
#define DEBUG14 0		/* int seq cost child has opened*/
#define DEBUG15 0		/* KheDrsTaskAssign             */
#define DEBUG16 0		/* KheTaskUnAssignResource      */
#define DEBUG17 0		/* Search                       */
#define DEBUG18 0		/* KheDrsDayOpen                */
#define DEBUG19 0		/* KheDrsMTaskOpen              */
#define DEBUG20 0		/* KheDrsAUIntervalFindRight    */
#define DEBUG21 0		/* Search statistics            */
#define DEBUG22 0		/* SolnExpand and SolnDoExpand  */
#define DEBUG24 0		/* rerun costs                  */
#define DEBUG25 0		/* dominating and dominated     */
#define DEBUG26 0		/* trie stats                   */
#define DEBUG27 0		/* trie stats                   */
#define DEBUG28 0		/* ancestor and dominates freqs */
#define DEBUG30 0		/* dominance debug              */
#define DEBUG31 0		/* priqueue insert and delete   */
#define DEBUG32 0		/* DistanceToAncestor           */
#define DEBUG34 0		/* prune cost                   */
#define DEBUG35 0		/* two extra selection          */
#define DEBUG36 0		/* dom tables                   */
#define DEBUG37 0		/* dom tables                   */
#define DEBUG40 0		/* one extra dominates          */
#define DEBUG41 0		/* mtask min cost               */
#define DEBUG42 0		/* dom tables                   */
#define DEBUG43 0		/* one extra dominates          */

#define DEBUG44 0		/* sub get                      */
#define DEBUG44_MONITOR_ID "Constraint:4/Nurse1"
#define DEBUG44_MONITOR(m) (DEBUG44 && (m) != NULL && 			\
  strcmp(KheMonitorId((KHE_MONITOR) (m)), DEBUG44_MONITOR_ID) == 0)

#define DEBUG45(day) (0 && ((day) == NULL || (day)->solve_expand_count == 1))
#define DEBUG46 0		/* various                      */
#define DEBUG47(day) (0 && ((day) == NULL || (day)->solve_expand_count == 1))

#define DEBUG49	0
#define DEBUG50	0
#define DEBUG51(day) (0 && (day) == NULL )
#define DEBUG52(day, ds) (0 &&						\
  ((day) == NULL || (day)->solve_expand_count == 1) &&			\
  (ds)->open_shift_index == 27)

#define DEBUG54 0
#define DEBUG55 0
#define DEBUG56 0
#define DEBUG57 0

/* ***
#define DEBUG58(mi)							\
  (false && strcmp(KheDrsMonitorId(mi), "Constraint:9/HN_1") == 0 )
*** */

#define DEBUG59 0
#define DEBUG60 0
#define DEBUG61 0
#define DEBUG63 0
#define DEBUG64 0
#define DEBUG64_ID "1Tue"
#define DEBUG65 0
#define DEBUG66 0
#define DEBUG67 0
#define DEBUG68 0

#define DEBUG69(open_day_index) (false && (open_day_index) == 0)
#define DEBUG70(ddr, odi) (false && (ddr).first <= (odi) && (odi) <= (ddr).last)

#define DEBUG71 0
#define DEBUG72 0
#define DEBUG73 0
#define DEBUG74 0
#define DEBUG75 0
#define DEBUG76 0
#define DEBUG77 0
#define DEBUG78 0
#define DEBUG79 0

#define DEBUG80 0
#define DEBUG80_TASK_ID "1Thu:E.0"

#define DEBUG81 0
#define DEBUG82 0
#define DEBUG83 0
#define DEBUG84 0
#define DEBUG85 0
#define DEBUG86 0
#define DEBUG87 0
#define DEBUG88 0

#define MON_ID "DemandConstraint:1A/EGDC:141"
#define DEBUG89(ec) false && strcmp(KheDrsMonitorId((ec)->monitor), MON_ID) == 0

#define DEBUG90 0

#define DEBUG91_WANTED false
#define DEBUG91_MONITOR_ID "DemandConstraint:3A/EGDC:170"
#define DEBUG91(m) (DEBUG91_WANTED && (m) != NULL && 			\
  strcmp(KheDrsMonitorId(m), DEBUG91_MONITOR_ID) == 0)

#define DEBUG92 0
#define DEBUG93 0

#define DEBUG94_WANTED false
#define DEBUG94_CONSTRAINT_ID "DemandConstraint:3A"
#define DEBUG94(c) (DEBUG94_WANTED && (c) != NULL && 			\
  strcmp(KheDrsConstraintId(c), DEBUG94_CONSTRAINT_ID) == 0)

#define DEBUG95	0

#define DEBUG96_WANTED false
#define DEBUG96_TASK_ID1 "Event854.1"
#define DEBUG96_TASK_ID2 "Event809.1"
#define DEBUG96(dt) (DEBUG96_WANTED &&					\
  (strcmp(KheDrsTaskId(dt), DEBUG96_TASK_ID1) == 0 ||			\
   strcmp(KheDrsTaskId(dt), DEBUG96_TASK_ID2) == 0))

#define DEBUG97 0
#define DEBUG98 0
#define DEBUG99 0

#define USE_DOM_CACHING 0


/*****************************************************************************/
/*                                                                           */
/*  Submodule "forward type declarations"                                    */
/*                                                                           */
/*  This submodule contains forward type declarations, but only if needed.   */
/*                                                                           */
/*****************************************************************************/

/* times */
typedef struct khe_drs_day_rec *KHE_DRS_DAY;
typedef HA_ARRAY(KHE_DRS_DAY) ARRAY_KHE_DRS_DAY;

/* resources */
typedef struct khe_drs_resource_rec *KHE_DRS_RESOURCE;
typedef HA_ARRAY(KHE_DRS_RESOURCE) ARRAY_KHE_DRS_RESOURCE;

typedef struct khe_drs_resource_on_day_rec *KHE_DRS_RESOURCE_ON_DAY;
typedef HA_ARRAY(KHE_DRS_RESOURCE_ON_DAY) ARRAY_KHE_DRS_RESOURCE_ON_DAY;

typedef struct khe_drs_resource_set_rec *KHE_DRS_RESOURCE_SET;
typedef HA_ARRAY(KHE_DRS_RESOURCE_SET) ARRAY_KHE_DRS_RESOURCE_SET;

/* events */
typedef struct khe_drs_task_rec *KHE_DRS_TASK;
typedef HA_ARRAY(KHE_DRS_TASK) ARRAY_KHE_DRS_TASK;

typedef struct khe_drs_task_on_day_rec *KHE_DRS_TASK_ON_DAY;
typedef HA_ARRAY(KHE_DRS_TASK_ON_DAY) ARRAY_KHE_DRS_TASK_ON_DAY;

typedef struct khe_drs_mtask_rec *KHE_DRS_MTASK;
typedef HA_ARRAY(KHE_DRS_MTASK) ARRAY_KHE_DRS_MTASK;

typedef struct khe_drs_shift_rec *KHE_DRS_SHIFT;
typedef HA_ARRAY(KHE_DRS_SHIFT) ARRAY_KHE_DRS_SHIFT;

typedef struct khe_drs_shift_pair_rec *KHE_DRS_SHIFT_PAIR;
typedef HA_ARRAY(KHE_DRS_SHIFT_PAIR) ARRAY_KHE_DRS_SHIFT_PAIR;

/* signatures and dominance */
typedef struct khe_drs_signature_rec *KHE_DRS_SIGNATURE;
typedef HA_ARRAY(KHE_DRS_SIGNATURE) ARRAY_KHE_DRS_SIGNATURE;

typedef struct khe_drs_signature_set_rec *KHE_DRS_SIGNATURE_SET;
typedef HA_ARRAY(KHE_DRS_SIGNATURE_SET) ARRAY_KHE_DRS_SIGNATURE_SET;

typedef struct khe_drs_correlator_rec *KHE_DRS_CORRELATOR;

typedef struct khe_drs_signer_rec *KHE_DRS_SIGNER;
typedef HA_ARRAY(KHE_DRS_SIGNER) ARRAY_KHE_DRS_SIGNER;

typedef struct khe_drs_signer_set_rec *KHE_DRS_SIGNER_SET;
typedef HA_ARRAY(KHE_DRS_SIGNER_SET) ARRAY_KHE_DRS_SIGNER_SET;

typedef struct khe_drs_dom_test_rec *KHE_DRS_DOM_TEST;
typedef HA_ARRAY(KHE_DRS_DOM_TEST) ARRAY_KHE_DRS_DOM_TEST;

/* monitors */
typedef struct khe_drs_monitor_rec *KHE_DRS_MONITOR;
typedef HA_ARRAY(KHE_DRS_MONITOR) ARRAY_KHE_DRS_MONITOR;

/* expressions */
typedef struct khe_drs_expr_rec *KHE_DRS_EXPR;
typedef HA_ARRAY(KHE_DRS_EXPR) ARRAY_KHE_DRS_EXPR;

/* solutions */
typedef struct khe_drs_soln_rec *KHE_DRS_SOLN;
typedef HA_ARRAY(KHE_DRS_SOLN) ARRAY_KHE_DRS_SOLN;
typedef HP_TABLE(KHE_DRS_SOLN) TABLE_KHE_DRS_SOLN;

typedef struct Khe_drs_mtask_soln_rec *KHE_DRS_MTASK_SOLN;
typedef HA_ARRAY(KHE_DRS_MTASK_SOLN) ARRAY_KHE_DRS_MTASK_SOLN;

typedef struct khe_drs_task_soln_rec KHE_DRS_TASK_SOLN;
typedef HA_ARRAY(KHE_DRS_TASK_SOLN) ARRAY_KHE_DRS_TASK_SOLN;

typedef struct khe_drs_task_soln_set *KHE_DRS_TASK_SOLN_SET;
typedef HA_ARRAY(KHE_DRS_TASK_SOLN_SET) ARRAY_KHE_DRS_TASK_SOLN_SET;

typedef struct khe_drs_shift_soln_rec *KHE_DRS_SHIFT_SOLN;
typedef HA_ARRAY(KHE_DRS_SHIFT_SOLN) ARRAY_KHE_DRS_SHIFT_SOLN;

typedef struct khe_drs_shift_soln_trie_rec *KHE_DRS_SHIFT_SOLN_TRIE;
typedef HA_ARRAY(KHE_DRS_SHIFT_SOLN_TRIE) ARRAY_KHE_DRS_SHIFT_SOLN_TRIE;

typedef struct khe_drs_shift_pair_soln_rec *KHE_DRS_SHIFT_PAIR_SOLN;
typedef HA_ARRAY(KHE_DRS_SHIFT_PAIR_SOLN) ARRAY_KHE_DRS_SHIFT_PAIR_SOLN;

/* sets of (day) solutions */
typedef struct khe_drs_soln_list_rec *KHE_DRS_SOLN_LIST;
typedef HA_ARRAY(KHE_DRS_SOLN_LIST) ARRAY_KHE_DRS_SOLN_LIST;
typedef HP_TABLE(KHE_DRS_SOLN_LIST) TABLE_KHE_DRS_SOLN_LIST;

typedef struct khe_drs_soln_set_rec *KHE_DRS_SOLN_SET;
typedef HA_ARRAY(KHE_DRS_SOLN_SET) ARRAY_KHE_DRS_SOLN_SET;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - times"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DAY_RANGE - a range of days                                 */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KHE_INTERVAL
typedef struct khe_drs_day_range_rec {
  int			first;
  int			last;
} KHE_DRS_DAY_RANGE;

typedef HA_ARRAY(KHE_DRS_DAY_RANGE) ARRAY_KHE_DRS_DAY_RANGE;
*** */

typedef HA_ARRAY(KHE_INTERVAL) ARRAY_KHE_INTERVAL;

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DAY - one day                                               */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_day_rec {
  int				frame_index;		/* index in frame    */
  int				open_day_index;		/* index when open   */
  KHE_TIME_GROUP		time_group;		/* day's time group  */
  ARRAY_KHE_DRS_SHIFT		shifts;			/* shifts            */
  KHE_DRS_SIGNER_SET		signer_set;		/* dom tests etc.    */
  KHE_DRS_SOLN_SET		soln_set;		/* today's solns     */
  KHE_DRS_SOLN_LIST		soln_list;		/* sorted soln_set   */
  int				soln_made_count;	/* solns made today  */
  int				solve_expand_count;	/* expanded today    */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - resources"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE_EXPAND_ROLE - resource's role in current expansion */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_RESOURCE_EXPAND_NO,
  KHE_DRS_RESOURCE_EXPAND_FIXED,
  KHE_DRS_RESOURCE_EXPAND_FREE
} KHE_DRS_RESOURCE_EXPAND_ROLE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE - one resource                                     */
/*                                                                           */
/*  A resource is open when it is selected.  However, that matters little,   */
/*  since each of its resource on day objects can be open independently.     */
/*                                                                           */
/*****************************************************************************/
typedef struct khe_drs_dim2_table_rec *KHE_DRS_DIM2_TABLE;

struct khe_drs_resource_rec {
  KHE_RESOURCE			resource;		/* the resource      */
  int				open_resource_index;	/* index when open   */
  ARRAY_KHE_DRS_RESOURCE_ON_DAY	days;			/* doing each day    */
  KHE_DRS_RESOURCE_EXPAND_ROLE	expand_role;		/* current expand    */
  ARRAY_KHE_DRS_SIGNATURE	expand_signatures;	/* current expand    */
  ARRAY_KHE_DRS_MTASK_SOLN	expand_mtask_solns;	/* current expand    */
  KHE_DRS_MTASK_SOLN		expand_free_mtask_soln;	/* current expand    */
  KHE_DRS_DIM2_TABLE		expand_dom_test_cache;	/* current expand    */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE_ON_DAY - what one resource is doing on one day     */
/*                                                                           */
/*  A resource on day is open when its resource is a selected resource, its  */
/*  day is a selected day, and either the resource is unassigned on that     */
/*  day or else there is no impediment to unassigning the task that it is    */
/*  assigned to on that day.                                                 */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_resource_on_day_rec {
  KHE_DRS_RESOURCE		encl_dr;		/* the resource      */
  KHE_DRS_DAY			day;			/* the day           */
  bool				open;			/* true when open    */
  KHE_DRS_TASK_ON_DAY		closed_dtd;		/* asst when closed  */
  KHE_DRS_TASK_ON_DAY		preasst_dtd;		/* asst when preass. */
  ARRAY_KHE_DRS_EXPR		external_today;		/* external today    */
  KHE_DRS_SIGNER		signer;			/* signer            */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE_SET - a set of resources                           */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_resource_set_rec {
  ARRAY_KHE_DRS_RESOURCE	resources;		/* the resources     */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - events"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_TASK_EXPAND_ROLE - task's role in current expansion              */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_TASK_EXPAND_NO_VALUE,
  KHE_DRS_TASK_EXPAND_FIXED,
  KHE_DRS_TASK_EXPAND_MUST,
  KHE_DRS_TASK_EXPAND_FREE
} KHE_DRS_TASK_EXPAND_ROLE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_TASK - one proper root task                                 */
/*                                                                           */
/*  The days are sorted chronologically, and this is used to test whether    */
/*  a given task on day is the first of its task.                            */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_task_rec {
  KHE_DRS_MTASK			encl_dmt;
  int				index_in_encl_dmt;
  KHE_DRS_TASK_EXPAND_ROLE	expand_role;		/* used by SolnExpand*/
  bool				open;			/* true when open    */
  KHE_TASK			task;			/* the KHE task      */
  KHE_DRS_RESOURCE		closed_dr;		/* asst when closed  */
  KHE_COST			non_asst_cost;		/* non-asst cost     */
  KHE_COST			asst_cost;		/* assignment cost   */
  ARRAY_KHE_DRS_TASK_ON_DAY	days;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_TASK_ON_DAY - what one proper root task is doing on one day */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_task_on_day_rec {
  KHE_DRS_TASK			encl_dt;		/* enclosing task    */
  KHE_DRS_DAY			day;			/* the day           */
  bool				not_last_day;		/* not encl_dl's last*/
  KHE_TASK			task;			/* atomic task today */
  KHE_TIME			time;			/* time on day       */
  KHE_DRS_RESOURCE_ON_DAY	closed_drd;		/* asst when closed  */
  ARRAY_KHE_DRS_EXPR		external_today;		/* leaf today        */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_MTASK - a set of tasks, taken from one KHE_MTASK            */
/*                                                                           */
/*  For efficiency, available_tasks is managed lazily.  Every unassigned     */
/*  task is in it (except not fixed tasks), but not in order, and there      */
/*  may be assigned tasks in it.  These defects are corrected when the       */
/*  mtask is opened for solving.                                             */
/*                                                                           */
/*  Field expand_prev_unfixed is the index in available_tasks of the         */
/*  unfixed task most recently given away to expansion, or -1 if no tasks    */
/*  are currently given away, or a large number if there are none to give    */
/*  away.                                                                    */
/*                                                                           */
/*  Obsolete:                                                                */
/*  Field expand_next_unfixed is the index in available_tasks of the next    */
/*  available unfixed unassigned task, unless it points beyond the end of    */
/*  available_tasks, in which case there is no such task.                    */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_mtask_rec {
  KHE_MTASK			orig_mtask;		/* origin of this    */
  KHE_DRS_SHIFT			encl_shift;		/* enclosing shift   */
  KHE_INTERVAL			day_range;		/* days running      */
  ARRAY_KHE_DRS_TASK		all_tasks;		/* all tasks in mtask*/
  ARRAY_KHE_DRS_TASK		available_tasks;	/* unassigned tasks  */
  int				expand_must_assign_count;
  int				expand_prev_unfixed;	/* used by SolnExpand*/
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SHIFT - the set of all mtasks with the same times           */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_shift_rec {
  KHE_DRS_DAY			encl_day;		/* enclosing day     */
  int				open_shift_index;	/* set when open     */
  int				expand_must_assign_count;
  int				expand_max_included_free_resource_count;
  ARRAY_KHE_DRS_MTASK		mtasks;			/* all mtasks        */
  ARRAY_KHE_DRS_MTASK		open_mtasks;		/* set when open     */
  ARRAY_KHE_DRS_SHIFT_PAIR	shift_pairs;		/* with self         */
  KHE_DRS_SIGNER		signer;			/* signer            */
  KHE_DRS_SHIFT_SOLN_TRIE	soln_trie;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SHIFT_PAIR - a pair of shifts starting on the same day      */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_shift_pair_rec {
  KHE_DRS_SHIFT			shift[2];		/* the two shifts    */
  KHE_DRS_SIGNER		signer;			/* signer            */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - dominance"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DOM_KIND - the kind of dominance testing to use             */
/*                                                                           */
/*****************************************************************************/

/* *** defined in khe_solvers.h
typedef enum {
  KHE_DRS_DOM_LIST_NONE,
  KHE_DRS_DOM_LIST_SEPARATE,
  KHE_DRS_DOM_LIST_TRADEOFF,
  KHE_DRS_DOM_LIST_TABULATED,
  KHE_DRS_DOM_HASH_EQUALITY,
  KHE_DRS_DOM_HASH_MEDIUM,
  KHE_DRS_DOM_TRIE_SEPARATE,
  KHE_DRS_DOM_TRIE_TRADEOFF,
  KHE_DRS_DOM_INDEXED_TRADEOFF,
  KHE_DRS_DOM_INDEXED_TABULATED
} KHE_DRS_DOM_KIND;
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_COST_TUPLE - unweighted costs stored in a dominance table   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_cost_tuple_rec {
  short				unweighted_psi;
  short				unweighted_psi0;
  /* bool			psi_plus_defined; deleted to save memory */
  short				unweighted_psi_plus;
} KHE_DRS_COST_TUPLE;

typedef HA_ARRAY(KHE_DRS_COST_TUPLE) ARRAY_KHE_DRS_COST_TUPLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM1_TABLE - a one-dimensional table of tuples                   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_dim1_table_rec {
  ARRAY_KHE_DRS_COST_TUPLE	children;
  int				offset;
} *KHE_DRS_DIM1_TABLE;

typedef HA_ARRAY(KHE_DRS_DIM1_TABLE) ARRAY_KHE_DRS_DIM1_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM2_TABLE - a two-dimensional table of tuples                   */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_dim2_table_rec {
  ARRAY_KHE_DRS_DIM1_TABLE	children;
  int				offset;
  /* char			*debug_str; */
};

typedef HA_ARRAY(KHE_DRS_DIM2_TABLE) ARRAY_KHE_DRS_DIM2_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM3_TABLE - a three-dimensional table of tuples                 */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_dim3_table_rec {
  ARRAY_KHE_DRS_DIM2_TABLE	children;
  int				offset;
  /* char			*debug_str; */
} *KHE_DRS_DIM3_TABLE;

typedef HA_ARRAY(KHE_DRS_DIM3_TABLE) ARRAY_KHE_DRS_DIM3_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM4_TABLE - a four-dimensional table of tuples                  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_dim4_table_rec {
  ARRAY_KHE_DRS_DIM3_TABLE	children;
  int				offset;
  /* char			*debug_str; */
} *KHE_DRS_DIM4_TABLE;

typedef HA_ARRAY(KHE_DRS_DIM4_TABLE) ARRAY_KHE_DRS_DIM4_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM5_TABLE - a five-dimensional table of tuples                  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_dim5_table_rec {
  ARRAY_KHE_DRS_DIM4_TABLE	children;
  int				offset;
  /* char			*debug_str; */
} *KHE_DRS_DIM5_TABLE;

typedef HA_ARRAY(KHE_DRS_DIM5_TABLE) ARRAY_KHE_DRS_DIM5_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_VALUE - the value of an expression; could be int or float   */
/*                                                                           */
/*****************************************************************************/

typedef union {
  int			i;			/* value, if int   */
  float			f;			/* value, if float */
} KHE_DRS_VALUE;

typedef HA_ARRAY(KHE_DRS_VALUE) ARRAY_KHE_DRS_VALUE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNATURE - a signature                                     */
/*                                                                           */
/*  We use reference counting to determine whether a signature object        */
/*  needs to be kept or not.  The reference count contains the number of     */
/*  references to the signature from heap objects.                           */
/*                                                                           */
/*  The places where signatures are referred to by heap objects are:         */
/*                                                                           */
/*    KHE_DRS_RESOURCE.expand_signatures                                     */
/*    KHE_DRS_SHIFT_SOLN.sig                                                 */
/*    KHE_DRS_MTASK_SOLN.sig                                                 */
/*    KHE_DRS_SIGNATURE_SET.signatures                                       */
/*                                                                           */
/*  It is the responsibility of these other types to keep the reference      */
/*  counts up to date.                                                       */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_signature_rec {
  int			reference_count;	/* reference count           */
  int			asst_to_shift_index;	/* index of enclosing a_to_s */
  KHE_COST		cost;			/* cost                      */
  ARRAY_KHE_DRS_VALUE	states;			/* expr states               */
  /* HA_ARRAY_SHORT	states;	*/		/* expr states               */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNATURE_SET - a set of signatures                         */
/*                                                                           */
/*  KHE_DRS_SIGNATURE_SET is a pointer type as usual, but in KHE_DRS_SOLN    */
/*  it is expanded ("struct khe_drs_signature_set_rec") to save memory.      */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_signature_set_rec {
  KHE_COST			cost;
  ARRAY_KHE_DRS_SIGNATURE	signatures;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_CORRELATOR - an expression correlator                       */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_correlator_rec {
  bool				make_correlated;
  HA_ARRAY_INT			children;
  HA_ARRAY_INT			positives;
  HA_ARRAY_INT			negatives;
  HA_ARRAY_INT			singles;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNER - a signature controller                             */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_SIGNER_DAY,
  KHE_DRS_SIGNER_RESOURCE_ON_DAY,
  KHE_DRS_SIGNER_SHIFT,
  KHE_DRS_SIGNER_SHIFT_PAIR
} KHE_DRS_SIGNER_TYPE;

struct khe_drs_signer_rec {
  KHE_DYNAMIC_RESOURCE_SOLVER	solver;			/* solver            */
  ARRAY_KHE_DRS_EXPR		internal_exprs;		/* that affect this  */
  ARRAY_KHE_DRS_DOM_TEST	dom_tests;
  HA_ARRAY_INT			eq_dom_test_indexes;	/* where = tests are */
  int				last_hard_cost_index;
  /* bool			debug_eval_if_rerun; */	/* debug evals       */
  KHE_DRS_CORRELATOR		correlator;		/* always present    */
  KHE_DRS_DAY			encl_day;		/*first incomplete di*/
  KHE_DRS_SIGNER_TYPE		type;
  union {
    KHE_DRS_DAY			day;
    KHE_DRS_RESOURCE_ON_DAY	resource_on_day;
    KHE_DRS_SHIFT		shift;
    KHE_DRS_SHIFT_PAIR		shift_pair;
  } u;
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNER_TEST - the signer tests to carry out (hard, soft, or all) */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_SIGNER_HARD,
  KHE_DRS_SIGNER_SOFT,
  KHE_DRS_SIGNER_ALL
} KHE_DRS_SIGNER_TEST;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNER_SET - a set of signers                               */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_signer_set_rec {
  ARRAY_KHE_DRS_SIGNER		signers;
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE - type of dom test                                 */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_DOM_TEST_UNUSED,			/* unused                    */
  KHE_DRS_DOM_TEST_SEPARATE_GENERIC,		/* separate (unspecified)    */
  KHE_DRS_DOM_TEST_SEPARATE_INT,		/* separate (int)            */
  KHE_DRS_DOM_TEST_SEPARATE_FLOAT,		/* separate (float)          */
  KHE_DRS_DOM_TEST_TRADEOFF,			/* tradeoff                  */
  KHE_DRS_DOM_TEST_TABULATED,			/* tabulated                 */
  KHE_DRS_DOM_TEST_CORR1_PARENT,		/* parent corr1              */
  KHE_DRS_DOM_TEST_CORR1_CHILD,			/* child corr1               */
  KHE_DRS_DOM_TEST_CORR2_CHILD,			/* child corr2               */
  KHE_DRS_DOM_TEST_CORR3_FIRST,			/* first corr3               */
  KHE_DRS_DOM_TEST_CORR3_MID,			/* in the middle             */
  KHE_DRS_DOM_TEST_CORR3_LAST,			/* last corr3                */
  KHE_DRS_DOM_TEST_CORR4_FIRST,			/* first corr4               */
  KHE_DRS_DOM_TEST_CORR4_MID,			/* in the middle             */
  KHE_DRS_DOM_TEST_CORR4_LAST			/* last corr4                */
} KHE_DRS_DOM_TEST_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DOM_TEST - a dominance test                                 */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_dom_test_rec {
  KHE_DRS_DOM_TEST_TYPE	type;			/* type of test              */
  int			correlated_delta;	/* if correlated             */
  KHE_DRS_EXPR		expr;			/* originating expression    */
  bool			allow_zero;		/* allow zero ("Z" in doc)   */
  /* int		min_limit; */           /* min limit (0 if none)     */
  /* int		max_limit; */           /* max limit (INT_MAX if ne) */
  KHE_DRS_VALUE		a;			/* "a" in doc                */
  KHE_DRS_VALUE		b;			/* "b" in doc                */
  KHE_COST		combined_weight;	/* tradeoff cost, if tradeoff*/
  KHE_DRS_DIM2_TABLE	main_dom_table2;	/* tabulated dominance table */
  KHE_DRS_DIM4_TABLE	corr_dom_table4;	/* tabulated dominance table */
  KHE_DRS_MONITOR	monitor;		/* monitor, if any           */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - constraints'                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_CONSTRAINT - one constraint from the instance               */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_constraint_rec {
  KHE_CONSTRAINT		constraint;		/* the constraint    */
  int				min_history;		/* smallest history  */
  int				max_history;		/* largest history   */
  int				max_child_count;	/* max child count   */
  bool				needs_corr_table;	/* true if any do    */
  KHE_DRS_EXPR			sample_expr;		/* expression        */
  KHE_DRS_DIM3_TABLE		counter_main_dom_table3;  /* dom tables      */
  KHE_DRS_DIM5_TABLE		counter_corr_dom_table5;
  KHE_DRS_DIM5_TABLE		sequence_main_dom_table5;
} *KHE_DRS_CONSTRAINT;

typedef HA_ARRAY(KHE_DRS_CONSTRAINT) ARRAY_KHE_DRS_CONSTRAINT;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_MONITOR - one monitor                                       */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_monitor_rec {
  KHE_MONITOR		monitor;
  KHE_COST		rerun_open_and_search_cost;
  KHE_COST		rerun_open_and_close_cost;
  KHE_DRS_EXPR		sample_expr;
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - expressions"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_OP - an open, search, or close operation type               */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_OPEN,
  KHE_DRS_SEARCH,
  KHE_DRS_CLOSE
} KHE_DRS_OP;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_TAG                                                    */
/*                                                                           */
/*  Tags for expression types.                                               */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_EXPR_ASSIGNED_TASK_TAG,
  KHE_DRS_EXPR_BUSY_TIME_TAG,
  KHE_DRS_EXPR_FREE_TIME_TAG,
  KHE_DRS_EXPR_WORK_TIME_TAG,
  KHE_DRS_EXPR_BUSY_DAY_TAG,
  KHE_DRS_EXPR_FREE_DAY_TAG,
  KHE_DRS_EXPR_WORK_DAY_TAG,
  KHE_DRS_EXPR_OR_TAG,
  KHE_DRS_EXPR_AND_TAG,	
  /* ***
  KHE_DRS_EXPR_INT_SUM_TAG,
  KHE_DRS_EXPR_FLOAT_SUM_TAG,
  KHE_DRS_EXPR_INT_DEV_TAG,
  KHE_DRS_EXPR_FLOAT_DEV_TAG,
  *** */
  KHE_DRS_EXPR_COUNTER_TAG,
  KHE_DRS_EXPR_SUM_INT_TAG,
  KHE_DRS_EXPR_SUM_FLOAT_TAG,
  KHE_DRS_EXPR_SEQUENCE_TAG
} KHE_DRS_EXPR_TAG;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_ADJUST_TYPE - types of signature value adjustments          */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_ADJUST_ORDINARY,
  KHE_DRS_ADJUST_NO_MAX,
  KHE_DRS_ADJUST_LINEAR,
  KHE_DRS_ADJUST_STEP
} KHE_DRS_ADJUST_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_EVAL_TYPE - type of expression evaluation required     */
/*                                                                           */
/*    KHE_DRS_EXPR_EVAL_NO                                                   */
/*                                                                           */
/*      No evaluation is required, and no dominance test is needed either.   */
/*                                                                           */
/*    KHE_DRS_EXPR_EVAL_NOT_LAST                                             */
/*                                                                           */
/*      Evaluation is required.  This is not the last time that this         */
/*      expression will be evaluated, so a position in the state array       */
/*      has to be reserved for its state, and a dominance test is needed.    */
/*                                                                           */
/*    KHE_DRS_EXPR_EVAL_LAST                                                 */
/*                                                                           */
/*      Evaluation is required.  This is the last time that this exprssion   */
/*      will be evaluated, so a position in the state array is not needed    */
/*      to hold its state, and no dominance test is needed either.           */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_EXPR_EVAL_NO,
  KHE_DRS_EXPR_EVAL_NOT_LAST,
  KHE_DRS_EXPR_EVAL_LAST
} KHE_DRS_EXPR_EVAL_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_PARENT - a expression's parent                              */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_parent_rec {
  KHE_DRS_EXPR		expr;			/* the parent expression */
  int			index;			/* index in parent       */
} KHE_DRS_PARENT;

typedef HA_ARRAY(KHE_DRS_PARENT) ARRAY_KHE_DRS_PARENT;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_OPEN_CHILD - one open child                                      */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_open_child_rec {
  KHE_DRS_EXPR		child_e;		/* the open child itself     */
  int			open_index;		/* open index of this child  */
  KHE_DRS_VALUE		rev_cum_value_ub;	/* reverse cumul. value ub   */
} KHE_DRS_OPEN_CHILD;

typedef HA_ARRAY(KHE_DRS_OPEN_CHILD) ARRAY_KHE_DRS_OPEN_CHILD;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_OPEN_CHILDREN_INDEX_TYPE - what open children are indexed by     */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_OPEN_CHILDREN_INDEX_DAY,
  KHE_DRS_OPEN_CHILDREN_INDEX_DAY_ADJUSTED,
  KHE_DRS_OPEN_CHILDREN_INDEX_SHIFT
} KHE_DRS_OPEN_CHILDREN_INDEX_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_OPEN_CHILDREN - a sequence of open children                      */
/*                                                                           */
/*  oc->open_children                                                        */
/*    A sequence of children sorted by the "open index" defined below.       */
/*    Each child contains fields open_child holding the child expression,    */
/*    open_index holding the open index, and rev_cum_value_ub holding the    */
/*    sum, over all open children from here to the end, of the value upper   */
/*    bounds of the children.  We call this "reverse cumulative".            */
/*                                                                           */
/*  oc->index_range                                                          */
/*    The range of open indexes of the children.  If there is at least one   */
/*    child this range must be non-empty; if there are no children it must   */
/*    be empty.                                                              */
/*                                                                           */
/*  oc->index_type                                                           */
/*    The type of the open index, one of                                     */
/*                                                                           */
/*      KHE_DRS_OPEN_CHILDREN_INDEX_DAY                                      */
/*        The open index is the last element of the child's open day range.  */
/*                                                                           */
/*      KHE_DRS_OPEN_CHILDREN_INDEX_DAY_ADJUSTED                             */
/*        Like KHE_DRS_OPEN_CHILDREN_INDEX_DAY, except that the range is     */
/*        adjusted so that its last day does not precede the previous        */
/*        child's last day.                                                  */
/*                                                                           */
/*      KHE_DRS_OPEN_CHILDREN_INDEX_SHIFT                                    */
/*        The index of the shift containing the assigned task of the         */
/*        child (in this case, the child always monitors an assigned task).  */
/*                                                                           */
/*  oc->child_indexes                                                        */
/*    An array of indexes into the oc->open_children array, such that for    */
/*    each open index oi of a child in the array,                            */
/*                                                                           */
/*      oc->child_indexes[oi - oc->index_range.first]                        */
/*                                                                           */
/*  is the index in oc->children of the first open child x such that the     */
/*  open index of x is oi or more.  Equivalently, it is the number of        */
/*  children x whose open index is strictly less than oi, as returned by     */
/*                                                                           */
/*    int KheDrsOpenChildrenBefore(KHE_DRS_OPEN_CHILDREN oc, int index)      */
/*                                                                           */
/*  There is also a sentinel at the end of oc->child_indexes holding the     */
/*  total number of children, so that for any open index oi, the iterator    */
/*                                                                           */
/*    #define KheDrsOpenChildrenForEach(oc, oi, x, i)                   \    */
/*      i1 = KheDrsOpenChildrenBefore((oc), (oi));                      \    */
/*      i2 = KheDrsOpenChildrenBefore((oc), (oi) + 1);                  \    */
/*      for( (i) = i1;                                                  \    */
/*           (i) < i2 ? ((x) = HaArray((oc)->children, (i)), true) : false;\ */
/*           (i)++ )                                                         */
/*                                                                           */
/*  sets x to each child in turn whose open index is oi.                     */
/*                                                                           */
/*  Obsolete:                                                                */
/*  Finally, oc->upper_bounds[index] is the sum of the value upper bounds    */
/*  of the children whose open index is index or more, where index satisfies */
/*                                                                           */
/*      oc->range_first <= index <= oc->range_last + 1                       */
/*                                                                           */
/*  If index == oc->range_last + 1 the result is 0.                          */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_open_children_rec {
  ARRAY_KHE_DRS_OPEN_CHILD		open_children;
  KHE_INTERVAL				index_range;
  KHE_DRS_OPEN_CHILDREN_INDEX_TYPE	index_type;
  bool					float_ub;
  HA_ARRAY_INT				child_indexes;
} *KHE_DRS_OPEN_CHILDREN;


/*****************************************************************************/
/*                                                                           */
/*  INHERIT_KHE_DRS_EXPR - fields inherited by all expression types          */
/*                                                                           */
/*****************************************************************************/

#define INHERIT_KHE_DRS_EXPR						\
  KHE_DRS_EXPR_TAG	tag;			/* tag field      */	\
  bool			gathered;		/* for opening    */	\
  bool			open;			/* open           */	\
  int			postorder_index;	/* expr index     */	\
  KHE_DRS_RESOURCE	resource;		/* if resource    */	\
  KHE_DRS_VALUE		value;			/* value          */	\
  KHE_DRS_VALUE		value_ub;		/* ub on value    */	\
  ARRAY_KHE_DRS_PARENT	parents;		/* parent exprs   */	\
  ARRAY_KHE_DRS_EXPR	children;		/* child exprs    */	\
  struct khe_drs_open_children_rec open_children_by_day; /* sorted by day  */ \
  HA_ARRAY_INT		sig_indexes;		/* for each day   */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR - abstract supertype of expressions                    */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_expr_rec {
  INHERIT_KHE_DRS_EXPR
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_ASSIGNED_TASK                                          */
/*                                                                           */
/*  An expression derived from a task on day, whose value is 1 if the task   */
/*  is assigned a resource from the given resource group, and 0 otherwise.   */
/*                                                                           */
/*  We include the task on day because we merge equal expressions of this    */
/*  kind, and that is only possible when the task is in the expression.      */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_assigned_task_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_TASK_ON_DAY		task_on_day;
  KHE_RESOURCE_GROUP		resource_group;
} *KHE_DRS_EXPR_ASSIGNED_TASK;

typedef HA_ARRAY(KHE_DRS_EXPR_ASSIGNED_TASK) ARRAY_KHE_DRS_EXPR_ASSIGNED_TASK;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_BUSY_TIME                                              */
/*                                                                           */
/*  An expression whose value is 1 when the resource is busy at a given      */
/*  time, and 0 when the resource is free then.                              */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_busy_time_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME			time;		/* the time we're busy at    */
} *KHE_DRS_EXPR_BUSY_TIME;

typedef HA_ARRAY(KHE_DRS_EXPR_BUSY_TIME) ARRAY_KHE_DRS_EXPR_BUSY_TIME;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_FREE_TIME                                              */
/*                                                                           */
/*  An expression whose value is 1 when the resource is free at a given      */
/*  time, and 0 when the resource is busy then.  The expression stores the   */
/*  time, but only for debugging.                                            */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_free_time_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME			time;		/* the time we're free at    */
} *KHE_DRS_EXPR_FREE_TIME;

typedef HA_ARRAY(KHE_DRS_EXPR_FREE_TIME) ARRAY_KHE_DRS_EXPR_FREE_TIME;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_WORK_TIME                                              */
/*                                                                           */
/*  An expression whose value is the workload incurred by the resource at    */
/*  a given time.  This will be 0.0 if the resource is free then.            */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_work_time_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME			time;		/* the time of this workload */
} *KHE_DRS_EXPR_WORK_TIME;

typedef HA_ARRAY(KHE_DRS_EXPR_WORK_TIME) ARRAY_KHE_DRS_EXPR_WORK_TIME;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_BUSY_DAY                                               */
/*                                                                           */
/*  An expression whose value is 1 when the resource is busy on a given      */
/*  day, and 0 when the resource is free then.  The expression stores the    */
/*  day (as a time group), and not just for debugging.                       */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_busy_day_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME_GROUP		time_group;	/* the day we're busy at     */
} *KHE_DRS_EXPR_BUSY_DAY;

typedef HA_ARRAY(KHE_DRS_EXPR_BUSY_DAY) ARRAY_KHE_DRS_EXPR_BUSY_DAY;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_FREE_DAY                                               */
/*                                                                           */
/*  An expression whose value is 1 when the resource is free on a given      */
/*  day, and 0 when the resource is busy then.  The expression stores the    */
/*  day (as a time group), and not just for debugging.                       */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_free_day_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME_GROUP		time_group;	/* the day we're free at     */
} *KHE_DRS_EXPR_FREE_DAY;

typedef HA_ARRAY(KHE_DRS_EXPR_FREE_DAY) ARRAY_KHE_DRS_EXPR_FREE_DAY;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_WORK_DAY                                               */
/*                                                                           */
/*  An expression whose value is the workload incurred by the resource on    */
/*  a given day.  This will be 0.0 if the resource is free then.  The        */
/*  expression stores the day (as a time group), but only for debugging.     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_work_day_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME_GROUP		time_group;	/* the day being monitored   */
} *KHE_DRS_EXPR_WORK_DAY;

typedef HA_ARRAY(KHE_DRS_EXPR_WORK_DAY) ARRAY_KHE_DRS_EXPR_WORK_DAY;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_OR                                                     */
/*                                                                           */
/*  An expression whose value is the logical `or' of its children's values.  */
/*  Here Booleans are represented by integers (0 for false, 1 for true).     */
/*                                                                           */
/*  The closed state here is the number of closed children with value 1.     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_or_rec {
  INHERIT_KHE_DRS_EXPR
  int				closed_state;
} *KHE_DRS_EXPR_OR;

typedef HA_ARRAY(KHE_DRS_EXPR_OR) ARRAY_KHE_DRS_EXPR_OR;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_AND                                                    */
/*                                                                           */
/*  An expression whose value is the logical `and' of its children's values. */
/*  Here Booleans are represented by integers (0 for false, 1 for true).     */
/*                                                                           */
/*  The closed state here is the number of closed children with value 0.     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_and_rec {
  INHERIT_KHE_DRS_EXPR
  int				closed_state;
} *KHE_DRS_EXPR_AND;

typedef HA_ARRAY(KHE_DRS_EXPR_AND) ARRAY_KHE_DRS_EXPR_AND;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_INT_SUM                                                */
/*                                                                           */
/*  An expression whose value is the sum of its children's values.           */
/*                                                                           */
/*  The closed state here is the sum of the values of the closed children.   */
/*                                                                           */
/*****************************************************************************/

/* ***
typedef struct khe_drs_expr_int_sum_rec {
  INHERIT_KHE_DRS_EXPR
  int				closed_state;
} *KHE_DRS_EXPR_INT_SUM;
*** */

/* typedef HA_ARRAY(KHE_DRS_EXPR_INT_SUM) ARRAY_KHE_DRS_EXPR_INT_SUM; */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_FLOAT_SUM                                              */
/*                                                                           */
/*  An expression whose value is the sum of its children's values.  The      */
/*  expression's value, and its children's values, have type float.          */
/*                                                                           */
/*  The closed state here is the sum of the values of the closed children.   */
/*                                                                           */
/*****************************************************************************/

/* ***
typedef struct khe_drs_expr_float_sum_rec {
  INHERIT_KHE_DRS_EXPR
  float				closed_state;
} *KHE_DRS_EXPR_FLOAT_SUM;
*** */

/* typedef HA_ARRAY(KHE_DRS_EXPR_FLOAT_SUM) ARRAY_KHE_DRS_EXPR_FLOAT_SUM; */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_INT_DEV                                                */
/*                                                                           */
/*  An expression whose value is the deviation of its integer-valued child.  */
/*                                                                           */
/*  The closed state here is the value of the closed child.  It is not       */
/*  stored here explicitly; it is retrieved from the child when required.    */
/*                                                                           */
/*****************************************************************************/

/* ***
typedef struct khe_drs_expr_int_dev_rec {
  INHERIT_KHE_DRS_EXPR
  int				min_limit;	** minimum limit             **
  int				max_limit;	** maximum limit             **
  bool				allow_zero;	** true if 0 has deviation 0 **
} *KHE_DRS_EXPR_INT_DEV;

typedef HA_ARRAY(KHE_DRS_EXPR_INT_DEV) ARRAY_KHE_DRS_EXPR_INT_DEV;
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_FLOAT_DEV                                              */
/*                                                                           */
/*  An expression whose value is the integer-valued deviation of its         */
/*  float-valued child.                                                      */
/*                                                                           */
/*  The closed state here is the value of the closed child.  It is not       */
/*  stored here explicitly; it is retrieved from the child when required.    */
/*                                                                           */
/*****************************************************************************/

/* ***
typedef struct khe_drs_expr_float_dev_rec {
  INHERIT_KHE_DRS_EXPR
  int				min_limit;	** minimum limit             **
  int				max_limit;	** maximum limit             **
  bool				allow_zero;	** true if 0 has deviation 0 **
} *KHE_DRS_EXPR_FLOAT_DEV;

typedef HA_ARRAY(KHE_DRS_EXPR_FLOAT_DEV) ARRAY_KHE_DRS_EXPR_FLOAT_DEV;
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_COST                                                   */
/*                                                                           */
/*  This type is the abstract supertype of KHE_DRS_EXPR_COUNTER,             */
/*  KHE_DRS_EXPR_SUM_INT, KHE_DRS_EXPR_SUM_FLOAT, KHE_DRS_EXPR_SEQUENCE.     */
/*  It is not used much.                                                     */
/*                                                                           */
/*****************************************************************************/

#define INHERIT_KHE_DRS_EXPR_COST					   \
  INHERIT_KHE_DRS_EXPR							   \
  KHE_COST_FUNCTION	cost_fn;		/* cost function        */ \
  KHE_COST		combined_weight;	/* combined weight      */ \
  KHE_DRS_MONITOR	monitor;		/* monitor              */

typedef struct khe_drs_expr_cost_rec {
  INHERIT_KHE_DRS_EXPR_COST
} *KHE_DRS_EXPR_COST;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_COUNTER                                                */
/*                                                                           */
/*  Combines INT_SUM, INT_DEV, and COST into a single expression.  This      */
/*  allows important optimizations in some cases, depending on adjust_type.  */
/*                                                                           */
/*  The closed state here is the sum of the values of the closed children.   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_counter_rec {
  INHERIT_KHE_DRS_EXPR_COST
  int				min_limit;		/* minimum limit     */
  int				max_limit;		/* maximum limit     */
  bool				allow_zero;		/* allow zero flag   */
  int				history_before;		/* history before    */
  int				history_after;		/* history after     */
  int				history;		/* history value     */
  KHE_DRS_ADJUST_TYPE		adjust_type;		/* type of adjustment*/
  int				closed_state;		/* closed state      */
  struct khe_drs_open_children_rec open_children_by_shift; /* sorted by shift*/
} *KHE_DRS_EXPR_COUNTER;

/*typedef HA_ARRAY(KHE_DRS_EXPR_COUNTER)ARRAY_KHE_DRS_EXPR_COUNTER;*/


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_SUM_INT                                                */
/*                                                                           */
/*  The value of an expression of this type is the sum of the values of      */
/*  its children (all integer-valued), converted to an integer deviation     */
/*  by comparison with limits and allow_zero.  This type is also capable     */
/*  of reporting a cost, if requested (if combined_weight > 0).              */
/*                                                                           */
/*  The closed state here is the sum of the values of the closed children.   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_sum_int_rec {
  INHERIT_KHE_DRS_EXPR_COST
  int			min_limit;			/* minimum limit     */
  int			max_limit;			/* maximum limit     */
  int			history_before;			/* history before    */
  int			history_after;			/* history after     */
  int			history;			/* history value     */
  KHE_DRS_ADJUST_TYPE	adjust_type;			/* type of adjustment*/
  int			closed_state;			/* closed state      */
  bool			allow_zero;			/* allow zero flag   */
} *KHE_DRS_EXPR_SUM_INT;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_SUM_FLOAT                                              */
/*                                                                           */
/*  Identical to KHE_DRS_EXPR_SUM_INT except that it and its children have   */
/*  values of type float.  The deviation still has type int.                 */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_sum_float_rec {
  INHERIT_KHE_DRS_EXPR_COST
  float			min_limit;			/* minimum limit     */
  float			max_limit;			/* maximum limit     */
  float			history_before;			/* history before    */
  float			history_after;			/* history after     */
  float			history;			/* history value     */
  KHE_DRS_ADJUST_TYPE	adjust_type;			/* type of adjustment*/
  float			closed_state;			/* closed state      */
  bool			allow_zero;			/* allow zero flag   */
} *KHE_DRS_EXPR_SUM_FLOAT;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_CLOSED_SEQ                                                  */
/*                                                                           */
/*  A sequence of consecutive closed children, summarized.  The fields are   */
/*                                                                           */
/*    start_index                                                            */
/*      The index in the sequence of all children where the sequence starts. */
/*                                                                           */
/*    stop_index                                                             */
/*      The index in the sequence of all children where the sequence         */
/*      stops; that is, the first child index not included in it.  So        */
/*      start_index <= stop_index, and when they are equal, the sequence     */
/*      is empty.                                                            */
/*                                                                           */
/*    active_at_left                                                         */
/*      The length of the a-interval at the extreme left, or 0 if none.      */
/*                                                                           */
/*    active_at_right                                                        */
/*      The length of the a-interval at the extreme right, or 0 if none.     */
/*                                                                           */
/*  If there is a single a-interval running the full length of the           */
/*  closed sequence (if all its children are active), then                   */
/*                                                                           */
/*    active_at_left == active_at_right == stop_index - start_index          */
/*                                                                           */
/*****************************************************************************/

typedef struct {
  int		start_index;
  int		stop_index;
  int		active_at_left;
  int		active_at_right;
} *KHE_DRS_CLOSED_SEQ;

typedef HA_ARRAY(KHE_DRS_CLOSED_SEQ) ARRAY_KHE_DRS_CLOSED_SEQ;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_A_INTERVAL - one a-interval                                 */
/*                                                                           */
/*  This may include indexes from history.                                   */
/*                                                                           */
/*  Note.  If the length of ai is 0, it does not matter whether an           */
/*  unassigned child precedes it or not, because its cost will be 0          */
/*  anyway.  However we must keep the value correct because there may        */
/*  be a merge later that makes it relevant.                                 */
/*                                                                           */
/*****************************************************************************/

typedef struct {
  int		start_index;		/* index of first child              */
  int		stop_index;		/* index after last child            */
  bool		unassigned_precedes;	/* true if unassigned child precedes */
} KHE_DRS_A_INTERVAL;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_AU_INTERVAL - one au-interval                               */
/*                                                                           */
/*  This may include indexes from history and history_after.                 */
/*                                                                           */
/*****************************************************************************/

typedef struct {
  int		start_index;		/* index of first child              */
  int		stop_index;		/* index after last child            */
  bool		has_active_child;	/* true if contains an active child  */
} KHE_DRS_AU_INTERVAL;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SEQ_TYPE - whether int seq type echoes whole frame               */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_SEQ_NONE,				/* none of what follows      */
  KHE_DRS_SEQ_DAYS_POSITIVE,			/* all days, all positive    */
  KHE_DRS_SEQ_DAYS_NEGATIVE,			/* all days, all negative    */
  KHE_DRS_SEQ_SINGLE_POSITIVE			/* single times, all days    */
} KHE_DRS_SEQ_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_SEQUENCE                                               */
/*                                                                           */
/*  Combines INT_SEQ, INT_DEV (without allow_zero), and COST into a single   */
/*  expression, for the same optimizations as KHE_DRS_EXPR_COUNTER.          */
/*                                                                           */
/*  Field e->closed_seqs contains the closed sequences of e.  When e is      */
/*  closed there is just one of these, containing all the children of e.     */
/*  When e is open there is one for each gap between adjacent open children  */
/*  of e, plus one for before the first open child and one for after the     */
/*  last open child.  This is one more closed sequence than the number of    */
/*  open children.  History children are not included in closed sequences.   */
/*                                                                           */
/*  Obsolete:                                                                */
/*  The first_open_child_index contains the index in open_children of the    */
/*  first open child.  This will usually be 0, but during closing the        */
/*  children are closed in increasing index order and this field increases.  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_sequence_rec {
  INHERIT_KHE_DRS_EXPR_COST
  int				min_limit;		/* minimum limit     */
  int				max_limit;		/* maximum limit     */
  int				history_before;		/* history before    */
  int				history_after;		/* history after     */
  int				history;		/* history value     */
  KHE_DRS_SEQ_TYPE		seq_type;		/* echoes frame      */
  int				seq_index;		/* SEQ_SINGLE_POSITV */
  KHE_DRS_ADJUST_TYPE		adjust_type;		/* type of adjustment*/
  /* KHE_DRS_DIM5_TABLE		dom_table; */		/* dominance tests   */
  /* int		  first_open_child_index; */	/* temporary         */
  ARRAY_KHE_DRS_CLOSED_SEQ	closed_seqs;		/* closed sequences  */
} *KHE_DRS_EXPR_SEQUENCE;

typedef HA_ARRAY(KHE_DRS_EXPR_SEQUENCE) ARRAY_KHE_DRS_EXPR_SEQUENCE;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - solutions"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_ASST_OP - an assignment operation                           */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_ASST_OP_UNASSIGN,
  KHE_DRS_ASST_OP_ASSIGN,
  KHE_DRS_ASST_OP_REPLACE
} KHE_DRS_ASST_OP;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN - a partial solution, part of the search tree          */
/*                                                                           */
/*  All valid values are non-NULL.  In the root solution the prev_soln and   */
/*  day fields are NULL, and prev_tasks is empty.                            */
/*                                                                           */
/*  Field priqueue_index of solution S is a back pointer into the priority   */
/*  queue, when that is in use.  Actually its meaning is more complicated:   */
/*                                                                           */
/*    * If S->priqueue_index == -1, S is not in the priority queue.  It has  */
/*      been expanded, which means that it cannot be freed until the end of  */
/*      the solve, since that would invalidate its descendants.              */
/*                                                                           */
/*    * If S->priqueue_index == 0, S is not in the priority queue, either    */
/*      because there is no priority queue, or because it has not been       */
/*      inserted yet, or because it was previously inserted but then it      */
/*      was deleted, owing to being found to be dominated by some other      */
/*      solution.  It has not been expanded, so it can be freed safely.      */
/*                                                                           */
/*  These first two cases apply whether there is a priority queue or not.    */
/*  When there is no priority queue the condition they record does not       */
/*  matter, but it is simpler to record it anyway.                           */
/*                                                                           */
/*    * If priqueue_index >= 1, this solution is currently in the priority   */
/*      queue, at position priqueue_index.  It has not been expanded.        */
/*                                                                           */
/*  Ordinarily, when a solution S lying in a solution set is found to be     */
/*  dominated by some other solution, it is removed from the solution set    */
/*  and from the priority queue and freed.  However, this is not done when   */
/*  S is expanded (when S->priqueue_index == -1).  Instead, S is left as     */
/*  is, just as though it was not dominated at all.  This has two benefits.  */
/*  First, every solution can be sure that its ancestors have not been       */
/*  freed; if any had been freed, that would make the prev_soln fields of    */
/*  solutions invalid.  Second, every unfreed solution lies in some          */
/*  solution set, so it will eventually be freed (at the end of the solve).  */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_soln_rec {
  struct khe_drs_signature_set_rec sig_set;	/* expanded to save space    */
  KHE_DRS_SOLN			prev_soln;	/* edge from prev_soln       */
  ARRAY_KHE_DRS_TASK_ON_DAY	prev_tasks;	/* indexed by open resources */
  int				priqueue_index;	/* priqueue index, if any    */
  int				durns_squared;	/* square of task durns      */
  /* int			total_durn; */	/* total durn of all tasks   */
#if TESTING
  int				sorted_rank;	/* rank when solns sorted    */
#endif
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_MTASK_SOLN - assignment of resource to mtask                */
/*                                                                           */
/*  An object of this type is always non-NULL.  It represents one            */
/*  assignment of a given resource on day to a given mtask or task           */
/*  on that day.                                                             */
/*                                                                           */
/*  If mtask != NULL, then the assignment is free to be to any task of that  */
/*  mtask.  Someone will have to decide which of the mtask's tasks to use    */
/*  before the assignment can actually be made.  In this case,               */
/*  fixed_task_on_day is not used.  Its value will be NULL.                  */
/*                                                                           */
/*  Otherwise, mtask == NULL.  The decision about which task to use has      */
/*  already been made, and fixed_task_on_day holds that decision.  It could  */
/*  be NULL, in which case the decision is to assign a free day.             */
/*                                                                           */
/*  The skip_assts array holds assignments that can be skipped (owing to     */
/*  two-extra selection) if this assignment is used.  A non-zero skip_count  */
/*  value indicates that this assignment may be skipped.                     */
/*                                                                           */
/*  The random field is a vaguely random number which is used to mix         */
/*  things up a bit when costs are equal.                                    */
/*                                                                           */
/*****************************************************************************/

struct Khe_drs_mtask_soln_rec {
  KHE_DRS_SIGNATURE			sig;
  KHE_DRS_RESOURCE_ON_DAY		resource_on_day;
  KHE_DRS_MTASK				mtask;
  KHE_DRS_TASK_ON_DAY			fixed_task_on_day;
  ARRAY_KHE_DRS_MTASK_SOLN		skip_assts;
  int					skip_count;
  int					random;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_TASK_SOLN - solution holding assignment of resource to task */
/*                                                                           */
/*  NB this is not a pointer type.                                           */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_task_soln_rec {
  KHE_DRS_MTASK_SOLN		mtask_soln;
  KHE_DRS_TASK_ON_DAY		fixed_dtd;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_TASK_SOLN_SET - a set of task solution objects              */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_task_soln_set {
  ARRAY_KHE_DRS_TASK_SOLN	task_solns;
};

#define KheDrsTaskSolnSetForEach(dtss, dts, i)			\
  HaArrayForEach(dtss->task_solns, dts, i)


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SHIFT_SOLN - an assignment of resources to one shift        */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_shift_soln_rec {
  KHE_DRS_SIGNATURE			sig;
  KHE_DRS_TASK_SOLN_SET			task_solns;
  ARRAY_KHE_DRS_SHIFT_SOLN		skip_assts;
  int					skip_count;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SHIFT_SOLN_TRIE - shift assignment lists indexed by r-set   */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_shift_soln_trie_rec {
  ARRAY_KHE_DRS_SHIFT_SOLN		shift_solns;	/* assts at this pt  */
  ARRAY_KHE_DRS_SHIFT_SOLN_TRIE		children;	/* children */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SHIFT_PAIR_SOLN - an assignment of resources to two shifts  */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_shift_pair_soln_rec {
  struct khe_drs_signature_set_rec	sig_set;
  KHE_DRS_SHIFT_SOLN			dss1;
  KHE_DRS_SHIFT_SOLN			dss2;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_PACKED_SOLN_DAY - one day of a packed solution              */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_packed_soln_day_rec {
  KHE_DRS_DAY			day;			/* day of this obj   */
  ARRAY_KHE_DRS_TASK_ON_DAY	prev_tasks;		/* tasks             */
} *KHE_DRS_PACKED_SOLN_DAY;

typedef HA_ARRAY(KHE_DRS_PACKED_SOLN_DAY) ARRAY_KHE_DRS_PACKED_SOLN_DAY;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_PACKED_SOLN - a packed solution object                      */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_packed_soln_rec {
  KHE_COST			cost;			/* at end of search  */
  ARRAY_KHE_DRS_PACKED_SOLN_DAY	days;			/* days of rerun     */
} *KHE_DRS_PACKED_SOLN;

typedef HA_ARRAY(KHE_DRS_PACKED_SOLN) ARRAY_KHE_DRS_PACKED_SOLN;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - expansion"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPANDER - global values that vary during expansion         */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expander_rec *KHE_DRS_EXPANDER;
typedef HA_ARRAY(KHE_DRS_EXPANDER) ARRAY_KHE_DRS_EXPANDER;

struct khe_drs_expander_rec {
  KHE_DYNAMIC_RESOURCE_SOLVER	solver;
  ARRAY_KHE_DRS_TASK_SOLN	task_solns;
  ARRAY_KHE_DRS_TASK_SOLN	tmp_task_solns;
  bool				whole_tasks;
  bool				open;
  int				durns_squared;
  KHE_COST			cost;
  KHE_COST			cost_limit;
  int				free_resource_count;
  int				must_assign_count;
  HA_ARRAY_INT			marks;
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - sets of solutions"                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN_LIST - a list of solns; has several uses               */
/*                                                                           */
/*****************************************************************************/

#define KHE_DRS_LIST_MAGIC 777777

struct khe_drs_soln_list_rec {
  KHE_DYNAMIC_RESOURCE_SOLVER	solver;
  ARRAY_KHE_DRS_SOLN		solns;
  int				list_magic;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN_TRIE - a trie of solns                                 */
/*                                                                           */
/*****************************************************************************/

/* *** withdrawn
typedef struct khe_drs_soln_trie_rec *KHE_DRS_SOLN_TRIE;
typedef HA_ARRAY(KHE_DRS_SOLN_TRIE) ARRAY_KHE_DRS_SOLN_TRIE;

struct khe_drs_soln_trie_rec {
  KHE_DRS_SOLN			soln;
  ARRAY_KHE_DRS_SOLN_TRIE	children;
  int				trie_base;	** first index of children **
};
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOFT_INDEXED_SOLN_SET - an array of soln lists, indexed     */
/*    by soft cost                                                           */
/*                                                                           */
/*****************************************************************************/

#define KHE_DRS_SOFT_MAGIC 888888

typedef struct khe_drs_soft_indexed_soln_set_rec {
  int				base;
  int				increment;
  ARRAY_KHE_DRS_SOLN_LIST	soln_lists;
  int				count;
  int				soft_magic;
} *KHE_DRS_SOFT_INDEXED_SOLN_SET;

typedef HA_ARRAY(KHE_DRS_SOFT_INDEXED_SOLN_SET)
  ARRAY_KHE_DRS_SOFT_INDEXED_SOLN_SET;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_HARD_INDEXED_SOLN_SET - an array of soft indexed solution   */
/*    sets, indexed by hard cost                                             */
/*                                                                           */
/*****************************************************************************/

#define KHE_DRS_HARD_MAGIC 999999

typedef struct khe_drs_hard_indexed_soln_set_rec {
  int					base;
  int					increment;
  ARRAY_KHE_DRS_SOFT_INDEXED_SOLN_SET	soft_sets;
  int					count;
  int					hard_magic;
} *KHE_DRS_HARD_INDEXED_SOLN_SET;

typedef HA_ARRAY(KHE_DRS_HARD_INDEXED_SOLN_SET)
  ARRAY_KHE_DRS_HARD_INDEXED_SOLN_SET;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN_SET_PART - one part of a solution set                  */
/*                                                                           */
/*****************************************************************************/

#define INHERIT_KHE_DRS_SOLN_SET_PART					\
  KHE_DRS_DOM_KIND		dom_kind;		/* kind of dominance */

typedef struct khe_drs_soln_set_part_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
} *KHE_DRS_SOLN_SET_PART;

typedef struct khe_drs_soln_set_part_dom_none_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_SOLN_LIST		soln_list;
} *KHE_DRS_SOLN_SET_PART_DOM_NONE;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_NONE)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_NONE;

typedef struct khe_drs_soln_set_part_dom_weak_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  TABLE_KHE_DRS_SOLN		soln_table;
} *KHE_DRS_SOLN_SET_PART_DOM_WEAK;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_WEAK)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_WEAK;

typedef struct khe_drs_soln_set_part_dom_medium_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  TABLE_KHE_DRS_SOLN_LIST	soln_list_table;
} *KHE_DRS_SOLN_SET_PART_DOM_MEDIUM;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_MEDIUM)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_MEDIUM;

typedef struct khe_drs_soln_set_part_dom_separate_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_SOLN_LIST		soln_list;
} *KHE_DRS_SOLN_SET_PART_DOM_SEPARATE;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_SEPARATE)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_SEPARATE;

/* *** withdrawn
typedef struct khe_drs_soln_set_part_dom_trie_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_SOLN_TRIE		soln_trie;
} *KHE_DRS_SOLN_SET_PART_DOM_TRIE;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_TRIE)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_TRIE;
*** */

typedef struct khe_drs_soln_set_part_dom_indexed_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_HARD_INDEXED_SOLN_SET	indexed_solns;
} *KHE_DRS_SOLN_SET_PART_DOM_INDEXED;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_INDEXED)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_INDEXED;

typedef struct khe_drs_soln_set_part_dom_tabulated_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_SOLN_LIST		soln_list;
} *KHE_DRS_SOLN_SET_PART_DOM_TABULATED;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_TABULATED)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_TABULATED;

typedef struct khe_drs_soln_set_part_dom_indexed_tabulated_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_HARD_INDEXED_SOLN_SET	indexed_solns;
} *KHE_DRS_SOLN_SET_PART_DOM_INDEXED_TABULATED;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_INDEXED_TABULATED)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_INDEXED_TABULATED;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN_SET - a set of solns, with a choice of implementations */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_soln_set_rec {
  KHE_DRS_SOLN_LIST		non_dominance_solns;	/* avoid domainance  */
  KHE_DRS_SOLN_SET_PART		cache;			/* cache (optional)  */
  KHE_DRS_SOLN_SET_PART		main;			/* main part         */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - solving"                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DYNAMIC_RESOURCE_SOLVER - a dynamic resource solver             */
/*                                                                           */
/*****************************************************************************/

struct khe_dynamic_resource_solver_rec {

  /* fields which are constant throughout the lifetime of the solver */
  HA_ARENA			arena;
  KHE_SOLN			soln;
  KHE_RESOURCE_TYPE		resource_type;
  KHE_OPTIONS			options;
  KHE_FRAME			days_frame;
  /* KHE_EVENT_TIMETABLE_MONITOR	etm; */
  KHE_MTASK_FINDER		mtask_finder;
  HA_ARRAY_FLOAT		max_workload_per_time;
  ARRAY_KHE_DRS_RESOURCE	all_resources;
  ARRAY_KHE_DRS_DAY		all_days;
  ARRAY_KHE_DRS_TASK		all_root_tasks;
  ARRAY_KHE_DRS_CONSTRAINT	all_constraints;
  ARRAY_KHE_DRS_MONITOR		all_monitors;
  /* int			limit_workload_monitor_count; */
  int				postorder_count;
  int				soft_increment;	  /* gcd of soft weights */
  int				hard_increment;	  /* gcd of hard weights */

  /* free list fields */
  ARRAY_KHE_DRS_RESOURCE_SET		   resource_set_free_list;
  ARRAY_KHE_DRS_DIM1_TABLE		   table1_free_list;
  ARRAY_KHE_DRS_SIGNATURE		   signature_free_list;
  ARRAY_KHE_DRS_SIGNER			   signer_free_list;
  ARRAY_KHE_DRS_SIGNER_SET		   signer_set_free_list;
  ARRAY_KHE_DRS_DOM_TEST		   dom_test_free_list;
  ARRAY_KHE_DRS_CLOSED_SEQ		   closed_seq_free_list;
  ARRAY_KHE_DRS_SOLN			   soln_free_list;
  ARRAY_KHE_DRS_MTASK_SOLN                 mtask_soln_free_list;
  ARRAY_KHE_DRS_TASK_SOLN_SET	  	   task_soln_set_free_list;
  ARRAY_KHE_DRS_SHIFT_SOLN		   shift_soln_free_list;
  ARRAY_KHE_DRS_SHIFT_SOLN_TRIE		   shift_soln_trie_free_list;
  ARRAY_KHE_DRS_SHIFT_PAIR_SOLN		   shift_pair_soln_free_list;
  ARRAY_KHE_DRS_PACKED_SOLN_DAY		   packed_soln_day_free_list;
  ARRAY_KHE_DRS_PACKED_SOLN		   packed_soln_free_list;
  ARRAY_KHE_DRS_EXPANDER		   expander_free_list;
  ARRAY_KHE_DRS_SOLN_LIST		   soln_list_free_list;
  /* ARRAY_KHE_DRS_SOLN_TRIE		   soln_trie_free_list; */
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_NONE	   soln_set_part_dom_none_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_WEAK	   soln_set_part_dom_weak_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_MEDIUM   soln_set_part_dom_medium_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_SEPARATE soln_set_part_dom_separate_free_list;
  /* ARRAY_KHE_DRS_SOLN_SET_PART_DOM_TRIE  soln_set_part_dom_trie_free_list; */
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_INDEXED  soln_set_part_dom_indexed_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_TABULATED
    				   soln_set_part_dom_tabulated_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_INDEXED_TABULATED
				   soln_set_part_dom_indexed_tabulated_free_list;
  ARRAY_KHE_DRS_SOFT_INDEXED_SOLN_SET	   soft_indexed_soln_set_free_list;
  ARRAY_KHE_DRS_HARD_INDEXED_SOLN_SET	   hard_indexed_soln_set_free_list;
  ARRAY_KHE_DRS_SOLN_SET		   soln_set_free_list;

  /* priority queue (always initialized, but actual use is optional) */
  KHE_PRIQUEUE			priqueue;

  /* fields that vary with the solve */
  ARRAY_KHE_INTERVAL		selected_day_ranges;	/* chosen for solve  */
  KHE_RESOURCE_SET		selected_resource_set;	/* chosen for solve  */
  bool				solve_priqueue;		/* use priqueue      */
  bool				solve_extra_selection;	/* use int. d. */
  bool				solve_expand_by_shifts;	/* expand by shifts  */
  bool				solve_shift_pairs;	/* test shift pairs  */
  bool				solve_correlated_exprs;
  int				solve_daily_expand_limit;
  int				solve_daily_prune_trigger;
  int				solve_resource_expand_limit;
  int				solve_dom_approx;
  /* KHE_DRS_DOM_KIND		solve_main_dom_kind; */
  /* bool			solve_cache; */
  /* KHE_DRS_DOM_KIND		solve_cache_dom_kind; */
  KHE_DRS_DOM_TEST_TYPE		solve_dom_test_type;	/* from main_dom_kind*/
  KHE_COST			solve_init_cost;	/* before solving    */
  KHE_COST			solve_start_cost;	/* before first day  */
  ARRAY_KHE_DRS_DAY		open_days;
  ARRAY_KHE_DRS_SHIFT		open_shifts;
  KHE_DRS_RESOURCE_SET		open_resources;
  ARRAY_KHE_DRS_EXPR		open_exprs;
  KHE_DRS_PACKED_SOLN		rerun_soln;		/* non-NULL if rerun */
  bool				time_limit_reached;

  /* fields for recording running time and other statistics */
#if TESTING
  KHE_TIMER			timer;
  HA_ARRAY_INT			solns_made_per_day;
  HA_ARRAY_INT			table_size_per_day;
  HA_ARRAY_FLOAT		running_time_per_day;
  HA_ARRAY_INT			ancestor_freq;
  HA_ARRAY_INT			dominates_freq;
  int				max_open_day_index;
#endif
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule " miscellaneous functions"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int min(int a, int b)                                                    */
/*                                                                           */
/*  Return the smaller of a and b, or either if they are equal.              */
/*                                                                           */
/*****************************************************************************/

static int min(int a, int b)
{
  return a < b ? a : b;
}


/*****************************************************************************/
/*                                                                           */
/*  int max(int a, int b)                                                    */
/*                                                                           */
/*  Return the larger of a and b, or either if they are equal.               */
/*                                                                           */
/*****************************************************************************/

static int max(int a, int b)
{
  return a > b ? a : b;
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "times"                                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DAY _RANGE" - replaced by KHE_INTERVAL                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DAY_RANGE KheDrsDayRangeMake(int first, int last)                */
/*                                                                           */
/*  Return a new interval with these endpoints.                              */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_DAY_RANGE KheDrsDayRangeMake(int first, int last)
{
  KHE_DRS_DAY_RANGE res;
  res.first = first;
  res.last = last;
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayRangeAdd(KHE_DRS_DAY_RANGE *ddr, int index)                */
/*                                                                           */
/*  Update *ddr to include index.                                            */
/*                                                                           */
/*  Implementation note.  Both cases are needed here, as proved by a trace   */
/*  of Add([1, 0], 4).  The second case wrongly produces [1, 4].             */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by ddr = KheIntervalUnion(*ddr, KheIntervalMake(index, index))
static void KheDrsDayRangeAdd(KHE_DRS_DAY_RANGE *ddr, int index)
{
  if( ddr->first > ddr->last )
  {
    ** ddr is empty, so make it cover just index **
    ddr->first = ddr->last = index;
  }
  else
  {
    ** ddr is non-empty, so expand it to include index **
    if( index < ddr->first )
      ddr->first = index;
    if( index > ddr->last )
      ddr->last = index;
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayRangeMerge(ARRAY_KHE_INTERVAL *day_ranges,                 */
/*    KHE_INTERVAL ddr)                                                      */
/*                                                                           */
/*  Merge ddr into *day_ranges, a sorted array of day ranges.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDayRangeMerge(ARRAY_KHE_INTERVAL *day_ranges,
  KHE_INTERVAL ddr)
{
  KHE_INTERVAL ddr2;  int i;

  /* if ddr is empty, it changes nothing (this can't happen, actually) */
  if( KheIntervalEmpty(ddr) )
    return;

  HaArrayForEach(*day_ranges, ddr2, i)
  {
    /***********************************************************************/
    /*                                                                     */
    /*  Invariant: day_ranges and ddr may have changed, but it remains     */
    /*  true that merging ddr with day_ranges gives the desired result.    */
    /*  Also, day_ranges[0..i-1] precede and do not interact with ddr.     */
    /*                                                                     */
    /***********************************************************************/

    if( KheIntervalLast(ddr2) + 1 < KheIntervalFirst(ddr) )
    {
      /* ddr2 precedes ddr and does not interact with it; continue */
      continue;
    }

    if( KheIntervalLast(ddr) + 1 < KheIntervalFirst(ddr2) )
    {
      /* ddr precedes ddr2 and does not interact with it, or with day_ranges */
      HaArrayAdd(*day_ranges, i, ddr);
      return;
    }

    /* ddr2 interacts with ddr, so expand ddr to include ddr2, then */
    /* delete ddr2 and continue */
    ddr = KheIntervalUnion(ddr, ddr2);
    /* ***
    if( ddr2.first < ddr.first )
      ddr.first = ddr2.first;
    if( ddr2.last > ddr.last )
      ddr.last = ddr2.last;
    *** */
    HaArrayDeleteAndShift(*day_ranges, i);
    i--;
  }

  /* ddr does not interact with any day_ranges, and it follows them all */
  HaArrayAddLast(*day_ranges, ddr);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDayRangeSubset(KHE_DRS_DAY_RANGE ddr1, KHE_DRS_DAY_RANGE ddr2)*/
/*                                                                           */
/*  Return true if ddr1 is a subset of ddr2.                                 */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheIntervalSubset
static bool KheDrsDayRangeSubset(KHE_DRS_DAY_RANGE ddr1, KHE_DRS_DAY_RANGE ddr2)
{
  return ddr1.first >= ddr2.first && ddr1.last <= ddr2.last;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayRangeDebug(KHE_INTERVAL ddr,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of ddr onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDayRangeDebug(KHE_INTERVAL ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "%s", KheIntervalShow(ddr, drs->days_frame));
  /* ***
  KHE_TIME_GROUP tg1, tg2;
  tg1 = KheFrameTimeGroup(drs->days_frame, ddr.first);
  tg2 = KheFrameTimeGroup(drs->days_frame, ddr.last);
  if( tg1 == tg2 )
    fprintf(fp, "%s", KheTimeGroupId(tg1));
  else
    fprintf(fp, "%s-%s", KheTimeGroupId(tg1), KheTimeGroupId(tg2));
  *** */
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DAY"                                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DAY KheDrsDayMake(int frame_index,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new day object with the given attributes.  Its mtasks,            */
/*  expressions, and dominance tests will be added later.                    */
/*                                                                           */
/*  Here frame_index is the index of the new day in the common frame.        */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SIGNER_SET KheDrsSignerSetMake(/* KHE_DRS_DAY day, */
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static KHE_DRS_DAY KheDrsDayMake(int frame_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY res;
  HaMake(res, drs->arena);
  res->frame_index = frame_index;
  res->open_day_index = -1;  /* signals not currently open for solving */
  res->time_group = KheFrameTimeGroup(drs->days_frame, frame_index);
  HaArrayInit(res->shifts, drs->arena);
  res->signer_set = KheDrsSignerSetMake(/* res, */ drs);
  res->soln_set = NULL;   /* initialized at the start of each solve */
  res->soln_list = NULL;  /* initialized at the start of each solve */
  res->soln_made_count = 0;
  res->solve_expand_count = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsDayId(KHE_DRS_DAY day)                                       */
/*                                                                           */
/*  Return the Id of day.  This is just the Id of its time group.  A NULL    */
/*  day is allowed and "-" is returned in that case.                         */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsDayId(KHE_DRS_DAY day)
{
  return day == NULL ? "-" : KheTimeGroupId(day->time_group);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayAddMTask(KHE_DRS_DAY day, KHE_DRS_MTASK dmt,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add dmt to the day's mtasks and shifts.                                  */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SHIFT KheDrsShiftMake(KHE_DRS_DAY day, KHE_DRS_MTASK dmt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static bool KheDrsShiftAcceptsMTask(KHE_DRS_SHIFT ds, KHE_DRS_MTASK dmt);

static void KheDrsDayAddMTask(KHE_DRS_DAY day, KHE_DRS_MTASK dmt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT ds;  int i;

  /* add dmt to day->shifts */
  HaArrayForEach(day->shifts, ds, i)
    if( KheDrsShiftAcceptsMTask(ds, dmt) )
    {
      dmt->encl_shift = ds;
      return;
    }

  /* need a new shift */
  ds = KheDrsShiftMake(day, dmt, drs);
  dmt->encl_shift = ds;
  HaArrayAddLast(day->shifts, ds);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayOpen(KHE_DRS_DAY day, KHE_INTERVAL ddr,                    */
/*    int open_day_index, KHE_DRS_DOM_KIND dom_kind,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Open day for solving.  It lies within day range ddr, its index in the    */
/*  the sequence of all open days is open_day_index, and dom_kind is the     */
/*  kind of dominance testing to use on this day.                            */
/*                                                                           */
/*  The signer is clear initially and cleared each time the day is closed,   */
/*  so there is no need to clear it here, and we don't.                      */
/*                                                                           */
/*  NB We open the day's mtasks separately, because they may contain         */
/*  tasks for later days and opening assumes that those days are open.       */
/*                                                                           */
/*  NBB The assignment "day->solve_prune_cost = drs->solve_init_cost" is     */
/*  safe here because KheDrsDayOpen is called after drs->solve_init_cost     */
/*  is initialized.  (This comment is obsolete.)                             */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsMTaskOpen(KHE_DRS_MTASK dmt, KHE_INTERVAL ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_SOLN_SET KheDrsSolnSetMake(KHE_DRS_DOM_KIND main_dom_kind,
  bool cache, KHE_DRS_DOM_KIND cache_dom_kind, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignerSetDayOpen(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignerSetDebug(KHE_DRS_SIGNER_SET signer_set,
  int verbosity, int indent, FILE *fp);

static void KheDrsDayOpen(KHE_DRS_DAY day, KHE_INTERVAL ddr,
  int open_day_index, KHE_DRS_DOM_KIND main_dom_kind, bool cache,
  KHE_DRS_DOM_KIND cache_dom_kind, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( DEBUG18 || DEBUG69(open_day_index) )
    fprintf(stderr, "[ KheDrsDayOpen(%s, %d-%d, %d)\n", KheDrsDayId(day),
      ddr.first, ddr.last, open_day_index);
  day->open_day_index = open_day_index;
  KheDrsSignerSetDayOpen(day->signer_set, day, drs);
  day->soln_set = KheDrsSolnSetMake(main_dom_kind, cache, cache_dom_kind, drs);
  day->soln_made_count = 0;
  day->solve_expand_count = 0;
  if( DEBUG18 || DEBUG69(open_day_index) )
    fprintf(stderr, "] KheDrsDayOpen\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayOpenShifts(KHE_DRS_DAY day, KHE_INTERVAL ddr,              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Open the shifts of day, including opening their mtasks.                  */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftOpen(KHE_DRS_SHIFT ds, KHE_INTERVAL ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsDayOpenShifts(KHE_DRS_DAY day, KHE_INTERVAL ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT shift;  int i;
  if( DEBUG18 || DEBUG70(ddr, day->frame_index) )
    fprintf(stderr, "[ KheDrsDayOpenShifts(%s, %d-%d)\n",
      KheDrsDayId(day), ddr.first, ddr.last);
  HaArrayForEach(day->shifts, shift, i)
    KheDrsShiftOpen(shift, ddr, drs);
  if( DEBUG18 || DEBUG70(ddr, day->frame_index) )
    fprintf(stderr, "] KheDrsDayOpenShifts\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayAddOpenExpr(KHE_DRS_DAY day, KHE_DRS_EXPR e)               */
/*                                                                           */
/*  Add open expression e to day.                                            */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsSignerAddOpenExpr(KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e);

static void KheDrsDayAddOpenExpr(KHE_DRS_DAY day, KHE_DRS_EXPR e)
{
  KHE_DRS_SIGNER dsg;
  HnAssert(e->resource == NULL, "KheDrsDayAddOpenExpr internal error");
  HnAssert(HaArrayCount(day->signer_set->signers) > 0,
    "KheDrsDayAddOpenExpr internal error");
  dsg = HaArrayLast(day->signer_set->signers);
  KheDrsSignerAddOpenExpr(dsg, e);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsDayAddDomTest(KHE_DRS_DAY day, KHE_DRS_DOM_TEST dom_test)      */
/*                                                                           */
/*  Add dom_test to day and return its index in day->dom_tests.              */
/*                                                                           */
/*****************************************************************************/

/* ***
static int KheDrsSignerAddDomTest(KHE_DRS_SIGNER dsg,
  KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static int KheDrsDayAddDomTest(KHE_DRS_DAY day, KHE_DRS_DOM_TEST dom_test,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNER dsg;
  HnAssert(HaArrayCount(day->signer_set->signers) > 0,
    "KheDrsDayAddOpenExpr internal error");
  dsg = HaArrayLast(day->signer_set->signers);
  return KheDrsSignerAddDomTest(dsg, dom_test, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNER KheDrsDaySigner(KHE_DRS_DAY day)                          */
/*                                                                           */
/*  Return the signer for day.                                               */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNER KheDrsDaySigner(KHE_DRS_DAY day)
{
  return HaArrayLast(day->signer_set->signers);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayClose(KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Close day.                                                               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftClose(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSolnListFreeSolns(KHE_DRS_SOLN_LIST soln_list,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSolnListFree(KHE_DRS_SOLN_LIST soln_list,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_SOLN_LIST KheDrsDayGatherSolns(KHE_DRS_DAY next_day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignerSetDayClose(KHE_DRS_SIGNER_SET signer_set,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsDayClose(KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i;  KHE_DRS_SHIFT shift;
  day->open_day_index = -1;  /* signals no longer open */
  HaArrayForEach(day->shifts, shift, i)
    KheDrsShiftClose(shift, drs);
  KheDrsSignerSetDayClose(day->signer_set, drs);

  /* delete solution sets, solution lists, and solutions as required */
  if( day->soln_list != NULL )
  {
    KheDrsSolnListFreeSolns(day->soln_list, drs);
    KheDrsSolnListFree(day->soln_list, drs);
    day->soln_list = NULL;
  }
  if( day->soln_set != NULL )
  {
    KheDrsDayGatherSolns(day, drs);
    KheDrsSolnListFreeSolns(day->soln_list, drs);
    KheDrsSolnListFree(day->soln_list, drs);
    day->soln_list = NULL;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayDebug(KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs,    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of day onto fp.                                              */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftDebug(KHE_DRS_SHIFT ds,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp);
static void KheDrsMTaskDebug(KHE_DRS_MTASK dmt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp);
static void KheDrsSolnSetDebug(KHE_DRS_SOLN_SET dns,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp);

static void KheDrsDayDebug(KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  int i;  KHE_DRS_SHIFT ds;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ Day %s (frame %d, ", indent, "",
      KheDrsDayId(day), day->frame_index);
    if( day->open_day_index >= 0 )
      fprintf(fp, "open %d)\n", day->open_day_index);
    else
      fprintf(fp, "closed)\n");
    if( verbosity >= 2 )
    {
      HaArrayForEach(day->shifts, ds, i)
	KheDrsShiftDebug(ds, drs, verbosity, indent + 2, fp);
    }
    if( verbosity >= 3 && day->soln_set != NULL )
      KheDrsSolnSetDebug(day->soln_set, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "%s", KheDrsDayId(day));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DAY - expansion"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayExpandBegin(KHE_DRS_DAY next_day, KHE_DRS_SOLN prev_soln,  */
/*    KHE_DRS_DAY prev_day, KHE_DRS_EXPANDER de)                             */
/*                                                                           */
/*  Begin the expansion of prev_soln into next_day.                          */
/*                                                                           */
/*  Implementation note - setting ds->expand_max_included_free_resource_count*/
/*  -----------------------------------------------------------------        */
/*  This field contains the maximum number of free resources that can be     */
/*  assigned to ds without making it impossible to fulfil the minimum        */
/*  requirements of other shifts.  First we calculate excess, the number of  */
/*  free resources that are not already committed to minimum requirements.   */
/*  Then ds->expand_max_included_free_resource_count is this number added to */
/*  the minimum for ds.  This can never exceed the number of free resources. */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSolnSetBeginCacheSegment(KHE_DRS_SOLN_SET soln_set,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsShiftExpandBegin(KHE_DRS_SHIFT ds, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY prev_day, KHE_DRS_EXPANDER de);
static int KheDrsExpanderExcessFreeResourceCount(KHE_DRS_EXPANDER de);

static void KheDrsDayExpandBegin(KHE_DRS_DAY next_day, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY prev_day, KHE_DRS_EXPANDER de)
{
  int i, excess;  KHE_DRS_SHIFT ds;

  /* begin caching expansions */
  KheDrsSolnSetBeginCacheSegment(next_day->soln_set, de->solver);

  /* begin expansion in each shift */
  HaArrayForEach(next_day->shifts, ds, i)
    KheDrsShiftExpandBegin(ds, prev_soln, prev_day, de);

  /* set expand_max_included_free_resource_count in each shift */
  excess = KheDrsExpanderExcessFreeResourceCount(de);
  HaArrayForEach(next_day->shifts, ds, i)
    ds->expand_max_included_free_resource_count =
      ds->expand_must_assign_count + excess;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayExpandEnd(KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)       */
/*                                                                           */
/*  End the previously begun expansion of prev_soln into next_day.           */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftExpandEnd(KHE_DRS_SHIFT ds, KHE_DRS_EXPANDER de);
static void KheDrsSolnSetEndCacheSegment(KHE_DRS_SOLN_SET soln_set,
  KHE_DRS_DAY soln_day, KHE_DRS_EXPANDER de, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsDayExpandEnd(KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)
{
  int i;  KHE_DRS_SHIFT ds;

  /* end expansion in each shift */
  HaArrayForEach(next_day->shifts, ds, i)
    KheDrsShiftExpandEnd(ds, de);

  /* end caching expansions */
  KheDrsSolnSetEndCacheSegment(next_day->soln_set, next_day, de, de->solver);
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "resources"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE_EXPAND_ROLE"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsResourceExpandRoleShow(KHE_DRS_RESOURCE_EXPAND_ROLE role)    */
/*                                                                           */
/*  Return a readable string representation of role.                         */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsResourceExpandRoleShow(KHE_DRS_RESOURCE_EXPAND_ROLE role)
{
  switch( role )
  {
    case KHE_DRS_RESOURCE_EXPAND_NO:	return "expand_no";
    case KHE_DRS_RESOURCE_EXPAND_FIXED:	return "expand_fixed";
    case KHE_DRS_RESOURCE_EXPAND_FREE:	return "expand_free";
    default:				return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsResourceId(KHE_DRS_RESOURCE dr)                              */
/*                                                                           */
/*  Return the Id of dr.                                                     */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsResourceId(KHE_DRS_RESOURCE dr)
{
  return KheResourceId(dr->resource);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE KheDrsResourceMake(KHE_RESOURCE r,                      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new resource object with these attributes.                        */
/*  Return NULL if unsuccessful.                                             */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayMake(KHE_DRS_RESOURCE dr,
  KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_DIM2_TABLE KheDrsDim2TableMake(/* char *debug_str, */HA_ARENA a);

static KHE_DRS_RESOURCE KheDrsResourceMake(KHE_RESOURCE r,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE res;  int i;  KHE_DRS_RESOURCE_ON_DAY drd;  KHE_DRS_DAY day;

  /* initialize res, including one resource on day object for each day */
  if( DEBUG07 )
    fprintf(stderr, "[ KheDrsResourceMake(%s, drs)\n", KheResourceId(r));
  HaMake(res, drs->arena);
  res->resource = r;
  res->open_resource_index = -1;
  HaArrayInit(res->days, drs->arena);
  HaArrayForEach(drs->all_days, day, i)
  {
    drd = KheDrsResourceOnDayMake(res, day, drs);
    HaArrayAddLast(res->days, drd);
  }
  res->expand_role = KHE_DRS_RESOURCE_EXPAND_NO;
  HaArrayInit(res->expand_signatures, drs->arena);
  /* HaArrayInit(res->expand_assts_to_shifts, drs->arena); */
  HaArrayInit(res->expand_mtask_solns, drs->arena);
  res->expand_free_mtask_soln = NULL;
  res->expand_dom_test_cache = KheDrsDim2TableMake(/*KheDrsResourceId(res), */
    drs->arena);

  if( DEBUG07 )
    fprintf(stderr, "] KheDrsResourceMake returning\n");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE KheDrsResource(KHE_RESOURCE r,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Assuming that r has the correct resource type, return its corresponding  */
/*  drs resource.                                                            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE KheDrsResource(KHE_RESOURCE r,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(KheResourceResourceTypeIndex(r) < HaArrayCount(drs->all_resources),
    "KheDrsResource internal error");
  return HaArray(drs->all_resources, KheResourceResourceTypeIndex(r));
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDay(KHE_DRS_RESOURCE dr,         */
/*    KHE_DRS_DAY day)                                                       */
/*                                                                           */
/*  Return the resource on day object representing dr on day.                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDay(KHE_DRS_RESOURCE dr,
  KHE_DRS_DAY day)
{
  return HaArray(dr->days, day->frame_index);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayAtTime(KHE_DRS_RESOURCE dr,   */
/*    KHE_TIME time, KHE_DYNAMIC_RESOURCE_SOLVER drs)                        */
/*                                                                           */
/*  Return the resource on day object whose resource is dr and whose day     */
/*  is time's day.                                                           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayAtTime(KHE_DRS_RESOURCE dr,
  KHE_TIME time, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int day_index;
  day_index = KheFrameTimeIndex(drs->days_frame, time);
  return HaArray(dr->days, day_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOpen(KHE_DRS_RESOURCE dr, int open_resource_index,    */
/*    KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)        */
/*                                                                           */
/*  Open dr.  This involves unassigning all the tasks assigned dr which      */
/*  lie in the selected day ranges of drs, and gathering for opening the     */
/*  expressions that depend on what dr is doing on those days, including     */
/*  on days where dr remains assigned.                                       */
/*                                                                           */
/*  Also, if init_soln != NULL, included the unassignments as assignments    */
/*  in init_soln, so that they can be undone later if required.              */
/*                                                                           */
/*  The signer is clear initially and cleared each time the resource is      */
/*  closed, so there is no need to clear it here, and we don't.              */
/*                                                                           */
/*****************************************************************************/
static void KheDrsResourceOnDayOpen(KHE_DRS_RESOURCE_ON_DAY drd,
  int open_resource_index, int open_day_index, KHE_INTERVAL ddr,
  KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsResourceDebug(KHE_DRS_RESOURCE dr, int verbosity,
  int indent, FILE *fp);

static void KheDrsResourceOpen(KHE_DRS_RESOURCE dr, int open_resource_index,
  KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_INTERVAL ddr;  int i, j, open_day_index; KHE_DRS_RESOURCE_ON_DAY drd;
  if( DEBUG11 )
  {
    fprintf(stderr, "[ KheDrsResourceOpen(%s, %d, drs)\n", KheDrsResourceId(dr),
      open_resource_index);
    KheDrsResourceDebug(dr, 2, 2, stderr);
  }
  dr->open_resource_index = open_resource_index;
  open_day_index = 0;
  HaArrayForEach(drs->selected_day_ranges, ddr, i)
    for( j = ddr.first;  j <= ddr.last;  j++ )
    {
      /* unassign any task assigned on drd, if it lies entirely in ddr */
      drd = HaArray(dr->days, j);
      KheDrsResourceOnDayOpen(drd, open_resource_index, open_day_index,
	ddr, init_soln, drs);

      /* increase open_day_index */
      open_day_index++;
    }
  if( DEBUG11 )
  {
    fprintf(stderr, "  at end of KheDrsResourceOpen(%s, %d, drs):\n",
      KheDrsResourceId(dr), open_resource_index);
    KheDrsResourceDebug(dr, 2, 2, stderr);
    fprintf(stderr, "] KheDrsResourceOpen\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceClose(KHE_DRS_RESOURCE dr)                            */
/*                                                                           */
/*  Close dr.                                                                */
/*                                                                           */
/*****************************************************************************/
static void KheDrsResourceOnDayClose(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceClose(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_INTERVAL ddr;  int i, j;  KHE_DRS_RESOURCE_ON_DAY drd;
  HaArrayForEach(drs->selected_day_ranges, ddr, i)
    for( j = ddr.first;  j <= ddr.last;  j++ )
    {
      drd = HaArray(dr->days, j);
      KheDrsResourceOnDayClose(drd, drs);
    }
  dr->open_resource_index = -1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceDebug(KHE_DRS_RESOURCE dr, int verbosity,             */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dr onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/
static void KheDrsResourceOnDayDebug(KHE_DRS_RESOURCE_ON_DAY drd,
  int verbosity, int indent, FILE *fp);

static void KheDrsResourceDebug(KHE_DRS_RESOURCE dr, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_RESOURCE_ON_DAY drd;  int i;
  if( verbosity > 1 && indent >= 0 )
  {
    /* show timetable in full */
    fprintf(fp, "%*s[ %s", indent, "", KheDrsResourceId(dr));
    if( dr->open_resource_index >= 0 )
      fprintf(fp, " (open %d)", dr->open_resource_index);
    fprintf(fp, "\n");
    HaArrayForEach(dr->days, drd, i)
      KheDrsResourceOnDayDebug(drd, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
  {
    if( indent > 0 )
      fprintf(fp, "%*s", indent, "");
    fprintf(fp, "%s", KheDrsResourceId(dr));
    if( indent >= 0 )
      fprintf(fp, "\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceDebugExpandTasks(KHE_DRS_RESOURCE dr,                 */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug dr, focussing on expand_mtask_solns.                               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsMTaskSolnDebug(KHE_DRS_MTASK_SOLN asst,
  int verbosity, int indent, FILE *fp);

static void KheDrsResourceDebugExpandTasks(KHE_DRS_RESOURCE dr,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_MTASK_SOLN asst;  int i;
  if( verbosity > 1 && indent >= 0 )
  {
    fprintf(fp, "%*s[ %s", indent, "", KheDrsResourceId(dr));
    if( dr->open_resource_index >= 0 )
      fprintf(fp, " (open %d)", dr->open_resource_index);
    fprintf(fp, "\n");
    HaArrayForEach(dr->expand_mtask_solns, asst, i)
      KheDrsMTaskSolnDebug(asst, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE - expansion"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNATURE KheDrsResourceSignatureMake(                           */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY dtd,                  */
/*    KHE_DRS_SIGNATURE prev_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Make and return a resource signature for the resource constraints that   */
/*  monitor drd's resource and the solution that consists of prev_sig's      */
/*  solution plus the assignment of drd to dtd.                              */
/*                                                                           */
/*  The signature has a 0 reference count on return.  It is up to the        */
/*  caller to decide whether the signature is referred to or not.            */
/*                                                                           */
/*****************************************************************************/
/* ***
static KHE_DRS_SIGNATURE KheDrsSignatureMake(KHE_DYNAMIC_RESOURCE_SOLVER drs);
*** */
static void KheDrsResourceOnDayLeafSet(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsResourceOnDayLeafClear(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_SIGNATURE KheDrsSignerEvalSignature(KHE_DRS_SIGNER dsg,
  /* bool debug_if_rerun, */ KHE_DRS_SIGNATURE prev_sig, /* int next_di, */
  KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug);

static KHE_DRS_SIGNATURE KheDrsResourceSignatureMake(
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_SIGNATURE prev_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNATURE res;

  /* set leaf expressions of the resource on day */
  KheDrsResourceOnDayLeafSet(drd, dtd, drs);

  /* make and populate the result signature */
  res = KheDrsSignerEvalSignature(drd->signer, prev_sig, drs, false);

  /* clear leaf expressions of the resource on day */
  KheDrsResourceOnDayLeafClear(drd, dtd, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetSignature(KHE_DRS_SIGNATURE next_sig,              */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY dtd,                  */
/*    KHE_DRS_SIGNATURE prev_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Build a signature object for the effect on resource monitors when        */
/*  drd is assigned to dtd.  Only extra costs are included, not the          */
/*  cost of prev_soln.                                                       */
/*                                                                           */
/*****************************************************************************/

/* *** inlined into KheDrsResourceSignatureMake, its only call site
static void KheDrsResourceSetSignature(KHE_DRS_SIGNATURE next_sig,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_SIGNATURE prev_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ** set leaf expressions of the resource on day **
  KheDrsResourceOnDayLeafSet(drd, dtd, drs);

  ** build the signature **
  KheDrsSignerEvalSignature(drd->signer, true, prev_sig,
    drd->day->open_day_index, next_sig, drs, false);

  ** clear leaf expressions of the resource on day **
  KheDrsResourceOnDayLeafClear(drd, dtd, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_MTASK_SOLN KheDrsResourceOptionallyAddMTaskSoln(                 */
/*    KHE_DRS_SIGNATURE sig, KHE_DRS_RESOURCE_ON_DAY drd,                    */
/*    KHE_DRS_MTASK dmt, KHE_DRS_TASK_ON_DAY fixed_dtd,                      */
/*    bool force, KHE_DRS_EXPANDER de)                                       */
/*                                                                           */
/*  If it would be competitive, or force is true, build an assignment to     */
/*  mtask object and add it to drd->encl_dr->expand_mtask_solns.             */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_MTASK_SOLN KheDrsMTaskSolnMake(KHE_DRS_SIGNATURE sig,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_MTASK dmt,
  KHE_DRS_TASK_ON_DAY fixed_dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static bool KheDrsMTaskSuitsFreeResource(KHE_DRS_MTASK dmt,
  KHE_DRS_RESOURCE dr);
static bool KheDrsExpanderOpenToExtraCost(KHE_DRS_EXPANDER de,
  KHE_COST extra_cost);

static KHE_DRS_MTASK_SOLN KheDrsResourceOptionallyAddMTaskSoln(
  KHE_DRS_SIGNATURE sig, KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_MTASK dmt,
  KHE_DRS_TASK_ON_DAY fixed_dtd, bool force, KHE_DRS_EXPANDER de)
{
  KHE_DRS_MTASK_SOLN res;
  if( force || KheDrsExpanderOpenToExtraCost(de, sig->cost) )
  {
    res = KheDrsMTaskSolnMake(sig, drd, dmt, fixed_dtd, de->solver);
    HaArrayAddLast(drd->encl_dr->expand_mtask_solns, res);
    return res;
  }
  else
    return NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceAddExpandSignature(KHE_DRS_RESOURCE dr,               */
/*    KHE_DRS_SIGNATURE sig)                                                 */
/*                                                                           */
/*  Add sig to the expand signatures of dr.  Update sig's reference count.   */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSignatureRefer(KHE_DRS_SIGNATURE sig);

static void KheDrsResourceAddExpandSignature(KHE_DRS_RESOURCE dr,
  KHE_DRS_SIGNATURE sig)
{
  HaArrayAddLast(dr->expand_signatures, sig);
  KheDrsSignatureRefer(sig);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceClearExpandSignatures(KHE_DRS_RESOURCE dr,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Clear the the expand signatures of dr.  Update their reference counts.   */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSignatureUnRefer(KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceClearExpandSignatures(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNATURE sig;
  while( HaArrayCount(dr->expand_signatures) > 0 )
  {
    sig = HaArrayLastAndDelete(dr->expand_signatures);
    KheDrsSignatureUnRefer(sig, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSortAndReduceMTaskSolns(KHE_DRS_RESOURCE dr,                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Sort the signatures of dr, and keep the best dr->resource_expand_limit   */
/*  of them, or more if there are ties.                                      */
/*                                                                           */
/*****************************************************************************/
static void KheDrsMTaskSolnFree(KHE_DRS_MTASK_SOLN asst,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_SIGNATURE KheDrsMTaskSolnSignature(KHE_DRS_MTASK_SOLN dms);
static int KheDrsMTaskSolnCmp(const void *t1, const void *t2);

static void KheDrsSortAndReduceMTaskSolns(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_MTASK_SOLN dms, dms2;  KHE_DRS_SIGNATURE sig, sig2;

  /* sort the mtask solutions by increasing cost */
  HaArraySort(dr->expand_mtask_solns, &KheDrsMTaskSolnCmp);

  /* if there is a resource expand limit and it will make a difference here */
  if( drs->solve_resource_expand_limit > 0 &&
      drs->solve_resource_expand_limit < HaArrayCount(dr->expand_mtask_solns) )
  {
    /* find dms, a signature with the largest cost that we want to keep */
    dms = HaArray(dr->expand_mtask_solns, drs->solve_resource_expand_limit-1);
    sig = KheDrsMTaskSolnSignature(dms);

    /* delete and free all signatures whose cost exceeds dms's */
    dms2 = HaArrayLast(dr->expand_mtask_solns);
    sig2 = KheDrsMTaskSolnSignature(dms2);
    while( sig2->cost > sig->cost )
    {
      HaArrayDeleteLast(dr->expand_mtask_solns);
      if( dms2 == dr->expand_free_mtask_soln )
        dr->expand_free_mtask_soln = NULL;
      KheDrsMTaskSolnFree(dms2, drs);
      dms2 = HaArrayLast(dr->expand_mtask_solns);
      sig2 = KheDrsMTaskSolnSignature(dms2);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceAdjustSignatureCosts(KHE_DRS_RESOURCE dr,             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Adjust the signature costs in the resource assignments of dr, by         */
/*  finding the minimum cost (that must be incurred whatever dr is           */
/*  assigned to) and moving it out of the resource assignments and           */
/*  into drs->expand_cost.                                                   */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderAddCost(KHE_DRS_EXPANDER de, KHE_COST cost);
static void KheDrsSignatureDebug(KHE_DRS_SIGNATURE sig, int verbosity,
  int indent, FILE *fp);

static void KheDrsResourceAdjustSignatureCosts(KHE_DRS_RESOURCE dr,
  KHE_DRS_EXPANDER de)
{
  KHE_COST movable_cost;  int i;
  KHE_DRS_MTASK_SOLN dms;  KHE_DRS_SIGNATURE sig;
  if( HaArrayCount(dr->expand_mtask_solns) > 0 )
  {
    dms = HaArrayFirst(dr->expand_mtask_solns);
    movable_cost = dms->sig->cost;
    if( DEBUG74 )
      fprintf(stderr, "    [ adjusting %s costs by %.5f:\n",
	KheDrsResourceId(dr), KheCostShow(movable_cost));
    KheDrsExpanderAddCost(de, movable_cost);
    HaArrayForEach(dr->expand_signatures, sig, i)
    {
      sig->cost -= movable_cost;
      HnAssert(sig->cost >= 0,
	"KheDrsResourceAdjustSignatureCosts internal error");
      if( DEBUG74 )
	KheDrsSignatureDebug(sig, 2, 6, stderr);
    }
    if( DEBUG74 )
      fprintf(stderr, "    ]\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsResourceOnDayIsFixed(KHE_DRS_RESOURCE_ON_DAY drd,             */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs,                    */
/*    KHE_DRS_TASK_ON_DAY *dtd)                                              */
/*                                                                           */
/*  If the assignment of drd is fixed, return true and set *dtd to the task  */
/*  on day that it is fixed to.                                              */
/*                                                                           */
/*  There are four reasons why drd may be fixed:                             */
/*                                                                           */
/*    (1) this run is a rerun so we take the assignment from a packed soln;  */
/*                                                                           */
/*    (2) drd may be preassigned to a task on this day;                      */
/*                                                                           */
/*    (3) drd may have a closed assignment;                                  */
/*                                                                           */
/*    (4) drd's resource may have been assigned (in soln) to some task on    */
/*        the previous day, and that task may still be running.              */
/*                                                                           */
/*  We can find out whether drd's resource has been assigned to some task    */
/*  on the previous day by consulting (the edge leading into) soln.          */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_TASK_ON_DAY KheDrsPackedSolnTaskOnDay(
  KHE_DRS_PACKED_SOLN packed_soln, KHE_DRS_DAY open_day,
  KHE_DRS_RESOURCE open_resource);
static bool KheDrsSolnResourceIsAssigned(KHE_DRS_SOLN soln,
  KHE_DRS_RESOURCE dr, KHE_DRS_TASK_ON_DAY *dtd);
static bool KheDrsTaskRunningOnDay(KHE_DRS_TASK dt, KHE_DRS_DAY day,
  KHE_DRS_TASK_ON_DAY *dtd);

static bool KheDrsResourceOnDayIsFixed(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_TASK_ON_DAY *dtd)
{
  KHE_DRS_TASK_ON_DAY dtd1, dtd2;

  /* (1) if this is a rerun, that fixes the assignment */
  if( drs->rerun_soln != NULL )
    return *dtd = KheDrsPackedSolnTaskOnDay(drs->rerun_soln,
      drd->day, drd->encl_dr), true;

  /* (2) if drd is preassigned to a task on this day, it's fixed to that */
  if( drd->preasst_dtd != NULL )
    return *dtd = drd->preasst_dtd, true;

  /* (3) if drd has a closed assignment, it's fixed to that */
  if( drd->closed_dtd != NULL )
    return *dtd = drd->closed_dtd, true;

  /* (4) if drd's resource is assigned to a task in soln which is still */
  /* running, then drd is fixed to that */
  if( KheDrsSolnResourceIsAssigned(soln, drd->encl_dr, &dtd1) &&
	KheDrsTaskRunningOnDay(dtd1->encl_dt, drd->day, &dtd2) )
    return *dtd = dtd2, true;

  /* otherwise drd has no fixed assignment */
  return *dtd = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceDominanceTestCacheDebug(KHE_DRS_RESOURCE dr,          */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of dr's dominance test cache.                                */
/*                                                                           */
/*****************************************************************************/
static void KheDrsDim2TableDebug(KHE_DRS_DIM2_TABLE d2, int verbosity,
  int indent, FILE *fp);

static void KheDrsResourceDominanceTestCacheDebug(KHE_DRS_RESOURCE dr,
  int verbosity, int indent, FILE *fp)
{
  fprintf(fp, "%*s[ Dominance test cache for %s:\n", indent, "",
    KheDrsResourceId(dr));
  KheDrsDim2TableDebug(dr->expand_dom_test_cache, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceBuildDominanceTestCache(KHE_DRS_RESOURCE dr,          */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DYNAMIC_RESOURCE_SOLVER drs)          */
/*                                                                           */
/*  Build dr's dominance test cache, assuming that its asst to shifts        */
/*  are built.                                                               */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsSignerDoDominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_DRS_SIGNER_TEST test, int trie_start_depth, bool stop_on_neg,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp);
static KHE_DRS_COST_TUPLE KheDrsCostTupleMake(short psi, short psi0,
  /* bool psi_plus_defined, */ short psi_plus);
static void KheDrsDim2TablePut(KHE_DRS_DIM2_TABLE d2, int index2,
  int index1, KHE_DRS_COST_TUPLE ct, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignatureAddAsstToShiftIndex(KHE_DRS_SIGNATURE sig, int val);

static void KheDrsResourceBuildDominanceTestCache(KHE_DRS_RESOURCE dr,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, j;  KHE_COST cost;  KHE_DRS_COST_TUPLE ct;
  KHE_DRS_SIGNATURE sig1, sig2;
  if( USE_DOM_CACHING )
  {
    /* HaArrayForEach(dr->expand_assts_to_shifts, sasst1, i) */
    HaArrayForEach(dr->expand_signatures, sig1, i)
    {
      KheDrsSignatureAddAsstToShiftIndex(sig1, i);
      HaArrayForEach(dr->expand_signatures, sig2, j)
      {
	cost = 0;
	KheDrsSignerDoDominates(drd->signer, sig1, sig2,
	  KHE_DRS_SIGNER_ALL, 0, false, &cost, 0, 0, NULL);
	ct = KheDrsCostTupleMake(cost, 0, 0);
	KheDrsDim2TablePut(dr->expand_dom_test_cache, i, j, ct, drs);
      }
    }
    if( DEBUG67 )
      KheDrsResourceDominanceTestCacheDebug(dr, 2, 2, stderr);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceExpandBegin(KHE_DRS_RESOURCE dr,                      */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,                          */
/*    KHE_DRS_RESOURCE_SET free_resources,                                   */
/*    KHE_DRS_TASK_SOLN_SET fixed_assts, KHE_DRS_EXPANDER de)                */
/*                                                                           */
/*  Carry out the first part of beginning an expansion using open resource   */
/*  dr, which is to work out whether dr is involved in a fixed assignment,   */
/*  and then either add that assignment to fixed_assts (if it is) or add     */
/*  the resource to free_resources (if it isn't).                            */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderDeleteFreeResource(KHE_DRS_EXPANDER de);
static KHE_DRS_TASK_SOLN KheDrsTaskSolnMake(KHE_DRS_MTASK_SOLN asst,
  KHE_DRS_TASK_ON_DAY fixed_dtd);
static void KheDrsTaskSolnSetAddLast(KHE_DRS_TASK_SOLN_SET dats,
  KHE_DRS_TASK_SOLN dat);
static void KheDrsResourceSetAddLast(KHE_DRS_RESOURCE_SET rs,
  KHE_DRS_RESOURCE dr);
static char *KheDrsTaskExpandRoleShow(KHE_DRS_TASK_EXPAND_ROLE dter);

static void KheDrsResourceExpandBegin(KHE_DRS_RESOURCE dr,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,
  KHE_DRS_RESOURCE_SET free_resources,
  KHE_DRS_TASK_SOLN_SET fixed_assts, KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK_ON_DAY fixed_dtd;  KHE_DRS_RESOURCE_ON_DAY drd;
  KHE_DRS_TASK_SOLN dts;  KHE_DRS_MTASK_SOLN dms;
  KHE_DRS_TASK dt;  KHE_DRS_SIGNATURE prev_sig, sig;

  /* there should be no old assts to shifts or assignments to mtasks */
  HnAssert(HaArrayCount(dr->expand_signatures) == 0,
    "KheDrsResourceExpandBegin internal error 1");
  HnAssert(HaArrayCount(dr->expand_mtask_solns) == 0,
    "KheDrsResourceExpandBegin internal error 2");
  HnAssert(dr->expand_free_mtask_soln == NULL,
    "KheDrsResourceExpandBegin internal error 3");

  /* add signatures, and assignments to mtasks */
  drd = KheDrsResourceOnDay(dr, next_day);
  if( KheDrsResourceOnDayIsFixed(drd, prev_soln, de->solver, &fixed_dtd) )
  {
    /* mark the resource as fixed, and inform the expander */
    dr->expand_role = KHE_DRS_RESOURCE_EXPAND_FIXED;
    KheDrsExpanderDeleteFreeResource(de);

    /* fixed assignment to fixed_dtd; make and add sasst and dms */
    /* NB the optional addition is forced here, so sig is always referred to */
    prev_sig = HaArray(prev_soln->sig_set.signatures, dr->open_resource_index);
    sig = KheDrsResourceSignatureMake(drd, fixed_dtd, prev_sig, de->solver);
    dms = KheDrsResourceOptionallyAddMTaskSoln(sig, drd, NULL, fixed_dtd,
      true, de);
    KheDrsResourceAddExpandSignature(dr, sig);

    /* mark the task as fixed */
    if( fixed_dtd != NULL )
    {
      dt = fixed_dtd->encl_dt;
      HnAssert(dt->expand_role == KHE_DRS_TASK_EXPAND_NO_VALUE,
	"KheDrsResourceExpandBegin internal error 4 (role %s)",
	KheDrsTaskExpandRoleShow(dt->expand_role));
      dt->expand_role = KHE_DRS_TASK_EXPAND_FIXED;
    }

    /* make dts and add to fixed_assts */
    dts = KheDrsTaskSolnMake(dms, fixed_dtd);
    KheDrsTaskSolnSetAddLast(fixed_assts, dts);

    /* and build dominance cache */
    KheDrsResourceBuildDominanceTestCache(dr, drd, de->solver);
  }
  else
  {
    /* mark the resource as free */
    dr->expand_role = KHE_DRS_RESOURCE_EXPAND_FREE;

    /* add dr to free_resources */
    KheDrsResourceSetAddLast(free_resources, dr);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceExpandBeginFree(KHE_DRS_RESOURCE dr,                  */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)     */
/*                                                                           */
/*  Carry out the second part of beginning an expansion using open resource  */
/*  dr, which is now known to be free.  So add its assignments in.           */
/*                                                                           */
/*  Implementation note.  This code is separated from the previous function  */
/*  because there are cases where, by adding fixed assignments to the        */
/*  expander, the first function might allow more pruning in the second.     */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsMTaskAcceptResourceBegin(KHE_DRS_MTASK dmt,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY *dtd);
static void KheDrsMTaskAcceptResourceEnd(KHE_DRS_MTASK dmt,
  KHE_DRS_TASK_ON_DAY dtd);
static bool KheDrsSignatureOptionallyFree(KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceExpandBeginFree(KHE_DRS_RESOURCE dr,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)
{
  int i, j;  KHE_DRS_RESOURCE_ON_DAY drd;  KHE_DRS_TASK_ON_DAY dtd;
  KHE_DRS_MTASK dmt;  KHE_DRS_SHIFT ds;  KHE_DRS_SIGNATURE prev_sig, sig;

  /* unfixed assignment; make signatures and mtask solns as required */
  HnAssert(dr->expand_role == KHE_DRS_RESOURCE_EXPAND_FREE,
    "KheDrsResourceExpandBeginFree internal error");
  drd = KheDrsResourceOnDay(dr, next_day);
  prev_sig = HaArray(prev_soln->sig_set.signatures, dr->open_resource_index);
  HaArrayForEach(next_day->shifts, ds, i)
  {
    sig = NULL;
    HaArrayForEach(ds->open_mtasks, dmt, j)
    {
      if( KheDrsMTaskSuitsFreeResource(dmt, drd->encl_dr) &&
	  KheDrsMTaskAcceptResourceBegin(dmt, drd, &dtd) )
      {
	if( sig == NULL )
	  sig = KheDrsResourceSignatureMake(drd, dtd, prev_sig, de->solver);
	KheDrsResourceOptionallyAddMTaskSoln(sig, drd, dmt, NULL, false, de);
	KheDrsMTaskAcceptResourceEnd(dmt, dtd);
      }
    }
    if( sig != NULL && !KheDrsSignatureOptionallyFree(sig, de->solver) )
      KheDrsResourceAddExpandSignature(dr, sig);
  }

  /* make one signature and mtask soln for a free day */
  sig = KheDrsResourceSignatureMake(drd, NULL, prev_sig, de->solver);
  dr->expand_free_mtask_soln =
    KheDrsResourceOptionallyAddMTaskSoln(sig, drd, NULL, NULL, false, de);
  if( !KheDrsSignatureOptionallyFree(sig, de->solver) )  /* yes, we need this */
    KheDrsResourceAddExpandSignature(dr, sig);

  /* sort expand_mtask_solns by increasing cost and keep only the best */
  KheDrsSortAndReduceMTaskSolns(dr, de->solver);

  /* move any unavoidable cost into the expander */
  KheDrsResourceAdjustSignatureCosts(dr, de);

  /* add a dominance test cache */
  KheDrsResourceBuildDominanceTestCache(dr, drd, de->solver);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceExpandEnd(KHE_DRS_RESOURCE dr, KHE_DRS_EXPANDER de)   */
/*                                                                           */
/*  End dr's part of the expansion of one solution.                          */
/*                                                                           */
/*****************************************************************************/
static void KheDrsDim2TableClear(KHE_DRS_DIM2_TABLE d2,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceExpandEnd(KHE_DRS_RESOURCE dr, KHE_DRS_EXPANDER de)
{
  KHE_DRS_MTASK_SOLN dms;

  /* mark the resource as not involved in any expansion */
  dr->expand_role = KHE_DRS_RESOURCE_EXPAND_NO;

  /* clear out the dominance test cache */
  if( USE_DOM_CACHING )
    KheDrsDim2TableClear(dr->expand_dom_test_cache, de->solver);

  /* clear signatures */
  KheDrsResourceClearExpandSignatures(dr, de->solver);

  /* clear assignments to mtasks */
  while( HaArrayCount(dr->expand_mtask_solns) > 0 )
  {
    dms = HaArrayLastAndDelete(dr->expand_mtask_solns);
    KheDrsMTaskSolnFree(dms, de->solver);
  }
  dr->expand_free_mtask_soln = NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE_ON_DAY"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayMake(KHE_DRS_RESOURCE dr,     */
/*    KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                      */
/*                                                                           */
/*  Make a new resource on day object with these attributes.                 */
/*  This function assumes that dr is not assigned any task on day.           */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SIGNER KheDrsSignerMake(KHE_DRS_DAY day,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_SHIFT ds,
  KHE_DRS_SHIFT_PAIR dsp, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayMake(KHE_DRS_RESOURCE dr,
  KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_ON_DAY res;
  HaMake(res, drs->arena);
  res->encl_dr = dr;
  res->day = day;
  res->open = false;
  res->closed_dtd = NULL;
  res->preasst_dtd = NULL;
  HaArrayInit(res->external_today, drs->arena);
  res->signer = KheDrsSignerMake(NULL, res, NULL, NULL, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsResourceOnDayIndex(KHE_DRS_RESOURCE_ON_DAY drd)                */
/*                                                                           */
/*  Return the open resource index of drd.  This will be -1 if the resource  */
/*  is not open                                                              */
/*                                                                           */
/*****************************************************************************/

static int KheDrsResourceOnDayIndex(KHE_DRS_RESOURCE_ON_DAY drd)
{
  return drd->encl_dr->open_resource_index;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsResourceOnDayId(KHE_DRS_RESOURCE_ON_DAY drd)                 */
/*                                                                           */
/*  Return an Id for drd.  The result is held in a static array and is       */
/*  best used for debugging only.                                            */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsResourceOnDayId(KHE_DRS_RESOURCE_ON_DAY drd)
{
  static char buff[200];
  snprintf(buff, 200, "%s:%s", KheDrsResourceId(drd->encl_dr),
    KheDrsDayId(drd->day));
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsResourceOnDayHasTimeExpr(KHE_DRS_RESOURCE_ON_DAY drd,         */
/*    KHE_DRS_EXPR_TAG tag, KHE_TIME time,                                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_EXPR *res_e)                  */
/*                                                                           */
/*  If drd has a BUSY_TIME, FREE_TIME, or WORK_TIME expression with the      */
/*  given tag, dom_test, and time, then return true and set *res_e to it.    */
/*  Otherwise return false with *res_e set to NULL.                          */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsResourceOnDayHasTimeExpr(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_EXPR_TAG tag, KHE_TIME time,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_EXPR *res_e)
{
  int i;  KHE_DRS_EXPR e;  KHE_DRS_EXPR_BUSY_TIME ebt;
  KHE_DRS_EXPR_FREE_TIME eft;  KHE_DRS_EXPR_WORK_TIME ewt;
  HaArrayForEach(drd->external_today, e, i)
    if( e->tag == tag )
    {
      switch( e->tag )
      {
	case KHE_DRS_EXPR_BUSY_TIME_TAG:

	  ebt = (KHE_DRS_EXPR_BUSY_TIME) e;
	  if( ebt->time == time )
	    return *res_e = e, true;
	  break;

	case KHE_DRS_EXPR_FREE_TIME_TAG:

	  eft = (KHE_DRS_EXPR_FREE_TIME) e;
	  if( eft->time == time )
	    return *res_e = e, true;
	  break;

	case KHE_DRS_EXPR_WORK_TIME_TAG:

	  ewt = (KHE_DRS_EXPR_WORK_TIME) e;
	  if( ewt->time == time )
	    return *res_e = e, true;
	  break;

	default:

	  HnAbort("KheDrsResourceOnDayHasTimeExpr: invalid tag (%d)", e->tag);
      }
    }
  return *res_e = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsResourceOnDayHasDayExpr(KHE_DRS_RESOURCE_ON_DAY drd,          */
/*    KHE_DRS_EXPR_TAG tag, KHE_TIME_GROUP tg,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_EXPR *res_e)                  */
/*                                                                           */
/*  If drd has a BUSY_DAY, FREE_DAY, or WORK_DAY expression with the given   */
/*  tag, dom_test, and time group, then return true and set *res_e to it.    */
/*  Otherwise return false with *res_e set to NULL.                          */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsResourceOnDayHasDayExpr(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_EXPR_TAG tag, KHE_TIME_GROUP tg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_EXPR *res_e)
{
  int i;  KHE_DRS_EXPR e;  KHE_DRS_EXPR_BUSY_DAY ebd;
  KHE_DRS_EXPR_FREE_DAY efd;  KHE_DRS_EXPR_WORK_DAY ewd;
  HaArrayForEach(drd->external_today, e, i)
    if( e->tag == tag )
    {
      switch( e->tag )
      {
	case KHE_DRS_EXPR_BUSY_DAY_TAG:

	  ebd = (KHE_DRS_EXPR_BUSY_DAY) e;
	  if( KheTimeGroupEqual(ebd->time_group, tg) )
	    return *res_e = e, true;
	  break;

	case KHE_DRS_EXPR_FREE_DAY_TAG:

	  efd = (KHE_DRS_EXPR_FREE_DAY) e;
	  if( KheTimeGroupEqual(efd->time_group, tg) )
	    return *res_e = e, true;
	  break;

	case KHE_DRS_EXPR_WORK_DAY_TAG:

	  ewd = (KHE_DRS_EXPR_WORK_DAY) e;
	  if( KheTimeGroupEqual(ewd->time_group, tg) )
	    return *res_e = e, true;
	  break;

	default:

	  HnAbort("KheDrsResourceHasDayExpr: invalid tag (%d)", e->tag);
      }
    }
  return *res_e = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayOpen(KHE_DRS_RESOURCE_ON_DAY drd,                */
/*    int open_resource_index, int open_day_index, KHE_INTERVAL ddr,         */
/*    KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)        */
/*                                                                           */
/*  Open drd, if that proves to be possible.                                 */
/*                                                                           */
/*****************************************************************************/
static char *KheDrsTaskId(KHE_DRS_TASK dt);
static bool KheDrsTaskUnAssign(KHE_DRS_TASK dt, bool task);
static void KheDrsExprGatherForOpening(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsPackedSolnSetTaskOnDay(KHE_DRS_PACKED_SOLN packed_soln,
  int open_day_index, int open_resource_index, KHE_DRS_TASK_ON_DAY dtd);
static void KheDrsSignerMakeCorrelated(KHE_DRS_SIGNER dsg);
static void KheDrsExprDebug(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp);

static void KheDrsResourceOnDayOpen(KHE_DRS_RESOURCE_ON_DAY drd,
  int open_resource_index, int open_day_index, KHE_INTERVAL ddr,
  KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK dt;  KHE_DRS_EXPR e;  int i;  KHE_DRS_TASK_ON_DAY dtd;
  KHE_INTERVAL open_day_range;  KHE_RESOURCE r;

  /* unassign any affected task; possibly add the assts to init_soln */
  HnAssert(!drd->open, "KheDrsResourceOnDayOpen internal error");
  dtd = drd->closed_dtd;
  if( dtd != NULL )
  {
    dt = dtd->encl_dt;
    if( !KheIntervalSubset(dt->encl_dmt->day_range, ddr) )
    {
      /* dt does not satisfy condition (1) - busy day range */
      if( DEBUG11 )
	fprintf(stderr, "  did not unassign task %s (%s not subset %s)\n",
	  KheDrsTaskId(dt),
	  KheIntervalShow(dt->encl_dmt->day_range, drs->days_frame),
	  KheIntervalShow(ddr, drs->days_frame));
    }
    else if( KheTaskIsPreassigned(dt->task, &r) )
    {
      /* dt does not satisfy condition (2) - preassignment */
      if( DEBUG11 )
	fprintf(stderr, "  did not unassign task %s (preassignment)\n",
	  KheDrsTaskId(dt));
    }
    else if( !KheDrsTaskUnAssign(dt, true) )
    {
      /* dt does not satisfy condition (3) - will not unassign */
      if( DEBUG11 )
	fprintf(stderr, "  did not unassign task %s (no unassign)\n",
	  KheDrsTaskId(dt));
    }
    else
    {
      /* dt has been successfully unassigned */
      drs->solve_start_cost -= dt->asst_cost;
      if( RERUN_DEBUG(drs) && dt->asst_cost > 0 )
	fprintf(stderr, "  KheDrsResourceOnDayOpen: drs->solve_start_cost "
	  "-= dt->asst_cost (%.5f) = %.5f from %s\n",
	  KheCostShow(dt->asst_cost), KheCostShow(drs->solve_start_cost), 
	  KheDrsTaskId(dt));
      if( init_soln != NULL )
	KheDrsPackedSolnSetTaskOnDay(init_soln, open_day_index,
	  open_resource_index, dtd);
    }
  }

  /* set make_correlated field of signer */
  if( drs->solve_correlated_exprs )
    KheDrsSignerMakeCorrelated(drd->signer);

  if( drd->closed_dtd == NULL )
  {
    /* open drd and gather its expressions for opening */
    drd->open = true;
    open_day_range = KheIntervalMake(open_day_index, open_day_index);
    HaArrayForEach(drd->external_today, e, i)
    {
      e->open_children_by_day.index_range = open_day_range;
      KheDrsExprGatherForOpening(e, drs);
      if( DEBUG11 )
	KheDrsExprDebug(e, drs, 2, 2, stderr);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayClose(KHE_DRS_RESOURCE_ON_DAY drd,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Close drd.                                                               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSignerClear(KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceOnDayClose(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSignerClear(drd->signer, drs);
  drd->open = false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayAddExpr(KHE_DRS_RESOURCE_ON_DAY drd,             */
/*    KHE_DRS_EXPR e)                                                        */
/*                                                                           */
/*  Add e, known to be new, to drd.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceOnDayAddExpr(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_EXPR e)
{
  HaArrayAddLast(drd->external_today, e);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayAddOpenExpr(KHE_DRS_RESOURCE_ON_DAY drd,         */
/*    KHE_DRS_EXPR e)                                                        */
/*                                                                           */
/*  Add internal expression e to drd.                                        */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsResourceOnDayAddOpenExpr(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_EXPR e)
{
  KheDrsSignerAddOpenExpr(drd->signer, e);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsResourceOnDayAddDomTest(KHE_DRS_RESOURCE_ON_DAY drd,           */
/*    KHE_DRS_DOM_TEST dom_test)                                             */
/*                                                                           */
/*  Add dom_test to drd and return its sig index.                            */
/*                                                                           */
/*****************************************************************************/

/* ***
static int KheDrsResourceOnDayAddDomTest(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsSignerAddDomTest(drd->signer, dom_test, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNER KheDrsResourceOnDaySigner(KHE_DRS_RESOURCE_ON_DAY drd)    */
/*                                                                           */
/*  Return drd's signer.                                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNER KheDrsResourceOnDaySigner(KHE_DRS_RESOURCE_ON_DAY drd)
{
  return drd->signer;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayLeafSet(KHE_DRS_RESOURCE_ON_DAY drd,             */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)              */
/*                                                                           */
/*  Call KheDrsExprLeafSet for each external expression affected by drd.     */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceOnDayLeafSet(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR e;  int i;
  if( drd->closed_dtd == NULL && dtd != NULL )
    HaArrayForEach(drd->external_today, e, i)
      KheDrsExprLeafSet(e, dtd, drd->encl_dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayLeafClear(KHE_DRS_RESOURCE_ON_DAY drd,           */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)              */
/*                                                                           */
/*  Call KheDrsExprLeafClear for each external expression affected by drd.   */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprLeafClear(KHE_DRS_EXPR e,KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceOnDayLeafClear(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR e;  int i;
  if( drd->closed_dtd == NULL && dtd != NULL )
    HaArrayForEach(drd->external_today, e, i)
      KheDrsExprLeafClear(e, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayDebug(KHE_DRS_RESOURCE_ON_DAY drd,               */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of drd onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/
static void KheDrsTaskOnDayDebug(KHE_DRS_TASK_ON_DAY dtd,
  int verbosity, int indent, FILE *fp);

static void KheDrsResourceOnDayDebug(KHE_DRS_RESOURCE_ON_DAY drd,
  int verbosity, int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "%s: ", KheDrsDayId(drd->day));
  if( drd->closed_dtd != NULL )
    KheDrsTaskOnDayDebug(drd->closed_dtd, 1, -1, fp);
  else
    fprintf(fp, "-");
  fprintf(fp, " (external_today %d)", HaArrayCount(drd->external_today));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE_SET"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_SET KheDrsResourceSetMake(                              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make or get a new, empty resource set.                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE_SET KheDrsResourceSetMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_SET res;
  if( HaArrayCount(drs->resource_set_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->resource_set_free_list);
    HaArrayClear(res->resources);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->resources, drs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetFree(KHE_DRS_RESOURCE_SET rs,                      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free rs.                                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetFree(KHE_DRS_RESOURCE_SET rs,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->resource_set_free_list, rs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetClear(KHE_DRS_RESOURCE_SET rs)                     */
/*                                                                           */
/*  Clear rs.                                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetClear(KHE_DRS_RESOURCE_SET rs)
{
  HaArrayClear(rs->resources);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsResourceSetCount(KHE_DRS_RESOURCE_SET rs)                      */
/*                                                                           */
/*  Return the number of resources in rs.                                    */
/*                                                                           */
/*****************************************************************************/

static int KheDrsResourceSetCount(KHE_DRS_RESOURCE_SET rs)
{
  return HaArrayCount(rs->resources);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE KheDrsResourceSetResource(KHE_DRS_RESOURCE_SET rs,      */
/*    int i)                                                                 */
/*                                                                           */
/*  Return the i'th resource of rs.                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE KheDrsResourceSetResource(KHE_DRS_RESOURCE_SET rs,
  int i)
{
  return HaArray(rs->resources, i);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetAddLast(KHE_DRS_RESOURCE_SET rs,                   */
/*    KHE_DRS_RESOURCE dr)                                                   */
/*                                                                           */
/*  Add dr to the end of rs.                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetAddLast(KHE_DRS_RESOURCE_SET rs,
  KHE_DRS_RESOURCE dr)
{
  HaArrayAddLast(rs->resources, dr);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetDeleteLast(KHE_DRS_RESOURCE_SET rs)                */
/*                                                                           */
/*  Delete the last element of rs.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetDeleteLast(KHE_DRS_RESOURCE_SET rs)
{
  HnAssert(KheDrsResourceSetCount(rs) > 0,
    "KheDrsResourceSetDeleteLast internal error");
  HaArrayDeleteLast(rs->resources);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsResourceSetContains(KHE_DRS_RESOURCE_SET rs,                  */
/*    KHE_DRS_RESOURCE dr)                                                   */
/*                                                                           */
/*  Return true if rs contains dr.                                           */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsResourceSetContains(KHE_DRS_RESOURCE_SET rs,
  KHE_DRS_RESOURCE dr)
{
  int pos;
  return HaArrayContains(rs->resources, dr, &pos);
}


/*****************************************************************************/
/*                                                                           */
/*  KheDrsResourceSetForEach(rs, dr, i)                                      */
/*                                                                           */
/*  Iterate over the resources dr of rs; like HaArrayForEach.                */
/*                                                                           */
/*****************************************************************************/

#define KheDrsResourceSetForEach(rs, dr, i)				\
  HaArrayForEach((rs)->resources, (dr), (i))


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDaySetDebug(KHE_DRS_RESOURCE_ON_DAY_SET drds,       */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug printf of rs onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetDebug(KHE_DRS_RESOURCE_SET rs,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_RESOURCE dr;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "{");
  KheDrsResourceSetForEach(rs, dr, i)
  {
    if( i > 0 )
      fprintf(fp, ", ");
    fprintf(fp, "%s", KheDrsResourceId(dr));
  }
  fprintf(fp, "}");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "events"                                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK_EXPAND_ROLE"                                     */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsTaskExpandRoleShow(KHE_DRS_TASK_EXPAND_ROLE dter)
{
  switch( dter )
  {
    case KHE_DRS_TASK_EXPAND_NO_VALUE:	return "expand_no_value";
    case KHE_DRS_TASK_EXPAND_FIXED:	return "expand_fixed";
    case KHE_DRS_TASK_EXPAND_MUST:	return "expand_must";
    case KHE_DRS_TASK_EXPAND_FREE:	return "expand_free";
    default:				return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskAddDays(KHE_TASK task, KHE_DRS_TASK dt,                      */
/*    int first_solve_index, int last_solve_index,                           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add the days covered by task and its descendants to dt.                  */
/*                                                                           */
/*  Return false if this can't be done because some resource is              */
/*  preassigned to two tasks on some day.                                    */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_TASK_ON_DAY KheDrsTaskOnDayMake(KHE_DRS_TASK encl_dt,
  KHE_DRS_DAY day, KHE_TASK task, KHE_TIME time,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static bool KheTaskAddDays(KHE_TASK task, KHE_DRS_TASK dt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_TASK child_task;  int i, durn, day_index;  KHE_DRS_DAY day;
  KHE_MEET meet;  KHE_TIME start_time, time;  KHE_DRS_TASK_ON_DAY dtd;
  KHE_RESOURCE r;  KHE_DRS_RESOURCE dr;  KHE_DRS_RESOURCE_ON_DAY drd;

  /* do it for task */
  meet = KheTaskMeet(task);
  HnAssert(meet != NULL, "KheTaskAddDays internal error (no meet)");
  start_time = KheMeetAsstTime(meet);
  HnAssert(start_time != NULL,  "KheTaskAddDays internal error (no time)");
  durn = KheMeetDuration(meet);
  for( i = 0;  i < durn;  i++ )
  {
    time = KheTimeNeighbour(start_time, i);
    day_index = KheFrameTimeIndex(drs->days_frame, time);
    day = HaArray(drs->all_days, day_index);
    dtd = KheDrsTaskOnDayMake(dt, day, task, time, drs);
    HaArrayAddLast(dt->days, dtd);
    if( KheTaskIsPreassigned(task, &r) )
    {
      /* add preasst_dtd */
      dr = KheDrsResource(r, drs);
      drd = KheDrsResourceOnDay(dr, day);
      if( drd->preasst_dtd != NULL )
	return false;
      drd->preasst_dtd = dtd;
    }
  }

  /* do it for task's descendants */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( !KheTaskAddDays(child_task, dt, drs) )
      return false;
  }

  /* all good */
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskMake(KHE_DRS_MTASK encl_dmt, int index_in_encl_dmt,       */
/*    KHE_TASK task, KHE_COST non_asst_cost, KHE_COST asst_cost,             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_TASK *dt)                     */
/*                                                                           */
/*  Make a new drs task with these attributes, add it to drs->all_root_tasks */
/*  at the same index as it has in the solution, and return it in *dt.       */
/*                                                                           */
/*  Return true if all this has been done successfully.  The reasons why     */
/*  false could be returned are                                              */
/*                                                                           */
/*    (1) the task is preassigned but not assigned (currently not in force); */
/*                                                                           */
/*    (2) the call to KheDrsTaskAssign uncovered a case of a resource        */
/*        assigned to two tasks on the same day.                             */
/*                                                                           */
/*****************************************************************************/
static void KheDrsTaskOnDaySetLastDay(KHE_DRS_TASK_ON_DAY dtd);
static void KheDrsAddWeight(KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_COST weight);
static int KheDrsTaskOnDayCmp(const void *t1, const void *t2);
static bool KheDrsTaskAssign(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr, bool task);
static void KheDrsTaskDebug(KHE_DRS_TASK dt, int verbosity,
  int indent, FILE *fp);

static bool KheDrsTaskMake(KHE_DRS_MTASK encl_dmt, int index_in_encl_dmt,
  KHE_TASK task, KHE_COST non_asst_cost, KHE_COST asst_cost,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_TASK *dt)
{
  KHE_DRS_TASK res;  int task_index;  KHE_RESOURCE r;

  /* get the basic object and initialize its fields */
  HnAssert(KheTaskIsProperRoot(task), "KheDrsTaskMake internal error");
  HaMake(res, drs->arena);
  res->encl_dmt = encl_dmt;
  res->index_in_encl_dmt = index_in_encl_dmt;
  res->expand_role = KHE_DRS_TASK_EXPAND_NO_VALUE;
  res->open = false;
  res->task = task;
  res->closed_dr = NULL;
  res->non_asst_cost = non_asst_cost;
  res->asst_cost = asst_cost;
  KheDrsAddWeight(drs, non_asst_cost);
  KheDrsAddWeight(drs, asst_cost);
  HaArrayInit(res->days, drs->arena);

  /* if task is preassigned but not assigned, fail */
  /* *** let's try to do without this
  if( r == NULL && KheTaskIsPreassigned(task, &r) )
    return *dt = false, NULL;
  *** */

  /* add, sort, and organize the task on day objects */
  if( !KheTaskAddDays(task, res, drs) )
    return *dt = NULL, false;
  HaArraySort(res->days, &KheDrsTaskOnDayCmp);
  KheDrsTaskOnDaySetLastDay(HaArrayLast(res->days));

  /* add res to drs->all_root_tasks */
  task_index = KheTaskSolnIndex(task);
  HaArrayFill(drs->all_root_tasks, task_index + 1, NULL);
  HnAssert(HaArray(drs->all_root_tasks, task_index) == NULL,
    "KheDrsTaskMake internal error");
  HaArrayPut(drs->all_root_tasks, task_index, res);

  /* if task is assigned, make that assignment in res */
  r = KheTaskAsstResource(task);
  if( r != NULL )
  {
    if( !KheDrsTaskAssign(res, KheDrsResource(r, drs), false) )
      return *dt = false, NULL;
  }

  if( DEBUG86 )
  {
    fprintf(stderr, "  made drs task ");
    KheDrsTaskDebug(res, 2, 0, stderr);
  }

  /* and also return res */
  return *dt = res, true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskAssign(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr, bool task)   */
/*                                                                           */
/*  Assign dr (which must be non-NULL) to dt, by setting the closed asst     */
/*  fields in dt, dt->days, and dr.  If task is true, also assign dt's KHE   */
/*  task to dr's KHE resource, aborting if that does not succeed.            */
/*                                                                           */
/*  The enclosing mtask has a list of unassigned tasks; but do not           */
/*  remove dt from it, because we are being lazy:  the unassigned tasks      */
/*  will be sorted out later, by KheDrsMTaskOrganizeAvailableTasks.         */
/*                                                                           */
/*  Return true if the assignment was successful.  The only reason for       */
/*  failure is if the assignment gives dr two tasks on one day.              */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskAssign(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr, bool task)
{
  KHE_DRS_TASK_ON_DAY dtd;  int i;  KHE_DRS_RESOURCE_ON_DAY drd;
  if( DEBUG15 || DEBUG96(dt) )
    fprintf(stderr, "[ KheDrsTaskAssign(%s (%d days), %s, %s)\n",
      KheDrsTaskId(dt), HaArrayCount(dt->days), KheDrsResourceId(dr),
      bool_show(task));
  HnAssert(dt->closed_dr == NULL, "KheDrsTaskAssign internal error 1");
  HnAssert(dr != NULL, "KheDrsTaskAssign internal error 2");
  if( task && !KheTaskAssignResource(dt->task, dr->resource) )
    HnAbort("KheDrsTaskAssign internal error 3 (cannot assign %s to %s)",
      KheDrsResourceId(dr), KheDrsTaskId(dt));
  dt->closed_dr = dr;
  HaArrayForEach(dt->days, dtd, i)
  {
    drd = KheDrsResourceOnDay(dr, dtd->day);
    HnAssert(dtd->closed_drd == NULL, "KheDrsTaskAssign internal error 4");
    if( drd->closed_dtd != NULL )
    {
      if( DEBUG15 || DEBUG96(dt) )
      {
	fprintf(stderr, "] KheDrsTaskAssign returning false (on day %s, "
	  "%s already assigned %s:\n", KheDrsDayId(dtd->day),
	  KheDrsResourceId(dr), KheDrsTaskId(drd->closed_dtd->encl_dt));
	KheDrsTaskDebug(drd->closed_dtd->encl_dt, 2, 2, stderr);
      }
      return false;
    }
    dtd->closed_drd = drd;
    drd->closed_dtd = dtd;
  }
  if( DEBUG15 || DEBUG96(dt) )
    fprintf(stderr, "] KheDrsTaskAssign returning true\n");
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskUnAssign(KHE_DRS_TASK dt, bool task)                      */
/*                                                                           */
/*  Unassign dt (which must be assigned some drs resource dr), by clearing   */
/*  the closed asst fields in dt, dt->days, and dr.  If task is true, also   */
/*  unassign dt's KHE task, or do nothing if the task will not unassign.     */
/*  Return true if the unassignment was successful.                          */
/*                                                                           */
/*  The enclosing mtask has a list of unassigned tasks; add dt to it.        */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskUnAssign(KHE_DRS_TASK dt, bool task)
{
  KHE_DRS_TASK_ON_DAY dtd;  int i;
  KHE_DRS_RESOURCE dr;  KHE_DRS_RESOURCE_ON_DAY drd;
  dr = dt->closed_dr;
  HnAssert(dr != NULL, "KheDrsTaskUnAssign internal error 1");
  if( DEBUG15 || DEBUG96(dt) )
    fprintf(stderr, "[ KheDrsTaskUnAssign(%s, %s, %s)\n", KheDrsTaskId(dt),
      KheDrsResourceId(dr), bool_show(task));
  if( task )
  {
    HnAssert(KheTaskAsstResource(dt->task) == dr->resource,
      "KheDrsTaskUnAssign internal error 2");
    if( !KheTaskUnAssignResource(dt->task) )
    {
      if( DEBUG15 || DEBUG96(dt) )
	fprintf(stderr, "  failed to unassign %s\n", KheDrsTaskId(dt));
      return false;
    }
    if( DEBUG16 )
      fprintf(stderr, "  after unassigning %s, cost = %.5f\n",
	KheDrsTaskId(dt), KheCostShow(KheSolnCost(KheTaskSoln(dt->task))));
  }
  HaArrayForEach(dt->days, dtd, i)
  {
    drd = KheDrsResourceOnDay(dr, dtd->day);
    HnAssert(dtd->closed_drd == drd, "KheDrsTaskUnAssign internal error 3");
    HnAssert(drd->closed_dtd == dtd, "KheDrsTaskUnAssign internal error 4");
    dtd->closed_drd = NULL;
    drd->closed_dtd = NULL;
  }
  dt->closed_dr = NULL;
  HaArrayAddLast(dt->encl_dmt->available_tasks, dt);
  if( DEBUG15 || DEBUG96(dt) )
    fprintf(stderr, "] KheDrsTaskUnAssign returning\n");
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsTaskIndexInEnclCmp(const void *t1, const void *t2)             */
/*                                                                           */
/*  Comparison function for sorting an array of drs tasks into increasing    */
/*  order of their index in their enclosing mtask.                           */
/*                                                                           */
/*  In addition, it moves any assigned tasks to the end.                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsTaskIndexInEnclCmp(const void *t1, const void *t2)
{
  KHE_DRS_TASK dt1 = * (KHE_DRS_TASK *) t1;
  KHE_DRS_TASK dt2 = * (KHE_DRS_TASK *) t2;
  if( dt1->closed_dr != NULL )
  {
    if( dt2->closed_dr != NULL )
    {
      /* dt1 is assigned, dt2 is assigned */
      return 0;
    }
    else
    {
      /* dt1 is assigned, dt2 is not assigned */
      return 1;
    }
  }
  else
  {
    if( dt2->closed_dr != NULL )
    {
      /* dt1 is not assigned, dt2 is assigned */
      return -1;
    }
    else
    {
      /* dt1 is not assigned, dt2 is not assigned */
      return dt1->index_in_encl_dmt - dt2->index_in_encl_dmt;
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOpen(KHE_DRS_TASK dt, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Open task for solving.  We'll have checked previously that its days      */
/*  all lie in the current open interval, and it must currently be closed    */
/*  but unassigned; so opening dt basically just gathers for opening all     */
/*  expressions that depend on dt's assignment.                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOpen(KHE_DRS_TASK dt, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_EXPR e;  int i, j, di;
  KHE_INTERVAL open_day_range;

  /* open dt */
  if( DEBUG19 )
    fprintf(stderr, "[ KheDrsTaskOpen(%s, drs)\n", KheDrsTaskId(dt));
  HnAssert(!dt->open, "KheDrsTaskOpen internal error 1");
  HnAssert(dt->closed_dr == NULL, "KheDrsTaskOpen internal error 2");
  dt->open = true;

  /* gather external expressions for opening */
  HaArrayForEach(dt->days, dtd, i)
  {
    HnAssert(dtd->closed_drd == NULL, "KheDrsTaskOpen internal error 3");
    di = dtd->day->open_day_index;
    HnAssert(di >= 0, "KheDrsTaskOpen internal error 3");
    open_day_range = KheIntervalMake(di, di);
    HaArrayForEach(dtd->external_today, e, j)
    {
      e->open_children_by_day.index_range = open_day_range;
      KheDrsExprGatherForOpening(e, drs);
    }
  }
  if( DEBUG19 )
    fprintf(stderr, "] KheDrsTaskOpen\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClose(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr)               */
/*                                                                           */
/*  If dt is open, close it, with a call to KheDrsTaskAssign if dr != NULL.  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClose(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr)
{
  if( dt->open )
  {
    if( DEBUG19 )
      fprintf(stderr, "[ KheDrsTaskClose(%s, drs)\n", KheDrsTaskId(dt));
    dt->open = false;
    if( dr != NULL )
    {
      if( !KheDrsTaskAssign(dt, dr, true) )
      {
	fprintf(stderr, "  KheDrsTaskClose failing:\n");
        KheDrsTaskDebug(dt, 4, 2, stderr);
	HnAbort("#%d KheDrsTaskClose internal error",
	  KheSolnDiversifier(KheTaskSoln(dt->task)));
      }
    }
    if( DEBUG19 )
      fprintf(stderr, "] KheDrsTaskClose\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskRunningOnDay(KHE_DRS_TASK dt, KHE_DRS_DAY day,            */
/*    KHE_DRS_TASK_ON_DAY *dtd)                                              */
/*                                                                           */
/*  If dt is running on day, return true with *dtd set to its task on day    */
/*  object for that day.  Otherwise return false with *dtd set to NULL.      */
/*                                                                           */
/*  Implementation note.  The task on days are sorted chronologically,       */
/*  but we don't bother to use that property here.                           */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskRunningOnDay(KHE_DRS_TASK dt, KHE_DRS_DAY day,
  KHE_DRS_TASK_ON_DAY *dtd)
{
  KHE_DRS_TASK_ON_DAY dtd2;  int i;
  HaArrayForEach(dt->days, dtd2, i)
    if( dtd2->day == day )
      return *dtd = dtd2, true;
  return *dtd = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskDoMinCost(KHE_TASK task, KHE_DRS_ASST_OP op,                 */
/*    KHE_RESOURCE r, KHE_COST *cost)                                        */
/*                                                                           */
/*  Do the actual work of KheTaskMinCost, defined just below.                */
/*                                                                           */
/*  Implementation note.  For the step cost function there can always        */
/*  be no change in cost, when we are beyond the upper limit.  So the        */
/*  minimum cost change is never positive.                                   */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskDoMinCost(KHE_TASK task, KHE_DRS_ASST_OP op,
  KHE_RESOURCE ra, KHE_RESOURCE rb, KHE_COST *cost)
{
  KHE_EVENT_RESOURCE er;  KHE_SOLN soln;  int i, durn;  KHE_MONITOR m;
  KHE_TASK child_task;  KHE_RESOURCE_GROUP rg;  KHE_COST weight;
  KHE_PREFER_RESOURCES_MONITOR prm;
  KHE_LIMIT_RESOURCES_MONITOR lrm;  KHE_LIMIT_RESOURCES_CONSTRAINT lrc;

  /* do it for task */
  er = KheTaskEventResource(task);
  if( er != NULL )
  {
    soln = KheTaskSoln(task);
    for( i = 0;  i < KheSolnEventResourceMonitorCount(soln, er);  i++ )
    {
      m = KheSolnEventResourceMonitor(soln, er, i);
      weight = KheMonitorCombinedWeight(m);
      durn = KheTaskDuration(task);
      if( KheMonitorAttachedToSoln(m) && weight > 0 )
	switch( KheMonitorCostFunction(m) )
	{
	  case KHE_STEP_COST_FUNCTION:

	    switch( KheMonitorTag(m) )
	    {
	      case KHE_ASSIGN_RESOURCE_MONITOR_TAG:

		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:

		    /* no change */
		    break;

		  case KHE_DRS_ASST_OP_ASSIGN:

		    *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    /* no change */
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      case KHE_PREFER_RESOURCES_MONITOR_TAG:

		prm = (KHE_PREFER_RESOURCES_MONITOR) m;
		rg = KhePreferResourcesMonitorDomain(prm);
		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_ASSIGN:

		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      case KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR_TAG:

		/* not handling this one */
		return false;

	      case KHE_LIMIT_RESOURCES_MONITOR_TAG:

		lrm = (KHE_LIMIT_RESOURCES_MONITOR) m;
		lrc = KheLimitResourcesMonitorConstraint(lrm);
		rg = KheLimitResourcesConstraintDomain(lrc);
		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:
		  case KHE_DRS_ASST_OP_ASSIGN:

		    if( KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    if( KheResourceGroupContains(rg, ra) !=
			KheResourceGroupContains(rg, rb) )
		      *cost -= weight * durn;
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      default:

		HnAbort("KheDrsTaskDoMinCost: unexpected monitor tag (%s)",
		  KheMonitorTagShow(KheMonitorTag(m)));
	    }
	    break;

	  case KHE_LINEAR_COST_FUNCTION:

	    switch( KheMonitorTag(m) )
	    {
	      case KHE_ASSIGN_RESOURCE_MONITOR_TAG:

		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:

		    *cost += weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_ASSIGN:

		    *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    /* no change */
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      case KHE_PREFER_RESOURCES_MONITOR_TAG:

		prm = (KHE_PREFER_RESOURCES_MONITOR) m;
		rg = KhePreferResourcesMonitorDomain(prm);
		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_ASSIGN:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost += weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    if( !KheResourceGroupContains(rg, rb) )
		      *cost += weight * durn;
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      case KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR_TAG:

		/* not handling this one */
		return false;

	      case KHE_LIMIT_RESOURCES_MONITOR_TAG:

		lrm = (KHE_LIMIT_RESOURCES_MONITOR) m;
		lrc = KheLimitResourcesMonitorConstraint(lrm);
		rg = KheLimitResourcesConstraintDomain(lrc);
		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:
		  case KHE_DRS_ASST_OP_ASSIGN:

		    if( KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    if( KheResourceGroupContains(rg, ra) !=
			KheResourceGroupContains(rg, rb) )
		      *cost -= weight * durn;
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      default:

		HnAbort("KheDrsTaskDoMinCost: unexpected monitor tag (%s)",
		  KheMonitorTagShow(KheMonitorTag(m)));
	    }
	    break;

	  case KHE_QUADRATIC_COST_FUNCTION:

	    /* not handling this one */
	    return false;
	}
    }
  }

  /* do it for task's children */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( !KheTaskDoMinCost(child_task, op, ra, rb, cost) )
      return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskMinCost(KHE_DRS_TASK dt, KHE_DRS_ASST_OP op,              */
/*    KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb, KHE_COST *cost)            */
/*                                                                           */
/*  This function evaluates three expressions from the documentation:        */
/*                                                                           */
/*    op                           function                                  */
/*    ---------------------------------------------------------------        */
/*    KHE_DRS_ASST_OP_UNASSIGN     c-(M, ra)                                 */
/*    KHE_DRS_ASST_OP_ASSIGN       c+(M, ra)                                 */
/*    KHE_DRS_ASST_OP_REPLACE      c-+(M, ra, rb)                            */
/*    ---------------------------------------------------------------        */
/*                                                                           */
/*  where M is the set of monitors that monitor dt.                          */
/*                                                                           */
/*  Instead of returning the value it sets *cost to it.  The return value    */
/*  is true if the function can calculate this cost, and false if not.       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskMinCost(KHE_DRS_TASK dt, KHE_DRS_ASST_OP op,
  KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb, KHE_COST *cost)
{
  bool res;
  if( DEBUG41 )
    fprintf(stderr, "[ KheDrsTaskMinCost\n");
  *cost = 0;
  res = KheTaskDoMinCost(dt->task, op, dra->resource,
    drb == NULL ? NULL : drb->resource, cost);
  if( DEBUG41 )
    fprintf(stderr, "] KheDrsTaskMinCost returning %s\n", bool_show(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskDebugHeader(KHE_DRS_TASK dt, FILE *fp)                    */
/*                                                                           */
/*  Debug print of the header part of dt.                                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskDebugHeader(KHE_DRS_TASK dt, FILE *fp)
{
  fprintf(fp,"DrsTask %s(non_asst_cost %.5f, asst_cost %.5f, index %d, %s, %s)",
    KheDrsTaskId(dt), KheCostShow(dt->non_asst_cost),
    KheCostShow(dt->asst_cost), dt->index_in_encl_dmt,
    dt->open ? "open" : "closed",
    dt->closed_dr == NULL ? "-" : KheDrsResourceId(dt->closed_dr));
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsTaskId(KHE_DRS_TASK dt)                                      */
/*                                                                           */
/*  Return dt's Id.                                                          */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsTaskId(KHE_DRS_TASK dt)
{
  return dt == NULL ? "-" : KheTaskId(dt->task);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskDebug(KHE_DRS_TASK dt, int verbosity,                     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dt onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskDebug(KHE_DRS_TASK dt, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_TASK_ON_DAY dtd;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ ", indent, "");
    KheDrsTaskDebugHeader(dt, fp);
    fprintf(fp, "\n");
    KheTaskDebug(dt->task, verbosity, indent + 2, fp);
    HaArrayForEach(dt->days, dtd, i)
      KheDrsTaskOnDayDebug(dtd, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    KheDrsTaskDebugHeader(dt, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK_ON_DAY"                                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_TASK_ON_DAY KheDrsTaskOnDayMake(KHE_DRS_TASK encl_dt,            */
/*    KHE_DRS_DAY day, KHE_TASK task, KHE_TIME time,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new task on day object with these attributes.                     */
/*  This function assumes that the task is not assigned anything.            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_TASK_ON_DAY KheDrsTaskOnDayMake(KHE_DRS_TASK encl_dt,
  KHE_DRS_DAY day, KHE_TASK task, KHE_TIME time,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY res;
  HnAssert(task != NULL && time != NULL, "KheDrsTaskOnDayMake internal error");
  HaMake(res, drs->arena);
  res->encl_dt = encl_dt;
  res->day = day;
  res->not_last_day = true;  /* may be reset later */
  res->task = task;
  res->time = time;
  res->closed_drd = NULL;
  HaArrayInit(res->external_today, drs->arena);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDaySetLastDay(KHE_DRS_TASK_ON_DAY dtd)                  */
/*                                                                           */
/*  Inform dtd that it is running on the last day of its task.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDaySetLastDay(KHE_DRS_TASK_ON_DAY dtd)
{
  dtd->not_last_day = false;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsTaskOnDayCmp(const void *t1, const void *t2)                   */
/*                                                                           */
/*  Comparison function for sorting an array of task on day objects by       */
/*  increasing day index.                                                    */
/*                                                                           */
/*****************************************************************************/

static int KheDrsTaskOnDayCmp(const void *t1, const void *t2)
{
  KHE_DRS_TASK_ON_DAY dtd1, dtd2;
  dtd1 = * (KHE_DRS_TASK_ON_DAY *) t1;
  dtd2 = * (KHE_DRS_TASK_ON_DAY *) t2;
  return dtd1->day->frame_index - dtd2->day->frame_index;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOptionalResourceGroupEqual(KHE_RESOURCE_GROUP rg1,            */
/*    KHE_RESOURCE_GROUP rg2)                                                */
/*                                                                           */
/*  Return true if rg1 and rg2 are both NULL, or both non-NULL and equal.    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOptionalResourceGroupEqual(KHE_RESOURCE_GROUP rg1,
  KHE_RESOURCE_GROUP rg2)
{
  if( rg1 == NULL )
    return rg2 == NULL;
  else
    return rg2 != NULL && KheResourceGroupEqual(rg1, rg2);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskOnDayHasAssignedTaskExpr(KHE_DRS_TASK_ON_DAY dtd,         */
/*    KHE_RESOURCE_GROUP rg, KHE_DRS_EXPR_ASSIGNED_TASK *res)                */
/*                                                                           */
/*  If dtd has an assigned task expression with resource group attribute     */
/*  rg, return true with *res set to that expression; else return false.     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskOnDayHasAssignedTaskExpr(KHE_DRS_TASK_ON_DAY dtd,
  KHE_RESOURCE_GROUP rg, KHE_DRS_EXPR_ASSIGNED_TASK *res)
{
  KHE_DRS_EXPR e;  int i;  KHE_DRS_EXPR_ASSIGNED_TASK eat;
  HaArrayForEach(dtd->external_today, e, i)
  {
    eat = (KHE_DRS_EXPR_ASSIGNED_TASK) e;
    if( KheDrsOptionalResourceGroupEqual(rg, eat->resource_group) )
      return *res = eat, true;
  }
  return *res = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDayAddAssignedTaskExpr(KHE_DRS_TASK_ON_DAY dtd,         */
/*    KHE_DRS_EXPR_ASSIGNED_TASK eat)                                        */
/*                                                                           */
/*  Add eat (assumed new) to dtd.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDayAddAssignedTaskExpr(KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_EXPR_ASSIGNED_TASK eat)
{
  HaArrayAddLast(dtd->external_today, (KHE_DRS_EXPR) eat);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskOnDayIsLast(KHE_DRS_TASK_ON_DAY dtd)                      */
/*                                                                           */
/*  Return true if dtd is the last task on day of its task.                  */
/*  This function is used only when debugging.                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskOnDayIsLast(KHE_DRS_TASK_ON_DAY dtd)
{
  return dtd == HaArrayLast(dtd->encl_dt->days);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDayLeafSet(KHE_DRS_TASK_ON_DAY dtd,                     */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Call KheDrsExprLeafSet for each external expression affected by dtd.     */
/*  Or do nothing if dtd is NULL.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDayLeafSet(KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR e;  int i;
  if( dtd != NULL )
    HaArrayForEach(dtd->external_today, e, i)
      KheDrsExprLeafSet(e, dtd, dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDayLeafClear(KHE_DRS_TASK_ON_DAY dtd,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Call KheDrsExprLeafClear for each external expression affected by dtd.   */
/*  Or do nothing if dtd is NULL.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDayLeafClear(KHE_DRS_TASK_ON_DAY dtd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR e;  int i;
  if( dtd != NULL )
    HaArrayForEach(dtd->external_today, e, i)
    {
      if( DEBUG79 && !e->open )
      {
	fprintf(stderr, "KheDrsTaskOnDayLeafClear: e not open:\n");
	KheDrsExprDebug(e, drs, 2, 2, stderr);
	HnAbort("KheDrsTaskOnDayLeafClear aborting now");
      }
      KheDrsExprLeafClear(e, drs);
    }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDayDebug(KHE_DRS_TASK_ON_DAY dtd,                       */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of dtd onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDayDebug(KHE_DRS_TASK_ON_DAY dtd,
  int verbosity, int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  KheTaskDebug(dtd->task, 1, -1, fp);
  if( verbosity > 1 )
  {
    fprintf(fp, " %s", KheDrsTaskExpandRoleShow(dtd->encl_dt->expand_role));
    fprintf(fp, "(time %s, day %s, closed_drd %s)", KheTimeId(dtd->time),
      KheDrsDayId(dtd->day), dtd->closed_drd == NULL ? "-" :
      KheDrsResourceId(dtd->closed_drd->encl_dr));
  }
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_MTASK"                                                */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskIsFixedAndUnassigned(KHE_TASK task)
{
  return KheTaskAssignIsFixed(task) && KheTaskAsstResource(task) == NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsMTaskMake(KHE_MTASK mt, KHE_DYNAMIC_RESOURCE_SOLVER drs)      */
/*                                                                           */
/*  Make a new drs mtask based on mt and add it to its first day.            */
/*  Also make drs tasks corresponding to mt's tasks and add them to the      */
/*  drs mtask and to drs->all_root_tasks.                                    */
/*                                                                           */
/*  Return true if all this is done successfully.  The only reason why       */
/*  false is returned is that creating one of the tasks uncovered a case     */
/*  where a resource is assigned (or preassigned) to two tasks on one day.   */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsMTaskMake(KHE_MTASK mt, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_MTASK res;  int i;  KHE_TASK task;  KHE_DRS_TASK dt;
  KHE_DRS_DAY day;  KHE_COST non_asst_cost, asst_cost;

  /* get or make a drs mtask object */
  HaMake(res, drs->arena);
  res->orig_mtask = mt;
  res->encl_shift = NULL;  /* reset later */
  res->day_range = KheMTaskInterval(mt);
  /* ***
  res->day_range = KheIntervalMake(KheMTaskFirstDayIndex(mt),
    KheMTaskLastDayIndex(mt));
  *** */
  HaArrayInit(res->all_tasks, drs->arena);
  HaArrayInit(res->available_tasks, drs->arena);
  res->expand_must_assign_count = 0;
  res->expand_prev_unfixed = -1;

  /* make drs tasks for its khe tasks and add them */
  for( i = 0;  i < KheMTaskTaskCount(mt);  i++ )
  {
    task = KheMTaskTask(mt, i, &non_asst_cost, &asst_cost);
    if( !KheDrsTaskMake(res, i, task, non_asst_cost, asst_cost, drs, &dt) )
      return false;
    HaArrayAddLast(res->all_tasks, dt);
    if( dt->closed_dr == NULL && !KheTaskIsFixedAndUnassigned(task) )
      HaArrayAddLast(res->available_tasks, dt);
  }

  /* make res one of the mtasks of its first day */
  day = HaArray(drs->all_days, res->day_range.first);
  KheDrsDayAddMTask(day, res, drs);
  if( DEBUG50 && res->day_range.first < res->day_range.last )
    KheDrsMTaskDebug(res, drs, 2, 2, stderr);
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsMTaskCheckAvailableTasks(KHE_DRS_MTASK dmt)                   */
/*                                                                           */
/*  Return true if the available tasks of dmt all have no closed assts.      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsMTaskCheckAvailableTasks(KHE_DRS_MTASK dmt)
{
  KHE_DRS_TASK dt;  int i;
  HaArrayForEach(dmt->available_tasks, dt, i)
    if( dt->closed_dr != NULL )
      return false;
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMTaskOrganizeAvailableTasks(KHE_DRS_MTASK dmt)                */
/*                                                                           */
/*  Organize the unassigned tasks of dmt for solving.                        */
/*                                                                           */
/*  When this function is called, available_tasks may be out of order,       */
/*  and, because available_tasks is handled lazily, some of them may in      */
/*  fact be assigned.  This function fixes both of those problems.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMTaskOrganizeAvailableTasks(KHE_DRS_MTASK dmt)
{
  KHE_DRS_TASK dt;

  /* sort available_tasks into order with assigned tasks at the end */
  HaArraySortUnique(dmt->available_tasks, &KheDrsTaskIndexInEnclCmp);

  /* remove assigned tasks from the end */
  while( HaArrayCount(dmt->available_tasks) > 0 )
  {
    dt = HaArrayLast(dmt->available_tasks);
    if( dt->closed_dr == NULL )
      break;
    HaArrayDeleteLast(dmt->available_tasks);
  }

  /* check that none of the unassigned tasks have a closed asst */
  if( DEBUG71 )
    HnAssert(KheDrsMTaskCheckAvailableTasks(dmt),
      "KheDrsMTaskOrganizeAvailableTasks internal error");
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsMTaskOpen(KHE_DRS_MTASK dmt, KHE_INTERVAL ddr,                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  If dmt is suitable (if its day range lies within ddr, and its resource   */
/*  domain includes at least one open resource), then open it.  Opening dmt  */
/*  basically means organizing its unassigned tasks into decreasing          */
/*  desirability order and opening them.                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsMTaskOpen(KHE_DRS_MTASK dmt, KHE_INTERVAL ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK dt;  int i;  bool res;
  if( DEBUG19 || DEBUG70(ddr, dmt->encl_shift->encl_day->frame_index) )
  {
    fprintf(stderr, "[ KheDrsMTaskOpen(dmt, %d-%d)\n", ddr.first, ddr.last);
    KheDrsMTaskDebug(dmt, drs, 2, 2, stderr);
  }
  if( KheIntervalSubset(dmt->day_range, ddr) &&
      !KheResourceSetDisjointGroup(drs->selected_resource_set,
	KheMTaskDomain(dmt->orig_mtask)) )
  {
    /* dmt can open; organize and open available_tasks; say none are used */
    KheDrsMTaskOrganizeAvailableTasks(dmt);
    HaArrayForEach(dmt->available_tasks, dt, i)
      KheDrsTaskOpen(dt, drs);
    dmt->expand_must_assign_count = 0;
    dmt->expand_prev_unfixed = -1;
    res = true;
  }
  else
  {
    /* dmt can't open; set dmt->expand_prev_unfixed to make that clear */
    dmt->expand_prev_unfixed = HaArrayCount(dmt->available_tasks) - 1;
    res = false;
  }
  if( DEBUG19 || DEBUG70(ddr, dmt->encl_shift->encl_day->frame_index) )
    fprintf(stderr, "] KheDrsMTaskOpen returning %s\n", bool_show(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMTaskClose(KHE_DRS_MTASK dmt)                                 */
/*                                                                           */
/*  Close dmt.  Some of its tasks may be already closed; that's OK.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMTaskClose(KHE_DRS_MTASK dmt)
{
  KHE_DRS_TASK dt;  int i;
  HaArrayForEach(dmt->available_tasks, dt, i)
    KheDrsTaskClose(dt, NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMTaskAddMaxWorkloadPerTime(KHE_DRS_MTASK dmt,                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add max workload per time values reflecting the presence of dmt.         */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsMTaskAddMaxWorkloadPerTime(KHE_DRS_MTASK dmt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK dt;  KHE_DRS_TASK_ON_DAY dtd;  int i, ti;
  float workload_per_time;
  HnAssert(HaArrayCount(dmt->available_tasks) > 0,
    "KheDrsMTaskAddMaxWorkloadPerTime internal error");
  dt = HaArray(dmt->available_tasks, 0);
  HaArrayForEach(dt->days, dtd, i)
    if( dtd->task != NULL && dtd->time != NULL )  ** always true? **
    {
      workload_per_time = KheTaskWorkloadPerTime(dtd->task);
      ti = KheTimeIndex(dtd->time);
      if( workload_per_time > HaArray(drs->max_workload_per_time, ti) )
	HaArrayPut(drs->max_workload_per_time, ti, workload_per_time);
    }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMTaskDebug(KHE_DRS_MTASK dmt, int verbosity,                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of dmt onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMTaskDebug(KHE_DRS_MTASK dmt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_TASK dt;  int i, pos;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ MTask ", indent, "");
    if( dmt == NULL )
      fprintf(fp, "(free) ]\n");
    else
    {
      if( drs != NULL )
	KheDrsDayRangeDebug(dmt->day_range, drs, 1, 0, fp);
      HaArrayForEach(dmt->all_tasks, dt, i)
      {
	fprintf(fp, "%*s  ", indent, "");
	KheDrsTaskDebug(dt, verbosity, -1, fp);
	fprintf(fp, " %s%s\n", HaArrayContains(dmt->available_tasks, dt,&pos) ?
	  "" : "assigned ", KheDrsTaskExpandRoleShow(dt->expand_role));
      }
      fprintf(fp, "%*s]\n", indent, "");
    }
  }
  else
  {
    if( dmt == NULL )
      fprintf(fp, "{free}");
    else
    {
      fprintf(fp, "{");
      if( HaArrayCount(dmt->all_tasks) >= 2 )
      {
	KheDrsTaskDebug(HaArrayFirst(dmt->all_tasks), 1, -1, fp);
	fprintf(fp, "...(%d)", HaArrayCount(dmt->all_tasks));
      }
      else
	KheDrsTaskDebug(HaArrayFirst(dmt->all_tasks), 1, -1, fp);
      if( dmt->expand_must_assign_count > 0 )
	fprintf(fp, " must %d}", dmt->expand_must_assign_count);
      else
	fprintf(fp, "}");
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_MTASK - expansion"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMTaskExpandBegin(KHE_DRS_MTASK dmt,                           */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_EXPANDER de)                           */
/*                                                                           */
/*  Begin an expansion involving mtask dmt.                                  */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderAddMustAssign(KHE_DRS_EXPANDER de);

static void KheDrsMTaskExpandBegin(KHE_DRS_MTASK dmt,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day, KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK dt;  int i;
  dmt->expand_must_assign_count = 0;
  HaArrayForEach(dmt->available_tasks, dt, i)
  {
    HnAssert(dt->closed_dr == NULL,
      "KheDrsMTaskExpandBegin internal error 1");
    if( dt->expand_role != KHE_DRS_TASK_EXPAND_FIXED )
    {
      if( KheDrsExpanderOpenToExtraCost(de, dt->non_asst_cost) )
	dt->expand_role = KHE_DRS_TASK_EXPAND_FREE;
      else
      {
	/* dt must be assigned, otherwise cost will be too high */
	dt->expand_role = KHE_DRS_TASK_EXPAND_MUST;
	dmt->expand_must_assign_count++;
	dmt->encl_shift->expand_must_assign_count++;
	KheDrsExpanderAddMustAssign(de);
	if( DEBUG51(prev_day) )
	{
	  fprintf(stderr, "  KheDrsMTaskExpandBegin: must_assign task ");
	  KheDrsTaskDebug(dt, 2, 0, stderr);
	}
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMTaskExpandEnd(KHE_DRS_MTASK dmt, KHE_DRS_EXPANDER de)        */
/*                                                                           */
/*  End an expansion involving mtask dmt.                                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMTaskExpandEnd(KHE_DRS_MTASK dmt, KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK dt;  int i;
  HaArrayForEach(dmt->available_tasks, dt, i)
    dt->expand_role = KHE_DRS_TASK_EXPAND_NO_VALUE;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsMTaskSuitsFreeResource(KHE_DRS_MTASK dmt, KHE_DRS_RESOURCE dr)*/
/*                                                                           */
/*  Return true if dmt could be assigned dr, either because dmt represents   */
/*  a free day, or because dmt represents a set of tasks whose common        */
/*  domain includes dr and whose common duration includes only times         */
/*  for which dr is open.                                                    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsMTaskSuitsFreeResource(KHE_DRS_MTASK dmt, KHE_DRS_RESOURCE dr)
{
  int di;  KHE_DRS_RESOURCE_ON_DAY other_drd;

  /* a NULL dmt means a free day, which suits every unfixed resource */
  if( dmt == NULL )
    return true;

  /* domain has to suit */
  if( !KheResourceGroupContains(KheMTaskDomain(dmt->orig_mtask), dr->resource) )
    return false;

  /* dr has to be available for every day of dmt */
  for( di = dmt->day_range.first;  di <= dmt->day_range.last;  di++ )
  {
    other_drd = HaArray(dr->days, di);
    if( other_drd->closed_dtd != NULL )
      return false;
  }

  /* all good */
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsMTaskAcceptResourceBegin(KHE_DRS_MTASK dmt,                   */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY *dtd)                 */
/*                                                                           */
/*  Assuming KheDrsMTaskSuitsFreeResource(dmt, drd) is true, if dmt has      */
/*  an unassigned tasks, then return true and set *dtd to the relevant       */
/*  task on day of a task that drd can be assigned to.  Else return false.   */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsMTaskAcceptResourceBegin(KHE_DRS_MTASK dmt,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY *dtd)
{
  KHE_DRS_TASK dt;  int i, count;

  count = HaArrayCount(dmt->available_tasks);
  for( i = dmt->expand_prev_unfixed + 1;  i < count;  i++ )
  {
    dt = HaArray(dmt->available_tasks, i);
    if( dt->expand_role != KHE_DRS_TASK_EXPAND_FIXED )
    {
      if( !KheDrsTaskRunningOnDay(dt, drd->day, dtd) )
	HnAbort("KheDrsMTaskAcceptResourceBegin internal error");
      dmt->expand_prev_unfixed = i;
      return true;
    }
  }

  /* if we get here we've failed to identify a suitable task */
  return *dtd = NULL, false;
}

/* *** old version that incorporates the new KheDrsMTaskSuitsFreeResource
static bool KheDrsMTaskAcceptResourceBegin(KHE_DRS_MTASK dmt,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY *dtd)
{
  KHE_DRS_RESOURCE dr;  KHE_DRS_TASK dt;  int i, count, di;
  KHE_DRS_RESOURCE_ON_DAY other_drd;
  dr = drd->encl_dr;
  if( KheResourceGroupContains(KheMTaskDomain(dmt->orig_mtask), dr->resource) )
  {
    ** make sure dr is available for every day of dmt **
    for( di = dmt->day_range.first;  di <= dmt->day_range.last;  di++ )
    {
      other_drd = HaArray(dr->days, di);
      if( other_drd->closed_dtd != NULL )
	return *dtd = NULL, false;
    }

    ** search forwards for next unfixed task **
    count = HaArrayCount(dmt->available_tasks);
    for( i = dmt->expand_prev_unfixed + 1;  i < count;  i++ )
    {
      dt = HaArray(dmt->available_tasks, i);
      if( dt->expand_role != KHE_DRS_TASK_EXPAND_FIXED )
      {
	if( !KheDrsTaskRunningOnDay(dt, drd->day, dtd) )
	  HnAbort("KheDrsMTaskAcceptResourceBegin internal error");
        dmt->expand_prev_unfixed = i;
	return true;
      }
    }
  }

  ** if we get here we've failed to identify a suitable task **
  return *dtd = NULL, false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMTaskAcceptResourceEnd(KHE_DRS_MTASK dmt,                     */
/*    KHE_DRS_TASK_ON_DAY dtd)                                               */
/*                                                                           */
/*  Undo the effect of a corresponding KheDrsMTaskAcceptResourceBegin        */
/*  which returned true.                                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMTaskAcceptResourceEnd(KHE_DRS_MTASK dmt,
  KHE_DRS_TASK_ON_DAY dtd)
{
  KHE_DRS_TASK dt;  int i;
  HnAssert(dmt->expand_prev_unfixed < HaArrayCount(dmt->available_tasks) &&
    dtd->encl_dt == HaArray(dmt->available_tasks, dmt->expand_prev_unfixed),
    "KheDrsMTaskAcceptResourceEnd internal error");

  /* search backwards for next unfixed task, or start of array */
  for( i = dmt->expand_prev_unfixed - 1;  i >= 0;  i-- )
  {
    dt = HaArray(dmt->available_tasks, dmt->expand_prev_unfixed);
    if( dt->expand_role != KHE_DRS_TASK_EXPAND_FIXED )
      break;
  }
  dmt->expand_prev_unfixed = i;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_MTASK - one-extra and two-extra selection"            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsMTaskOneExtraAvailable(KHE_DRS_MTASK dmt,                     */
/*    int open_resource_count)                                               */
/*                                                                           */
/*  Return true if dmt is available for participating in a one-extra         */
/*  dominance test.  Parameter open_resource_count is the number of open     */
/*  resources.                                                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsMTaskOneExtraAvailable(KHE_DRS_MTASK dmt,
  int open_resource_count)
{
  return dmt == NULL ||
    open_resource_count - 1 < HaArrayCount(dmt->available_tasks);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsMTaskMinCost(KHE_DRS_MTASK dmt, KHE_DRS_ASST_OP op,           */
/*    KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb, int open_resource_count,   */
/*    KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)             */
/*                                                                           */
/*  This function evaluates three expressions from the documentation:        */
/*                                                                           */
/*    op                           function                                  */
/*    ---------------------------------------------------------------        */
/*    KHE_DRS_ASST_OP_UNASSIGN     min(p) c-(M, ra)                          */
/*    KHE_DRS_ASST_OP_ASSIGN       min(q) c+(M, ra)                          */
/*    KHE_DRS_ASST_OP_REPLACE      min(p) c-+(M, ra, rb)                     */
/*    ---------------------------------------------------------------        */
/*                                                                           */
/*  But instead of returning the value it adds it to *avail_cost.  The       */
/*  return value is true if the function is able to calculate this cost,     */
/*  and false if not.                                                        */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsMTaskMinCost(KHE_DRS_MTASK dmt, KHE_DRS_ASST_OP op,
  KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb, int open_resource_count,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)
{
  int p, max_p;  KHE_DRS_TASK dt;  KHE_COST cost, min_cost;

  if( fp != NULL )
  {
    fprintf(stderr, "[ KheDrsMTaskMinCost(");
    if( dmt == NULL )
      fprintf(stderr, "(free)");
    else
      KheDrsMTaskDebug(dmt, NULL, 1, -1, stderr);
    fprintf(stderr, ", op %d, dra %s, drb %s, count %d, &avail_cost)\n",
      (int) op, dra == NULL ? "-" : KheDrsResourceId(dra),
      drb == NULL ? "-" : KheDrsResourceId(drb), open_resource_count);
  }

  /* if dmt is NULL (represents a free day) there is no change to *avail_cost */
  if( dmt == NULL )
  {
    if( fp != NULL )
      fprintf(stderr, "] KheDrsMTaskMinCost returning true (free)\n");
    return true;
  }

  /* find the minimum, over unassigned tasks 0 ... max_p, of the cost */
  max_p = min(open_resource_count - 1, HaArrayCount(dmt->available_tasks) - 1);
  min_cost = KheCost(INT_MAX, INT_MAX);
  for( p = 0;  p <= max_p;  p++ )
  {
    if( fp != NULL )
      fprintf(stderr, "  p = %d\n", p);
    dt = HaArray(dmt->available_tasks, p);   /* the (p+1)st unassigned task */
    if( !KheDrsTaskMinCost(dt, op, dra, drb, &cost) )
    {
      if( fp != NULL )
	fprintf(stderr, "] KheDrsMTaskMinCost returning false\n");
      return false;
    }
    if( cost < min_cost )
      min_cost = cost;
  }

  /* wrapup */
  HnAssert(min_cost < KheCost(INT_MAX, INT_MAX),
    "KheDrsMTaskMinCost internal error");
  if( fp != NULL )
    fprintf(stderr, "] KheDrsMTaskMinCost returning true (min_cost %.5f)\n",
      KheCostShow(min_cost));
  *avail_cost += min_cost;
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT"                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT KheDrsShiftMake(KHE_DRS_DAY day, KHE_DRS_MTASK dmt,        */
/*    int index_in_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                     */
/*                                                                           */
/*  Make a new shift containing just dmt.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SHIFT KheDrsShiftMake(KHE_DRS_DAY day, KHE_DRS_MTASK dmt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT res;
  HaMake(res, drs->arena);
  res->encl_day = day;
  res->open_shift_index = -1;
  res->expand_must_assign_count = 0;
  res->expand_max_included_free_resource_count = 0;
  HaArrayInit(res->mtasks, drs->arena);
  HaArrayInit(res->open_mtasks, drs->arena);
  HaArrayInit(res->shift_pairs, drs->arena);
  res->signer = KheDrsSignerMake(NULL, NULL, res, NULL, drs);
  HaArrayAddLast(res->mtasks, dmt);
  res->soln_trie = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsShiftId(KHE_DRS_SHIFT ds)                                    */
/*                                                                           */
/*  Return an Id for shift.                                                  */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsShiftId(KHE_DRS_SHIFT ds)
{
  static char buff[195];
  if( ds->open_shift_index >= 0 )
    snprintf(buff, 195, "%s:%d", KheDrsDayId(ds->encl_day),
      ds->open_shift_index);
  else
    snprintf(buff, 195, "%s:?", KheDrsDayId(ds->encl_day));
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsShiftAcceptsMTask(KHE_DRS_SHIFT ds, KHE_DRS_MTASK dmt)        */
/*                                                                           */
/*  If ds can accept dmt, because ds's mtasks have the same times and        */
/*  the same total workload as dmt, then add dmt to ds and return true.      */
/*  Otherwise change nothing and return false.                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsShiftAcceptsMTask(KHE_DRS_SHIFT ds, KHE_DRS_MTASK dmt)
{
  KHE_DRS_MTASK dmt2;  int index;  KHE_INTERVAL in1, in2;
  KHE_MTASK mt1, mt2;  KHE_TIME time1, time2;  float workload1, workload2;

  /* get the two mtasks, mt1 and mt2, that have to be compared */
  dmt2 = HaArrayFirst(ds->mtasks);
  mt1 = dmt->orig_mtask;
  mt2 = dmt2->orig_mtask;

  /* intervals must be the same */
  in1 = KheMTaskInterval(mt1);
  in2 = KheMTaskInterval(mt2);
  if( !KheIntervalEqual(in1, in2) )
    return false;
  /* ***
  first_day_index = KheMTaskFirstDayIndex(mt1);
  if( KheMTaskFirstDayIndex(mt2) != first_day_index )
    return false;
  last_day_index = KheMTaskLastDayIndex(mt1);
  if( KheMTaskLastDayIndex(mt2) != last_day_index )
    return false;
  *** */

  /* busy times and workloads on each day must be the same */
  for( index = KheIntervalFirst(in1);  index <= KheIntervalLast(in1);  index++ )
  {
    time1 = KheMTaskDayTime(mt1, index, &workload1);
    time2 = KheMTaskDayTime(mt2, index, &workload2);
    if( time1 != time2 || workload1 != workload2 )
      return false;
  }

  /* all good, add dmt to ds and return true */
  HaArrayAddLast(ds->mtasks, dmt);
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftOpen(KHE_DRS_SHIFT ds, KHE_INTERVAL ddr,                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Open ds and its mtasks.                                                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftOpen(KHE_DRS_SHIFT ds, KHE_INTERVAL ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_MTASK dmt;  int i;
  ds->open_shift_index = HaArrayCount(drs->open_shifts);
  HaArrayAddLast(drs->open_shifts, ds);
  HnAssert(HaArrayCount(ds->open_mtasks) == 0,
    "KheDrsShiftOpen internal error");
  HaArrayForEach(ds->mtasks, dmt, i)
    if( KheDrsMTaskOpen(dmt, ddr, drs) )
    {
      HaArrayAddLast(ds->open_mtasks, dmt);
      /* KheDrsMTaskAddMaxWorkloadPerTime(dmt, drs); */
    }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftClose(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs) */
/*                                                                           */
/*  Close ds and its mtasks.                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftClose(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_MTASK dmt;  int i;
  HaArrayForEach(ds->open_mtasks, dmt, i)
    KheDrsMTaskClose(dmt);
  HaArrayClear(ds->open_mtasks);
  ds->open_shift_index = -1;
  KheDrsSignerClear(ds->signer, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAddOpenExpr(KHE_DRS_SHIFT ds, KHE_DRS_EXPR e)            */
/*                                                                           */
/*  Add open expression e to ds.                                             */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsShiftAddOpenExpr(KHE_DRS_SHIFT ds, KHE_DRS_EXPR e)
{
  KheDrsSignerAddOpenExpr(ds->signer, e);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAddDomTest(KHE_DRS_SHIFT ds, KHE_DRS_DOM_TEST dom_test)  */
/*                                                                           */
/*  Add dom_test to ds.                                                      */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsShiftAddDomTest(KHE_DRS_SHIFT ds, KHE_DRS_DOM_TEST dom_test,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSignerAddDomTest(ds->signer, dom_test, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNER KheDrsShiftSigner(KHE_DRS_SHIFT ds)                       */
/*                                                                           */
/*  Return ds's signer.                                                      */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNER KheDrsShiftSigner(KHE_DRS_SHIFT ds)
{
  return ds->signer;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT - expansion"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftExpandBegin(KHE_DRS_SHIFT ds, KHE_DRS_SOLN prev_soln,    */
/*    KHE_DRS_DAY prev_day, KHE_DRS_EXPANDER de)                             */
/*                                                                           */
/*  Initialize ds and its mtasks and tasks for expansion of                  */
/*  prev_soln.  Only don't initialize the assignment trie yet.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftExpandBegin(KHE_DRS_SHIFT ds, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY prev_day, KHE_DRS_EXPANDER de)
{
  KHE_DRS_MTASK dmt;  int i;
  ds->expand_must_assign_count = 0;
  ds->expand_max_included_free_resource_count = 0;
  HaArrayForEach(ds->open_mtasks, dmt, i)
    KheDrsMTaskExpandBegin(dmt, prev_soln, prev_day, de);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftExpandEnd(KHE_DRS_SHIFT ds, KHE_DRS_EXPANDER de)         */
/*                                                                           */
/*  An expansion involving this shift is ending; clear it out.               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftSolnTrieFree(KHE_DRS_SHIFT_SOLN_TRIE dsat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsShiftExpandEnd(KHE_DRS_SHIFT ds, KHE_DRS_EXPANDER de)
{
  KHE_DRS_MTASK dmt;  int i;
  HaArrayForEach(ds->open_mtasks, dmt, i)
    KheDrsMTaskExpandEnd(dmt, de);
  KheDrsShiftSolnTrieFree(ds->soln_trie, de->solver);
  ds->soln_trie = NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftBuildShiftSolnTrie(KHE_DRS_SHIFT ds,                     */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,                          */
/*    KHE_DRS_RESOURCE_SET free_resources, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Initialize the shift solution trie of ds.                                */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SHIFT_SOLN_TRIE KheDrsShiftSolnTrieBuild(KHE_DRS_SHIFT ds,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day,
  KHE_DRS_RESOURCE_SET all_free_resources, int all_free_resources_index,
  KHE_DRS_RESOURCE_SET included_free_resources,
  KHE_DRS_TASK_SOLN_SET all_fixed_assts, KHE_DRS_EXPANDER de);

static void KheDrsShiftBuildShiftSolnTrie(KHE_DRS_SHIFT ds,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day,
  KHE_DRS_RESOURCE_SET all_free_resources,
  KHE_DRS_TASK_SOLN_SET all_fixed_assts,
  KHE_DRS_EXPANDER de, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_SET included_free_resources;
  HnAssert(ds->soln_trie == NULL,
    "KheDrsShiftBuildShiftSolnTrie internal error");
  included_free_resources = KheDrsResourceSetMake(drs);
  ds->soln_trie = KheDrsShiftSolnTrieBuild(ds, prev_soln, prev_day,
    next_day, all_free_resources, 0, included_free_resources,
    all_fixed_assts, de);
  KheDrsResourceSetFree(included_free_resources, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftExpandByShifts(KHE_DRS_SHIFT ds, int shift_index,        */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,     */
/*    KHE_DRS_RESOURCE_SET free_resources)                                   */
/*                                                                           */
/*  Expand shift ds and beyond using free_resources.                         */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftSolnTrieExpandByShifts(KHE_DRS_SHIFT_SOLN_TRIE dsat,
  KHE_DRS_SHIFT ds, int shift_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources, int avail_index,
  int selected_resource_count, KHE_DRS_RESOURCE_SET omitted_resources);

static void KheDrsShiftExpandByShifts(KHE_DRS_SHIFT ds, int shift_index,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources)
{
  KHE_DRS_RESOURCE_SET omitted_resources;
  if( ds->soln_trie != NULL )
  {
    omitted_resources = KheDrsResourceSetMake(de->solver);
    KheDrsShiftSolnTrieExpandByShifts(ds->soln_trie, ds, shift_index,
      prev_soln, next_day, de, free_resources, 0, 0, omitted_resources);
    KheDrsResourceSetFree(omitted_resources, de->solver);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftDebug(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs, */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of shift ds onto fp with the given verbosity and indent.     */
/*  Here drs may be NULL, although the result will be more readable if it    */
/*  is non-NULL.                                                             */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsShiftSolnTrieResourceIsForced(KHE_DRS_SHIFT_SOLN_TRIE dsat,
  KHE_DRS_RESOURCE dr);
static void KheDrsShiftSolnTrieDebug(KHE_DRS_SHIFT_SOLN_TRIE dsat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp);

static void KheDrsShiftDebug(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_MTASK dmt;  KHE_DRS_RESOURCE dr;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ Shift %d (expand_min %d, expand_max %d)\n", indent, "",
      ds->open_shift_index, ds->expand_must_assign_count,
      ds->expand_max_included_free_resource_count);
    HaArrayForEach(ds->mtasks, dmt, i)
      KheDrsMTaskDebug(dmt, drs, verbosity, indent + 2, fp);
    if( verbosity >= 2 && ds->soln_trie != NULL )
    {
      KheDrsShiftSolnTrieDebug(ds->soln_trie, drs, verbosity-1, indent+2, fp);
      KheDrsResourceSetForEach(drs->open_resources, dr, i)
	if( KheDrsShiftSolnTrieResourceIsForced(ds->soln_trie, dr) )
	  fprintf(fp, "%*s  forced resource %s\n", indent, "",
	    KheDrsResourceId(dr));
    }
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT_PAIR"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT_PAIR KheDrsShiftPairMake(KHE_DRS_SHIFT ds1,                */
/*    KHE_DRS_SHIFT ds2, KHE_DYNAMIC_RESOURCE_SOLVER drs)                    */
/*                                                                           */
/*  Make a shift pair object with these attributes.                          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SHIFT_PAIR KheDrsShiftPairMake(KHE_DRS_SHIFT dsa,
  KHE_DRS_SHIFT dsb, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT_PAIR res;
  HaMake(res, drs->arena);
  res->shift[0] = dsa;
  res->shift[1] = dsb;
  res->signer = KheDrsSignerMake(NULL, NULL, NULL, res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsShiftPairId(KHE_DRS_SHIFT_PAIR dsp)                          */
/*                                                                           */
/*  Return an Id for dsp.  This one can't be used for very long.             */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsShiftPairId(KHE_DRS_SHIFT_PAIR dsp)
{
  static char buff1[195], buff2[195], buff[400];
  snprintf(buff1, 195, "%s", KheDrsShiftId(dsp->shift[0]));
  snprintf(buff2, 195, "%s", KheDrsShiftId(dsp->shift[1]));
  snprintf(buff, 400, "{%s, %s}", buff1, buff2);
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftPairAddOpenExpr(KHE_DRS_SHIFT_PAIR dsp, KHE_DRS_EXPR e)  */
/*                                                                           */
/*  Add e to dsp's signer.                                                   */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsShiftPairAddOpenExpr(KHE_DRS_SHIFT_PAIR dsp, KHE_DRS_EXPR e)
{
  KheDrsSignerAddOpenExpr(dsp->signer, e);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftPairAddDomTest(KHE_DRS_SHIFT_PAIR dsp,                   */
/*    KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs)            */
/*                                                                           */
/*  Add dom_test to dsp's signer.                                            */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsShiftPairAddDomTest(KHE_DRS_SHIFT_PAIR dsp,
  KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSignerAddDomTest(dsp->signer, dom_test, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNER KheDrsShiftPairSigner(KHE_DRS_SHIFT_PAIR dsp)             */
/*                                                                           */
/*  Return dsp's signer.                                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNER KheDrsShiftPairSigner(KHE_DRS_SHIFT_PAIR dsp)
{
  return dsp->signer;
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "signatures"                                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_VALUE"                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_VALUE KheDrsValueInt(int val)                                    */
/*                                                                           */
/*  Return a KHE_DRS_VALUE with this value.                                  */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static KHE_DRS_VALUE KheDrsValueInt(int val)
{
  KHE_DRS_VALUE res;
  res.i = val;
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_VALUE KheDrsValueFloat(float val)                                */
/*                                                                           */
/*  Return a KHE_DRS_VALUE with this value.                                  */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static KHE_DRS_VALUE KheDrsValueFloat(float val)
{
  KHE_DRS_VALUE res;
  res.f = val;
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SIGNATURE"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNATURE KheDrsSignatureMake(KHE_DYNAMIC_RESOURCE_SOLVER drs,   */
/*    KHE_COST cost)                                                         */
/*                                                                           */
/*  Return a new signature object with the given cost and empty states.      */
/*  This is assumed to include a reference to the result.                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNATURE KheDrsSignatureMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNATURE res;
  if( HaArrayCount(drs->signature_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->signature_free_list);
    HaArrayClear(res->states);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->states, drs->arena);
  }
  res->reference_count = 0;       /* this counts heap references; none yet */
  res->asst_to_shift_index = -1;  /* means none; may be reset later */
  res->cost = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureRefer(KHE_DRS_SIGNATURE sig)                         */
/*                                                                           */
/*  Record the fact that there is now an additional heap reference to sig.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureRefer(KHE_DRS_SIGNATURE sig)
{
  sig->reference_count++;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureUnRefer(KHE_DRS_SIGNATURE sig,                       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Record the fact that there is now one less heap reference to sig.  If    */
/*  the reference count drops to 0, free sig.                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureUnRefer(KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(sig->reference_count > 0, "KheDrsSignatureUnRefer internal error");
  sig->reference_count--;
  if( sig->reference_count == 0 )
    HaArrayAddLast(drs->signature_free_list, sig);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignatureOptionallyFree(KHE_DRS_SIGNATURE sig,                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  If sig is currently referred to by any heap object, free it and return   */
/*  true.  Otherwise return false.                                           */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignatureOptionallyFree(KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( sig->reference_count == 0 )
  {
    HaArrayAddLast(drs->signature_free_list, sig);
    return true;
  }
  else
    return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureAddAsstToShiftIndex(KHE_DRS_SIGNATURE sig, int val)  */
/*                                                                           */
/*  Add an assignment to shift index with value val to sig.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureAddAsstToShiftIndex(KHE_DRS_SIGNATURE sig, int val)
{
  HnAssert(sig->asst_to_shift_index == -1,
    "KheDrsSignatureAddAsstToShiftIndex internal error 2");
  sig->asst_to_shift_index = val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureClearAsstToShiftIndex(KHE_DRS_SIGNATURE sig)         */
/*                                                                           */
/*  Clear out the assignment to shift index of sig.                          */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static void KheDrsSignatureClearAsstToShiftIndex(KHE_DRS_SIGNATURE sig)
{
  HnAssert(sig->asst_to_shift_index >= 0,
    "KheDrsSignatureClearAsstToShiftIndex internal error 2");
  sig->asst_to_shift_index = -1;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureAddCost(KHE_DRS_SIGNATURE sig, KHE_COST cost)        */
/*                                                                           */
/*  Add cost to sig.                                                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureAddCost(KHE_DRS_SIGNATURE sig, KHE_COST cost)
{
  sig->cost += cost;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureAddState(KHE_DRS_SIGNATURE sig, int state,           */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e)                                    */
/*                                                                           */
/*  Add state to the end of sig.  Here dsg and e are just for checking.      */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSignerDebug(KHE_DRS_SIGNER dsg,
  int verbosity, int indent, FILE *fp);

static void KheDrsSignatureAddState(KHE_DRS_SIGNATURE sig, KHE_DRS_VALUE state,
  KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e)
{
  KHE_DRS_DOM_TEST dt;
  dt = HaArray(dsg->dom_tests, HaArrayCount(sig->states));
  if( dt->monitor != NULL && DEBUG44_MONITOR(dt->monitor) )
  {
    fprintf(stderr, "[ KheDrsSignatureAddState(%p, pos %d, state %d), e =\n",
      (void *) sig, HaArrayCount(sig->states), state.i);
    KheDrsExprDebug(e, NULL, 2, 2, stderr);
    fprintf(stderr, "]\n");
  }
  HaArrayAddLast(sig->states, state);
  HnAssert(HaArrayCount(sig->states) <= HaArrayCount(dsg->dom_tests),
    "KheDrsSignatureAddState internal error 1:  %d states > %d dom_tests",
    HaArrayCount(sig->states), HaArrayCount(dsg->dom_tests));
  if( DEBUG63 && dt->expr != e )
  {
    fprintf(stderr, "[ KheDrsSignatureAddState(%p, init states %d) failing:\n",
      (void *) sig, HaArrayCount(sig->states) - 1);
    fprintf(stderr, "  signer:\n");
    KheDrsSignerDebug(dsg, 2, 4, stderr);
    fprintf(stderr, "  expression argument:\n");
    if( e == NULL )
      fprintf(stderr, "    NULL\n");
    else
      KheDrsExprDebug(e, dsg->solver, 2, 4, stderr);
    fprintf(stderr, "  dom_test expression:\n");
    if( dt->expr == NULL )
      fprintf(stderr, "    NULL\n");
    else
      KheDrsExprDebug(dt->expr, dsg->solver, 2, 4, stderr);
    fprintf(stderr, "]\n");
  }
  /* ***
  if( DEBUG63 && state.i == 4 )
  {
    fprintf(stderr, "[ KheDrsSignatureAddState(%p) reporting state.i == 4\n",
      (void *) sig);
    KheDrsExprDebug(e, NULL, 2, 2, stderr);
    fprintf(stderr, "]\n");
  }
  *** */
  HnAssert(dt->expr == e, "KheDrsSignatureAddState internal error 2");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureDebug(KHE_DRS_SIGNATURE sig, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of sig onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureDebug(KHE_DRS_SIGNATURE sig, int verbosity,
  int indent, FILE *fp)
{
  int i;  KHE_DRS_VALUE val;
  HnAssert(sig->reference_count > 0, "KheDrsSignatureDebug internal error");
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  if( HaArrayCount(sig->states) > 0 )
  {
    fprintf(fp, "<%.5f:", KheCostShow(sig->cost));
    HaArrayForEach(sig->states, val, i)
      fprintf(fp, " %d", val.i);
    fprintf(fp, ">");
  }
  else
    fprintf(fp, "<%.5f>", KheCostShow(sig->cost));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SIGNATURE_SET"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureSetInit(KHE_DRS_SIGNATURE_SET sig_set,               */
/*    KHE_COST cost, HA_ARENA a)                                             */
/*                                                                           */
/*  Initialize sig_set to have the given cost and no signatures.             */
/*                                                                           */
/*  There is no KheDrsSignatureSetMake, because signature sets are used      */
/*  only in solution objects, and those signature sets are expanded, hence   */
/*  "Init" (which assumes that memory is already allocated), not "Make".     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureSetInit(KHE_DRS_SIGNATURE_SET sig_set,
  KHE_COST cost, HA_ARENA a)
{
  sig_set->cost = cost;
  HaArrayInit(sig_set->signatures, a);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureSetClear(KHE_DRS_SIGNATURE_SET sig_set,              */
/*    KHE_COST cost, KHE_DYNAMIC_RESOURCE_SOLVER drs)                        */
/*                                                                           */
/*  Clear sig_set back to the given cost and no signatures.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureSetClear(KHE_DRS_SIGNATURE_SET sig_set,
  KHE_COST cost, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNATURE sig;
  sig_set->cost = cost;
  while( HaArrayCount(sig_set->signatures) > 0 )
  {
    sig = HaArrayLastAndDelete(sig_set->signatures);
    KheDrsSignatureUnRefer(sig, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSignatureSetFullHash(KHE_DRS_SIGNATURE_SET sig_set)            */
/*                                                                           */
/*  Hash function for hashing all of sig_set.                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSignatureSetFullHash(KHE_DRS_SIGNATURE_SET sig_set)
{
  int res, i, j;  KHE_DRS_SIGNATURE sig;  KHE_DRS_VALUE val;
  res = 0;
  HaArrayForEach(sig_set->signatures, sig, i)
    HaArrayForEach(sig->states, val, j)
      res = (res + val.i) * 3;
  return (res < 0 ? - res : res);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignatureFullEqual(KHE_DRS_SIGNATURE sig1,                    */
/*    KHE_DRS_SIGNATURE sig2)                                                */
/*                                                                           */
/*  Return true if sig1 and sig2 are all equal.                              */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignatureSetFullEqual(KHE_DRS_SIGNATURE_SET sig_set1,
  KHE_DRS_SIGNATURE_SET sig_set2)
{
  int i, j;  KHE_DRS_SIGNATURE sig1, sig2;
  HnAssert(HaArrayCount(sig_set1->signatures) ==
    HaArrayCount(sig_set2->signatures),
    "KheDrsSignatureFullEqual internal error (1)");
  for( i = 0;  i < HaArrayCount(sig_set1->signatures);  i++ )
  {
    sig1 = HaArray(sig_set1->signatures, i);
    sig2 = HaArray(sig_set2->signatures, i);
    HnAssert(HaArrayCount(sig1->states) == HaArrayCount(sig2->states),
      "KheDrsSignatureFullEqual: signatures have different lengths");
    for( j = 0;  j < HaArrayCount(sig1->states);  j++ )
      if( HaArray(sig1->states, j).i != HaArray(sig2->states, j).i )
	return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureSetAddSignature(KHE_DRS_SIGNATURE_SET sig_set,       */
/*    KHE_DRS_SIGNATURE sig, bool with_cost)                                 */
/*                                                                           */
/*  Add sig to sig_set, but only include its cost if with_cost is true.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureSetAddSignature(KHE_DRS_SIGNATURE_SET sig_set,
  KHE_DRS_SIGNATURE sig, bool with_cost)
{
  if( with_cost )
    sig_set->cost += sig->cost;
  HaArrayAddLast(sig_set->signatures, sig);
  KheDrsSignatureRefer(sig);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureSetDebug(KHE_DRS_SIGNATURE_SET sig_set,              */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of sig_set onto fp with the given verbosity and indent.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureSetDebug(KHE_DRS_SIGNATURE_SET sig_set,
  int verbosity, int indent, FILE *fp)
{
  int i;  KHE_DRS_SIGNATURE sig;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "<%.5f: ", KheCostShow(sig_set->cost));
  HaArrayForEach(sig_set->signatures, sig, i)
    KheDrsSignatureDebug(sig, verbosity, -1, fp);
  fprintf(fp, ">");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_CORRELATOR" - handle correlated expressions           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CORRELATOR KheDrsCorrelatorMake(KHE_DYNAMIC_RESOURCE_SOLVER drs) */
/*                                                                           */
/*  Make a new, empty correlator object.                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CORRELATOR KheDrsCorrelatorMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CORRELATOR res;  HA_ARENA a;
  a = drs->arena;
  HaMake(res, a);
  res->make_correlated = false;		/* reset when solving */
  HaArrayInit(res->children, a);
  HaArrayInit(res->positives, a);
  HaArrayInit(res->negatives, a);
  HaArrayInit(res->singles, a);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorClear(KHE_DRS_CORRELATOR dc)                        */
/*                                                                           */
/*  Clear dc.                                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorClear(KHE_DRS_CORRELATOR dc)
{
  dc->make_correlated = false;
  HaArrayClear(dc->children);
  HaArrayClear(dc->positives);
  HaArrayClear(dc->negatives);
  HaArrayClear(dc->singles);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorMakeCorrelated(KHE_DRS_CORRELATOR dc)               */
/*                                                                           */
/*  Inform dc that correlated expressions are wanted.                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorMakeCorrelated(KHE_DRS_CORRELATOR dc)
{
  dc->make_correlated = true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorr1Debug(KHE_DRS_DOM_TEST prnt_dt,                          */
/*    KHE_DRS_DOM_TEST child_dt, KHE_DRS_SIGNER dsg,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of corr1 (prnt_dt, child_dt).                                */
/*                                                                           */
/*****************************************************************************/
static char *KheDrsSignerId(KHE_DRS_SIGNER dsg);

static void KheDrsCorr1Debug(KHE_DRS_DOM_TEST prnt_dt,
  KHE_DRS_DOM_TEST child_dt, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  if( DEBUG55 )
  {
    fprintf(fp, "%*s[ signer %s found corr1 (delta %d):\n", indent, "",
      KheDrsSignerId(dsg), prnt_dt->correlated_delta);
    KheDrsExprDebug(prnt_dt->expr, drs, verbosity, indent + 2, fp);
    KheDrsExprDebug(child_dt->expr, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorr3Debug(KHE_DRS_DOM_TEST dt, int dt_index,                 */
/*    KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of corr3, beginning at dt.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorr3Debug(KHE_DRS_DOM_TEST dt, int dt_index,
  KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  if( DEBUG55 )
  {
    fprintf(fp, "%*s[ signer %s found corr3:\n", indent, "",
      KheDrsSignerId(dsg));
    do
    {
      KheDrsExprDebug(dt->expr, drs, verbosity, indent + 2, fp);
      dt_index += dt->correlated_delta;
      dt = HaArray(dsg->dom_tests, dt_index);
    } while( dt->type != KHE_DRS_DOM_TEST_CORR3_LAST );
    KheDrsExprDebug(dt->expr, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorr4Debug(KHE_DRS_DOM_TEST dt, int dt_index,                 */
/*    KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of corr4, beginning at dt.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorr4Debug(KHE_DRS_DOM_TEST dt, int dt_index,
  KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  if( DEBUG55 )
  {
    fprintf(fp, "%*s[ signer %s found corr4:\n", indent, "",
      KheDrsSignerId(dsg));
    do
    {
      KheDrsExprDebug(dt->expr, drs, verbosity, indent + 2, fp);
      dt_index += dt->correlated_delta;
      dt = HaArray(dsg->dom_tests, dt_index);
    } while( dt->type != KHE_DRS_DOM_TEST_CORR4_LAST );
    KheDrsExprDebug(dt->expr, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*   void KheDrsCorrelatorAddChildDomTest(KHE_DRS_CORRELATOR dc,             */
/*     KHE_DRS_DOM_TEST child_dt, int child_dt_index, KHE_DRS_SIGNER dsg,    */
/*     KHE_DYNAMIC_RESOURCE_SOLVER drs)                                      */
/*                                                                           */
/*  Carry out KheDrsCorrelatorAddDomTest for the case where child_dt->expr   */
/*  is an OR expression which is the child of a KHE_DRS_EXPR_COUNTER         */
/*  expression.  The child always arrives before its parent.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorAddChildDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST child_dt, int child_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(child_dt->corr_dom_table4 != NULL,
    "KheDrsCorrelatorAddChildDomTest internal error");
  child_dt->type = KHE_DRS_DOM_TEST_CORR2_CHILD;
  HaArrayAddLast(dc->children, child_dt_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddParentDomTest(KHE_DRS_CORRELATOR dc,             */
/*    KHE_DRS_DOM_TEST prnt_dt, int prnt_dt_index, KHE_DRS_SIGNER dsg,       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Carry out the work of KheDrsCorrelatorAddDomTest for the case where      */
/*  prnt_dt->expr is an KHE_DRS_EXPR_COUNTER:  try to match it with a child  */
/*  dt, which must have arrived previously.                                  */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsParentIsFirstCounterParent(KHE_DRS_EXPR child,
  KHE_DRS_EXPR parent);

static void KheDrsCorrelatorAddParentDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST prnt_dt, int prnt_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST child_dt;  int child_dt_index, i;
  HaArrayForEach(dc->children, child_dt_index, i)
  {
    child_dt = HaArray(dsg->dom_tests, child_dt_index);
    if( KheDrsParentIsFirstCounterParent(child_dt->expr, prnt_dt->expr) )
    {
      prnt_dt->type = KHE_DRS_DOM_TEST_CORR1_PARENT;
      child_dt->type = KHE_DRS_DOM_TEST_CORR1_CHILD;
      child_dt->correlated_delta = prnt_dt_index - child_dt_index;
      HaArrayDeleteAndPlug(dc->children, i);
      KheDrsCorr1Debug(prnt_dt, child_dt, dsg, drs, 1, 0, stderr);
      return;
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddDaysPositiveDomTest(KHE_DRS_CORRELATOR dc,       */
/*    KHE_DRS_DOM_TEST pos_dt, int pos_dt_index, KHE_DRS_SIGNER dsg,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Carry out the work of KheDrsCorrelatorAddDomTest for the case where      */
/*  dt->expr is a days-positive KHE_DRS_EXPR_SEQUENCE.                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorAddDaysPositiveDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST pos_dt, int pos_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST neg_dt;  int neg_dt_index;
  if( HaArrayCount(dc->negatives) > 0 )
  {
    /* make a matching pair with any existing negative */
    neg_dt_index = HaArrayLastAndDelete(dc->negatives);
    neg_dt = HaArray(dsg->dom_tests, neg_dt_index);
    pos_dt->type = KHE_DRS_DOM_TEST_CORR4_FIRST;
    pos_dt->correlated_delta = neg_dt_index - pos_dt_index;
    neg_dt->type = KHE_DRS_DOM_TEST_CORR4_LAST;
    KheDrsCorr4Debug(pos_dt, pos_dt_index, dsg, drs, 1, 2, stderr);
  }
  else
  {
    /* store for a potential future match */
    HaArrayAddLast(dc->positives, pos_dt_index);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddDaysNegativeDomTest(KHE_DRS_CORRELATOR dc,       */
/*    KHE_DRS_DOM_TEST neg_dt, int neg_dt_index, KHE_DRS_SIGNER dsg,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Carry out the work of KheDrsCorrelatorAddDomTest for the case where      */
/*  dt->expr is a days-negative KHE_DRS_EXPR_SEQUENCE.                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorAddDaysNegativeDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST neg_dt, int neg_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST pos_dt;  int pos_dt_index;
  if( HaArrayCount(dc->positives) > 0 )
  {
    /* make a matching pair with any existing positive */
    pos_dt_index = HaArrayLastAndDelete(dc->positives);
    pos_dt = HaArray(dsg->dom_tests, pos_dt_index);
    pos_dt->type = KHE_DRS_DOM_TEST_CORR4_FIRST;
    pos_dt->correlated_delta = neg_dt_index - pos_dt_index;
    neg_dt->type = KHE_DRS_DOM_TEST_CORR4_LAST;
    KheDrsCorr4Debug(pos_dt, pos_dt_index, dsg, drs, 1, 2, stderr);
  }
  else
  {
    /* store for a potential future match */
    HaArrayAddLast(dc->negatives, neg_dt_index);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestSingleSeqIndexesEqual(KHE_DRS_DOM_TEST dt1,            */
/*    KHE_DRS_DOM_TEST dt2)                                                  */
/*                                                                           */
/*  Return true if the indexes of singles dt1 and dt2 are equal.             */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDomTestSingleSeqIndexesEqual(KHE_DRS_DOM_TEST dt1,
  KHE_DRS_DOM_TEST dt2)
{
  KHE_DRS_EXPR_SEQUENCE es1, es2;
  es1 = (KHE_DRS_EXPR_SEQUENCE) dt1->expr;
  es2 = (KHE_DRS_EXPR_SEQUENCE) dt2->expr;
  return es1->seq_index == es2->seq_index;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsCorrelatorSingleAdd(KHE_DRS_DOM_TEST single_dt,               */
/*    int single_dt_index, KHE_DRS_DOM_TEST new_single_dt,                   */
/*    int new_single_dt_index, KHE_DRS_SIGNER dsg,                           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Try to add new_single_dt (at new_single_dt_index) to the existing        */
/*  singles sequence beginning with single_dt (at single_dt_index).  If      */
/*  successful, update the existing singles sequence and return true.        */
/*  Otherwise change nothing and return false.                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsCorrelatorSingleAdd(KHE_DRS_DOM_TEST single_dt,
  int single_dt_index, KHE_DRS_DOM_TEST new_single_dt,
  int new_single_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST save_single_dt;  int save_single_dt_index;
  switch( single_dt->type )
  {
    case KHE_DRS_DOM_TEST_UNUSED:
    case KHE_DRS_DOM_TEST_SEPARATE_GENERIC:
    case KHE_DRS_DOM_TEST_SEPARATE_FLOAT:
    case KHE_DRS_DOM_TEST_CORR1_PARENT:
    case KHE_DRS_DOM_TEST_CORR1_CHILD:
    case KHE_DRS_DOM_TEST_CORR2_CHILD:

      /* should never happen */
      HnAbort("KheDrsCorrelatorSingleAdd internal error 1");
      return false;  /* keep compiler happy */

    case KHE_DRS_DOM_TEST_SEPARATE_INT:
    case KHE_DRS_DOM_TEST_TRADEOFF:
    case KHE_DRS_DOM_TEST_TABULATED:

      /* test and stop; if successful, result has two elements */
      if( KheDrsDomTestSingleSeqIndexesEqual(single_dt, new_single_dt) )
	return false;
      else
      {
        single_dt->type = KHE_DRS_DOM_TEST_CORR3_FIRST;
        single_dt->correlated_delta = new_single_dt_index - single_dt_index;
	new_single_dt->type = KHE_DRS_DOM_TEST_CORR3_LAST;
	KheDrsCorr3Debug(single_dt, single_dt_index, dsg, drs, 1, 2, stderr);
	return true;
      }

    case KHE_DRS_DOM_TEST_CORR3_FIRST:

      save_single_dt = single_dt;
      save_single_dt_index = single_dt_index;
      do
      {
	/* check new_single_dt against single_dt */
	if( KheDrsDomTestSingleSeqIndexesEqual(single_dt, new_single_dt) )
	  return false;

	/* move on to the next single_dt */
	single_dt_index += single_dt->correlated_delta;
	single_dt = HaArray(dsg->dom_tests, single_dt_index);
      } while( single_dt->type == KHE_DRS_DOM_TEST_CORR3_MID );

      /* check last, and if successful, add new_single_dt to the end */
      HnAssert(single_dt->type == KHE_DRS_DOM_TEST_CORR3_LAST,
        "KheDrsCorrelatorSingleAdd internal error 2");
      if( KheDrsDomTestSingleSeqIndexesEqual(single_dt, new_single_dt) )
	return false;
      else
      {
	single_dt->type = KHE_DRS_DOM_TEST_CORR3_MID;
	single_dt->correlated_delta = new_single_dt_index - single_dt_index;
	new_single_dt->type = KHE_DRS_DOM_TEST_CORR3_LAST;
	KheDrsCorr3Debug(save_single_dt, save_single_dt_index, dsg, drs,
	  1, 2, stderr);
	return true;
      }

    default:

      HnAbort("KheDrsCorrelatorSingleAdd internal error 2");
      return false;  /* keep compiler happy */
  }

  /* unreachable here */
  HnAbort("KheDrsCorrelatorSingleAdd internal error 3");
  return false;  /* keep compiler happy */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddSinglePositiveDomTest(KHE_DRS_CORRELATOR dc,     */
/*    KHE_DRS_DOM_TEST new_single_dt, int new_single_dt_index,               */
/*    KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Carry out the work of KheDrsCorrelatorAddDomTest for the case where      */
/*  new_single_dt->expr is a single-time positive KHE_DRS_EXPR_SEQUENCE.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorAddSinglePositiveDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST new_single_dt, int new_single_dt_index,
  KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int single_dt_index, i;  KHE_DRS_DOM_TEST single_dt;

  /* try adding to each singles sequence, and return if successful */
  HaArrayForEach(dc->singles, single_dt_index, i)
  {
    single_dt = HaArray(dsg->dom_tests, single_dt_index);
    if( KheDrsCorrelatorSingleAdd(single_dt, single_dt_index,
	  new_single_dt, new_single_dt_index, dsg, drs) )
      return;
  }

  /* no match so save new_single_dt for later */
  HaArrayAddLast(dc->singles, new_single_dt_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddDomTest(KHE_DRS_CORRELATOR dc,                   */
/*    KHE_DRS_DOM_TEST dt, int dt_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Add dt to dc.  This involves changing the type of dt to a correlated     */
/*  type..                                                                   */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsExprHasParent(KHE_DRS_EXPR child, KHE_DRS_EXPR_TAG tag);

static void KheDrsCorrelatorAddDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST dt, int dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_SEQUENCE es;
  if( dc->make_correlated )
  {
    if( DEBUG55 )
      fprintf(stderr, "[ KheDrsCorrelatorAddDomTest(dc, dt, %d, %s, drs)\n",
	dt_index, KheDrsSignerId(dsg));
    switch( dt->expr->tag )
    {
      case KHE_DRS_EXPR_OR_TAG:

        if( KheDrsExprHasParent(dt->expr, KHE_DRS_EXPR_COUNTER_TAG) )
	  KheDrsCorrelatorAddChildDomTest(dc, dt, dt_index, dsg, drs);
	break;

      case KHE_DRS_EXPR_COUNTER_TAG:

        KheDrsCorrelatorAddParentDomTest(dc, dt, dt_index, dsg, drs);
	break;

      case KHE_DRS_EXPR_SEQUENCE_TAG:

	es = (KHE_DRS_EXPR_SEQUENCE) dt->expr;
	if( es->seq_type == KHE_DRS_SEQ_DAYS_POSITIVE )
	  KheDrsCorrelatorAddDaysPositiveDomTest(dc, dt, dt_index, dsg, drs);
	else if( es->seq_type == KHE_DRS_SEQ_DAYS_NEGATIVE )
	  KheDrsCorrelatorAddDaysNegativeDomTest(dc, dt, dt_index, dsg, drs);
	else if( es->seq_type == KHE_DRS_SEQ_SINGLE_POSITIVE )
	  KheDrsCorrelatorAddSinglePositiveDomTest(dc, dt, dt_index, dsg, drs);
	break;

      default:

	/* ignore others */
	break;
    }
    if( DEBUG55 )
      fprintf(stderr, "] KheDrsCorrelatorAddDomTest returning\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SIGNER"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNER KheDrsSignerMake(KHE_DRS_DAY day,                         */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_SHIFT ds,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new, empty signer object for day, drd, or ds.                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNER KheDrsSignerMake(KHE_DRS_DAY day,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_SHIFT ds,
  KHE_DRS_SHIFT_PAIR dsp, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNER res;

  /* get res and make sure its arrays are initialized and empty */
  if( HaArrayCount(drs->signer_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->signer_free_list);
    HaArrayClear(res->internal_exprs);
    HaArrayClear(res->dom_tests);
    HaArrayClear(res->eq_dom_test_indexes);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->internal_exprs, drs->arena);
    HaArrayInit(res->dom_tests, drs->arena);
    HaArrayInit(res->eq_dom_test_indexes, drs->arena);
  }

  /* get a correlator, but only if drd type (never freed, by the way) */
  res->correlator = (drd != NULL ? KheDrsCorrelatorMake(drs) : NULL);

  /* set the other fields */
  res->solver = drs;
  res->last_hard_cost_index = -1;
  /* res->debug_eval_if_rerun = false; */  /* placeholder */
  if( day != NULL )
  {
    HnAssert(drd == NULL && ds == NULL, "KheDrsSignerMake internal error 1");
    res->encl_day = day;
    res->type = KHE_DRS_SIGNER_DAY;
    res->u.day = day;
  }
  else if( drd != NULL )
  {
    HnAssert(day == NULL && ds == NULL, "KheDrsSignerMake internal error 2");
    res->encl_day = drd->day;
    res->type = KHE_DRS_SIGNER_RESOURCE_ON_DAY;
    res->u.resource_on_day = drd;
  }
  else if( ds != NULL )
  {
    HnAssert(day == NULL && drd == NULL,"KheDrsSignerMake internal error 3");
    res->encl_day = ds->encl_day;
    res->type = KHE_DRS_SIGNER_SHIFT;
    res->u.shift = ds;
  }
  else if( dsp != NULL )
  {
    HnAssert(day == NULL && drd == NULL && ds == NULL,
      "KheDrsSignerMake internal error 3");
    res->encl_day = dsp->shift[0]->encl_day;
    res->type = KHE_DRS_SIGNER_SHIFT_PAIR;
    res->u.shift_pair = dsp;
  }
  else
    HnAbort("KheDrsSignerMake internal error 4");

  /* all good, return */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerMakeCorrelated(KHE_DRS_SIGNER dsg)                      */
/*                                                                           */
/*  Inform dsg that correlated expressions are wanted.                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerMakeCorrelated(KHE_DRS_SIGNER dsg)
{
  HnAssert(dsg->correlator != NULL,
    "KheDrsSignerMakeCorrelated internal error");
  KheDrsCorrelatorMakeCorrelated(dsg->correlator);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerFree(KHE_DRS_SIGNER dsg,                                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free dsg.                                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerFree(KHE_DRS_SIGNER dsg, 
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(dsg->correlator == NULL, "KheDrsSignerFree internal error");
  HaArrayAddLast(drs->signer_free_list, dsg);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerClear(KHE_DRS_SIGNER dsg, bool free_dom_tests,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Clear dsg ready for a new solve, without changing its type.  This        */
/*  includes freeing any dom tests left over from the previous solve.        */
/*                                                                           */
/*  Obsolete:  Also free                                                     */
/*  dsg's dom tests, unless dsg->type == KHE_DRS_SIGNER_RESOURCE_ON_DAY, in  */
/*  which case the dom tests are shared with the enclosing day signer, and   */
/*  that signer will free them.                                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerClear(KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i;
  if( DEBUG68 && dsg->type == KHE_DRS_SIGNER_SHIFT )
    fprintf(stderr, "  KheDrsSignerClear(dsg %p)\n", (void *) dsg);
  HaArrayClear(dsg->internal_exprs);
  HaArrayAppend(drs->dom_test_free_list, dsg->dom_tests, i);
  HaArrayClear(dsg->dom_tests);
  HaArrayClear(dsg->eq_dom_test_indexes);
  dsg->last_hard_cost_index = -1;
  /* dsg->debug_eval_if_rerun = false; */
  if( dsg->correlator != NULL )
    KheDrsCorrelatorClear(dsg->correlator);
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsSignerId(KHE_DRS_SIGNER dsg)                                 */
/*                                                                           */
/*  Return the Id if dsg.                                                    */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsSignerTypeShow(KHE_DRS_SIGNER_TYPE stype)
{
  switch( stype )
  {
    case KHE_DRS_SIGNER_DAY:

      return "Day";

    case KHE_DRS_SIGNER_RESOURCE_ON_DAY:

      return "ResourceOnDay";

    case KHE_DRS_SIGNER_SHIFT:

      return "Shift";

    case KHE_DRS_SIGNER_SHIFT_PAIR:

      return "ShiftPair";

    default:

      return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsSignerId(KHE_DRS_SIGNER dsg)                                 */
/*                                                                           */
/*  Return the Id if dsg.                                                    */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsSignerId(KHE_DRS_SIGNER dsg)
{
  switch( dsg->type )
  {
    case KHE_DRS_SIGNER_DAY:

      return KheDrsDayId(dsg->u.day);

    case KHE_DRS_SIGNER_RESOURCE_ON_DAY:

      return KheDrsResourceOnDayId(dsg->u.resource_on_day);

    case KHE_DRS_SIGNER_SHIFT:

      return KheDrsShiftId(dsg->u.shift);

    case KHE_DRS_SIGNER_SHIFT_PAIR:

      return KheDrsShiftPairId(dsg->u.shift_pair);

    default:

      return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerMonitorUpdateIfRerun(KHE_DRS_SIGNER dsg)                */
/*                                                                           */
/*  Return true if dsg goes with monitor updating during reruns.             */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignerMonitorUpdateIfRerun(KHE_DRS_SIGNER dsg)
{
  return dsg == NULL || dsg->type == KHE_DRS_SIGNER_DAY ||
    dsg->type == KHE_DRS_SIGNER_RESOURCE_ON_DAY;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSignerOpenDayIndex(KHE_DRS_SIGNER dsg)                         */
/*                                                                           */
/*  Return the open day index of dsg's day.                                  */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSignerOpenDayIndex(KHE_DRS_SIGNER dsg)
{
  return dsg->encl_day->open_day_index;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerAddOpenExpr(KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e)         */
/*                                                                           */
/*  Add e to dsg.                                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerAddOpenExpr(KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e)
{
  if( dsg->type == KHE_DRS_SIGNER_SHIFT )
  {
    HnAssert(e->tag == KHE_DRS_EXPR_COUNTER_TAG,
      "KheDrsSignerAddOpenExpr internal error");
    if( DEBUG68 )
      fprintf(stderr, "  KheDrsSignerAddOpenExpr(dsg %p, e %p)\n",
	(void *) dsg, (void *) e);
  }
  HaArrayAddLast(dsg->internal_exprs, e);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSignerAddDomTest(KHE_DRS_SIGNER dsg,                           */
/*    KHE_DRS_DOM_TEST dom_test)                                             */
/*                                                                           */
/*  Add dom_test to dsg and return its index in dsg.                         */
/*                                                                           */
/*  If dsg->make_correlated is set, identify pairs of correlated dom         */
/*  tests and adjust them appropriately.                                     */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsDomTestIsStrictEquality(KHE_DRS_DOM_TEST dom_test);
static KHE_COST KheDrsMonitorWeight(KHE_DRS_MONITOR m);

static int KheDrsSignerAddDomTest(KHE_DRS_SIGNER dsg,
  KHE_DRS_DOM_TEST dt, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int dt_index;

  /* add dt */
  HnAssert(dt != NULL, "KheDrsSignerAddDomTest internal error");
  dt_index = HaArrayCount(dsg->dom_tests);
  HaArrayAddLast(dsg->dom_tests, dt);

  /* if dt is equality, record it in eq_dom_test_indexes */
  if( KheDrsDomTestIsStrictEquality(dt) )
    HaArrayAddLast(dsg->eq_dom_test_indexes, dt_index);

  /* handle correlated expressions, if wanted */
  if( dsg->correlator != NULL )
    KheDrsCorrelatorAddDomTest(dsg->correlator, dt, dt_index, dsg, drs);

  /* keep track of where hard constraints end */
  if( KheDrsMonitorWeight(dt->monitor) >= KheCost(1, 0) )
    dsg->last_hard_cost_index = dt_index;

  /* return dt's index */
  return dt_index;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerAddExpr(KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e,             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int *index)                           */
/*                                                                           */
/*  See whether e needs to be added to dsg and do so if so.  If this         */
/*  involves adding a dom test to dsg, set *index to its index and           */
/*  return true, otherwise return false.                                     */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_EXPR_EVAL_TYPE KheDrsExprEvalType(KHE_DRS_EXPR e,
  KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_DOM_TEST *dom_test);

static bool KheDrsSignerAddExpr(KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int *index)
{
  KHE_DRS_DOM_TEST dom_test;
  switch( KheDrsExprEvalType(e, dsg, drs, &dom_test) )
  {
    case KHE_DRS_EXPR_EVAL_NO:

      /* do nothing */
      return *index = -1, false;

    case KHE_DRS_EXPR_EVAL_NOT_LAST:

      /* add expression and dom test */
      KheDrsSignerAddOpenExpr(dsg, e);
      return *index = KheDrsSignerAddDomTest(dsg, dom_test, drs), true;

    case KHE_DRS_EXPR_EVAL_LAST:

      /* add expression only */
      KheDrsSignerAddOpenExpr(dsg, e);
      return *index = -1, false;

    default:

      HnAbort("KheDrsSignerAddExpr internal error");
      return *index = -1, false;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNATURE KheDrsSignerEvalSignature(KHE_DRS_SIGNER dsg,          */
/*    KHE_DRS_SIGNATURE prev_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs,bool debug)*/
/*                                                                           */
/*  Using dsg, make and fill a new signature.                                */
/*                                                                           */
/*  The signature has a 0 reference count on return.  It is up to the        */
/*  caller to decide whether the signature is referred to or not.            */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, /* int next_di, */ KHE_DRS_SIGNATURE next_sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug);

static KHE_DRS_SIGNATURE KheDrsSignerEvalSignature(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)
{
  KHE_DRS_EXPR e;  int i;  KHE_DRS_SIGNATURE res;
  if( RERUN_DEBUG(drs) )
  {
    fprintf(stderr, "[ KheDrsSignerEvalSignature(");
    KheDrsSignerDebug(dsg, 1, -1, stderr);
    fprintf(stderr, "\n");
  }
  res = KheDrsSignatureMake(drs);
  HaArrayForEach(dsg->internal_exprs, e, i)
    KheDrsExprEvalSignature(e, dsg, prev_sig, res, drs, debug);
  if( RERUN_DEBUG(drs) )
    fprintf(stderr, "] KheDrsSignerEvalSignature\n");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDominanceDebugSpacer(int indent, FILE *fp)                    */
/*                                                                           */
/*  Print a spacer line in a debug print of a dominance test.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDominanceDebugSpacer(int indent, FILE *fp)
{
  if( fp != NULL )
    fprintf(fp, "%*s------------------------------------------\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsPrintLimits(char *tag, bool allow_zero, int min_lim,          */
/*    int max_lim, KHE_COST combined_wt, KHE_COST_FUNCTION cost_fn, FILE *fp)*/
/*                                                                           */
/*  Print all this stuff in an intuitive format.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsPrintLimits(char *tag, bool allow_zero, int min_lim,
  int max_lim, KHE_COST combined_wt, KHE_COST_FUNCTION cost_fn, FILE *fp)
{
  char buff[200], *buff2;  int soft_wt, hard_wt;

  /* print the initial tag */
  fprintf(fp, "%-4s ", tag);

  /* print allow_zero, min_lim, and max_lim if required */
  if( min_lim >= 0 )
  {
    /* sort out allow_zero flag */
    if( allow_zero && min_lim <= 1 )
      allow_zero = false, min_lim = 0;

    /* print min_lim and max_lim, where present */
    if( min_lim > 0 )
    {
      if( max_lim < INT_MAX )
      {
	/* have min_lim and max_lim */
	if( min_lim == max_lim )
	  sprintf(buff, "%s%d", allow_zero ? "0," : "", min_lim);
	else
	  sprintf(buff, "%s%d-%d", allow_zero ? "0," : "", min_lim, max_lim);
      }
      else
      {
	/* have min_lim only */
	sprintf(buff, "%s%d-inf", allow_zero ? "0," : "", min_lim);
      }
    }
    else
    {
      if( max_lim < INT_MAX )
      {
	/* have max_lim only */
	sprintf(buff, "%s0-%d", allow_zero ? "0," : "", max_lim);
      }
      else
      {
	/* have no limits at all */
	sprintf(buff, "%s-", allow_zero ? "0," : "");
      }
    }
  }
  else
    sprintf(buff, "%s", "");

  /* print bar and the the soft and hard weight */
  buff2 = &buff[strlen(buff)];
  soft_wt = KheSoftCost(combined_wt);
  hard_wt = KheHardCost(combined_wt);
  if( soft_wt > 0 )
  {
    if( hard_wt > 0 )
    {
      /* soft and hard weight */
      sprintf(buff2, "|h%d+s%d", hard_wt, soft_wt);
    }
    else
    {
      /* soft weight only */
      sprintf(buff2, "|s%d", soft_wt);
    }
  }
  else
  {
    if( hard_wt > 0 )
    {
      /* hard weight only */
      sprintf(buff2, "|h%d", hard_wt);
    }
    else
    {
      /* no weight at all */
      sprintf(buff2, "|0");
    }
  }

  /* print the cost function */
  buff2 = &buff[strlen(buff)];
  switch( cost_fn )
  {
    case KHE_STEP_COST_FUNCTION:

      sprintf(buff2, "s");
      break;

    case KHE_LINEAR_COST_FUNCTION:

      /* default, don't print anything */
      break;

    case KHE_QUADRATIC_COST_FUNCTION:

      sprintf(buff2, "q");
      break;

    default:

      sprintf(buff2, "?");
  }

  /* print it all out, fixed width */
  fprintf(fp, "%-8s", buff);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorDebug(KHE_DRS_MONITOR dm, FILE *fp)                    */
/*                                                                           */
/*  Debug print of m with a few key attributes.  Alternatively, if m is      */
/*  NULL, print nothing.                                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorDebug(KHE_DRS_MONITOR dm, FILE *fp)
{
  /* KHE_CONSTRAINT c; */  KHE_COST wt;  KHE_COST_FUNCTION cost_fn;
  int min_lim, max_lim;  bool allow_zero;
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;
  KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm;
  KHE_LIMIT_BUSY_TIMES_CONSTRAINT lbtc;
  KHE_LIMIT_BUSY_TIMES_MONITOR lbtm;
  KHE_LIMIT_WORKLOAD_CONSTRAINT lwc;
  KHE_LIMIT_WORKLOAD_MONITOR lwm;
  KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT laic;
  KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim;
  KHE_LIMIT_RESOURCES_CONSTRAINT lrc;
  KHE_LIMIT_RESOURCES_MONITOR lrm;
  KHE_MONITOR m;
  m = dm->monitor;
  if( m != NULL )
  {
    /* c = KheMonito rConstraint(m); */
    wt = KheMonitorCombinedWeight(m);
    cost_fn = KheMonitorCostFunction(m);
    /* switch( KheConstraintTag(c) ) */
    switch( KheMonitorTag(m) )
    {
      case KHE_ASSIGN_RESOURCE_MONITOR_TAG:

	KheDrsPrintLimits("arc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_PREFER_RESOURCES_MONITOR_TAG:

	KheDrsPrintLimits("prc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR_TAG:

	KheDrsPrintLimits("asac", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_AVOID_CLASHES_MONITOR_TAG:

	KheDrsPrintLimits("acc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG:

	KheDrsPrintLimits("autc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_LIMIT_IDLE_TIMES_MONITOR_TAG:

	KheDrsPrintLimits("litc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

	cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
	cbtc = KheClusterBusyTimesMonitorConstraint(cbtm);
	min_lim = KheClusterBusyTimesMonitorMinimum(cbtm);
	max_lim = KheClusterBusyTimesMonitorMaximum(cbtm);
	allow_zero = KheClusterBusyTimesConstraintAllowZero(cbtc);
	KheDrsPrintLimits("cbtc", allow_zero, min_lim, max_lim, wt, cost_fn,fp);
	break;

      case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

	lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
	lbtc = KheLimitBusyTimesMonitorConstraint(lbtm);
	min_lim = KheLimitBusyTimesConstraintMinimum(lbtc);
	max_lim = KheLimitBusyTimesConstraintMaximum(lbtc);
	allow_zero = KheLimitBusyTimesConstraintAllowZero(lbtc);
	KheDrsPrintLimits("lbtc", allow_zero, min_lim, max_lim, wt, cost_fn,fp);
	break;

      case KHE_LIMIT_WORKLOAD_MONITOR_TAG:

	lwm = (KHE_LIMIT_WORKLOAD_MONITOR) m;
	lwc = KheLimitWorkloadMonitorConstraint(lwm);
	min_lim = KheLimitWorkloadConstraintMinimum(lwc);
	max_lim = KheLimitWorkloadConstraintMaximum(lwc);
	allow_zero = KheLimitWorkloadConstraintAllowZero(lwc);
	KheDrsPrintLimits("lwc", allow_zero, min_lim, max_lim, wt, cost_fn, fp);
	break;

      case KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG:

	laim = (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) m;
	laic = KheLimitActiveIntervalsMonitorConstraint(laim);
	min_lim = KheLimitActiveIntervalsConstraintMinimum(laic);
	max_lim = KheLimitActiveIntervalsConstraintMaximum(laic);
	KheDrsPrintLimits("laic", false, min_lim, max_lim, wt, cost_fn, fp);
	break;

      case KHE_LIMIT_RESOURCES_MONITOR_TAG:

	lrm = (KHE_LIMIT_RESOURCES_MONITOR) m;
	lrc = KheLimitResourcesMonitorConstraint(lrm);
	min_lim = KheLimitResourcesConstraintMinimum(lrc);
	max_lim = KheLimitResourcesConstraintMaximum(lrc);
	KheDrsPrintLimits("lrc", false, min_lim, max_lim, wt, cost_fn, fp);
	break;

      default:

	fprintf(fp, "? %.5f \n", KheCostShow(wt));
	break;
    }
    fprintf(fp, " %s", KheMonitorId(m));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDominanceDebugCmpLine(bool res, KHE_COST avail_cost,          */
/*    KHE_DRS_DOM_TEST_TYPE type, int i, int v1, int v2, KHE_DRS_MONITOR m,  */
/*    char *details, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Print of one line of a dominance debug, but only if fp != NULL.          */
/*                                                                           */
/*****************************************************************************/
static char *KheDrsDomTestTypeShow(KHE_DRS_DOM_TEST_TYPE type);

static void KheDrsDominanceDebugCmpLine(KHE_COST avail_cost,
  KHE_DRS_DOM_TEST_TYPE type, int i, int v1, int v2, KHE_DRS_MONITOR m,
  int indent, FILE *fp)
{
  if( fp != NULL )
  {
    fprintf(fp, "%*s%8.5f  %2s[%2d] %2d %2d", indent, "",
      KheCostShow(avail_cost), KheDrsDomTestTypeShow(type), i, v1, v2);
    if( m != NULL )
    {
      fprintf(fp, "  ");
      KheDrsMonitorDebug(m, fp);
    }
    fprintf(fp, "\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerCorr1Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost, int verbosity,     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Carry out a CORR1 dominance test whose first (child) element is dt.      */
/*                                                                           */
/*  Implementation note.  Care is needed here:  the weight must come         */
/*  from other_dt (the parent), not from dt (the child).                     */
/*  Tables:  corr_dom_table4                                                 */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_COST_TUPLE KheDrsDim4TableGet4(KHE_DRS_DIM4_TABLE d4,
  int index4, int index3, int index2, int index1);

static void KheDrsSignerCorr1Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int a1, int a2, KHE_COST *avail_cost, int verbosity,
  int indent, FILE *fp)
{
  int other_dt_index, l1, l2;  KHE_DRS_DOM_TEST other_dt;  KHE_COST psi;
  KHE_DRS_COST_TUPLE ct;
  other_dt_index = dt_index + dt->correlated_delta;
  other_dt = HaArray(dsg->dom_tests, other_dt_index);
  HnAssert(other_dt->type == KHE_DRS_DOM_TEST_CORR1_PARENT,
    "KheDrsSignerCorr1Dominates internal error (other_dt_index = %d)",
    other_dt_index);
  l1 = HaArray(sig1->states, other_dt_index).i;
  l2 = HaArray(sig2->states, other_dt_index).i;
  ct = KheDrsDim4TableGet4(dt->corr_dom_table4, a1, a2, l1, l2);
  psi = ct.unweighted_psi * other_dt->combined_weight;
  if( fp != NULL )
    fprintf(fp, "\n%*s[ Corr1(a1 %d, a2 %d, l1 %d, l2 %d) -> %.5f ]\n\n",
      indent, "", a1, a2, l1, l2, KheCostShow(psi));
  *avail_cost += psi;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerCorr2Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost, int verbosity,     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Carry out a CORR2 dominance test whose sole (child) element is dt.       */
/*                                                                           */
/*  Implementation note.  Care is needed here:  the weight must come         */
/*  from es (the parent), not from dt (the child).                           */
/*  Tables:  corr_dom_table4                                                 */
/*                                                                           */
/*****************************************************************************/
static int KheDrsExprCounterInitialValue(KHE_DRS_EXPR_COUNTER ec);

static void KheDrsSignerCorr2Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int a1, int a2, KHE_COST *avail_cost, int verbosity,
  int indent, FILE *fp)
{
  int l1, l2;  KHE_DRS_EXPR_COUNTER ec;  KHE_DRS_PARENT prnt;
  KHE_DRS_COST_TUPLE ct;  KHE_COST psi;
  HnAssert(HaArrayCount(dt->expr->parents) > 0,
    "KheDrsSignerCorr2Dominates internal error (no parents)");
  prnt = HaArrayFirst(dt->expr->parents);
  HnAssert(prnt.expr->tag == KHE_DRS_EXPR_COUNTER_TAG,
    "KheDrsSignerCorr2Dominates internal error (prnt.expr->tag = %d)",
    prnt.expr->tag);
  ec = (KHE_DRS_EXPR_COUNTER) prnt.expr;
  l1 = l2 = KheDrsExprCounterInitialValue(ec);
  ct = KheDrsDim4TableGet4(dt->corr_dom_table4, a1, a2, l1, l2);
  psi = ct.unweighted_psi * ec->combined_weight;
  if( fp != NULL )
    fprintf(fp, "\n%*s[ Corr2(a1 %d, a2 %d, l1 %d, l2 %d) -> %.5f ]\n\n",
      indent, "", a1, a2, l1, l2, KheCostShow(psi));
  *avail_cost += psi;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDebugCorr3Header(int verbosity, int indent, FILE *fp)         */
/*  void KheDrsDebugCorr4Header(int verbosity, int indent, FILE *fp)         */
/*                                                                           */
/*  Debug print of the header of a Corr3 or Corr4 test (only if fp != NULL). */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDebugCorr3Header(int verbosity, int indent, FILE *fp)
{
  if( fp != NULL )
    fprintf(fp, "\n%*s[ %-6s %2s %2s  %9s %9s %9s %9s\n", indent, "",
      "Corr3", "v1", "v2", "Psi", "Psi0", "min(diff)", "sum(Psi0)");
}

static void KheDrsDebugCorr4Header(int verbosity, int indent, FILE *fp)
{
  if( fp != NULL )
    fprintf(fp, "\n%*s[ %-6s %2s %2s  %9s %9s %9s %9s\n", indent, "",
      "Corr4", "v1", "v2", "Psi+", "Psi0", "min(diff)", "sum(Psi0)");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDebugCorr34Line(KHE_DRS_DOM_TEST dt, int dt_index,            */
/*    int v1, int v2, KHE_COST psi, KHE_COST psi0, KHE_COST diff,            */
/*    KHE_COST min_diff, KHE_COST sum_psi0, int indent, FILE *fp)            */
/*                                                                           */
/*  Print one line of a corr3 or corr4 debug (only if fp != NULL).           */
/*                                                                           */
/*****************************************************************************/
static void KheDrsMonitorDebug(KHE_DRS_MONITOR m, FILE *fp);

static void KheDrsDebugCorr34Line(KHE_DRS_DOM_TEST dt, int dt_index,
  int v1, int v2, KHE_COST psi, KHE_COST psi0, KHE_COST min_diff,
  KHE_COST sum_psi0, int indent, FILE *fp)
{
  if( fp != NULL )
  {
    fprintf(fp, "%*s  %2s[%2d] %2d %2d  %9.5f %9.5f %9.5f %9.5f ",
      indent, "", KheDrsDomTestTypeShow(dt->type), dt_index, v1, v2,
      KheCostShow(psi), KheCostShow(psi0), KheCostShow(min_diff),
      KheCostShow(sum_psi0));
    KheDrsMonitorDebug(dt->monitor, fp);
    fprintf(fp, "\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerNextCorrelated(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, int *dt_index,         */
/*    KHE_DRS_DOM_TEST *dt, int *v1, int *v2)                                */
/*                                                                           */
/*  Move to the next spot in the correlated sequence.                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerNextCorrelated(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, int *dt_index,
  KHE_DRS_DOM_TEST *dt, int *v1, int *v2)
{
  *dt_index += (*dt)->correlated_delta;
  *dt = HaArray(dsg->dom_tests, *dt_index);
  *v1 = HaArray(sig1->states, *dt_index).i;
  *v2 = HaArray(sig2->states, *dt_index).i;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerPsiGet(KHE_DRS_DOM_TEST dt, int v1, int v2,             */
/*    KHE_COST *psi, KHE_COST *psi0, KHE_COST *psi_plus)                     */
/*                                                                           */
/*  Get *psi, *psi0, and (optionally) *psi_plus from dt's main table.        */
/*  Tables:  main_dom_table2                                                 */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_COST_TUPLE KheDrsDim2TableGet2(KHE_DRS_DIM2_TABLE d2,
  int index2, int index1);

static void KheDrsSignerPsiGet(KHE_DRS_DOM_TEST dt, int v1, int v2,
  KHE_COST *psi, KHE_COST *psi0, KHE_COST *psi_plus)
{
  KHE_DRS_COST_TUPLE ct;
  ct = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
  *psi = ct.unweighted_psi * dt->combined_weight;
  *psi0 = ct.unweighted_psi0 * dt->combined_weight;
  if( psi_plus != NULL )
    *psi_plus = ct.unweighted_psi_plus * dt->combined_weight;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerPsiCumInit(KHE_COST psi, KHE_COST psi0,                 */
/*    KHE_COST psi_plus, KHE_COST *sum_psi, KHE_COST *sum_psi0,              */
/*    KHE_COST *min_psi_diff, KHE_COST *min_psi_plus_diff)                   */
/*                                                                           */
/*  void KheDrsSignerPsiCumAdd(KHE_COST psi, KHE_COST psi0,                  */
/*    KHE_COST psi_plus, KHE_COST *sum_psi, KHE_COST *sum_psi0,              */
/*    KHE_COST *min_psi_diff, KHE_COST *min_psi_plus_diff)                   */
/*                                                                           */
/*  Initialize and accumulate values into the cumulative variables:          */
/*                                                                           */
/*    Variable             Formula                                           */
/*    -----------------------------------------                              */
/*    *sum_psi             sum(psi)                                          */
/*    *sum_psi0            sum(psi0)                                         */
/*    *min_psi_diff        min(psi - psi0)                                   */
/*    *min_psi_plus_diff   min(psi_plus - psi0)                              */
/*    -----------------------------------------                              */
/*                                                                           */
/*  Omit *min_psi_plus_diff if min_psi_plus_diff == NULL.                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerPsiCumInit(KHE_COST psi, KHE_COST psi0,
  KHE_COST psi_plus, KHE_COST *sum_psi, KHE_COST *sum_psi0,
  KHE_COST *min_psi_diff, KHE_COST *min_psi_plus_diff)
{
  *sum_psi = psi;
  *sum_psi0 = psi0;
  *min_psi_diff = psi - psi0;
  if( min_psi_plus_diff != NULL )
    *min_psi_plus_diff = psi_plus - psi0;
}

static void KheDrsSignerPsiCumAdd(KHE_COST psi, KHE_COST psi0,
  KHE_COST psi_plus, KHE_COST *sum_psi, KHE_COST *sum_psi0,
  KHE_COST *min_psi_diff, KHE_COST *min_psi_plus_diff)
{
  KHE_COST tmp;
  *sum_psi += psi;
  *sum_psi0 += psi0;
  tmp = psi - psi0;
  if( tmp < *min_psi_diff )
    *min_psi_diff = tmp;
  if( min_psi_plus_diff != NULL )
  {
    tmp = psi_plus - psi0;
    if( tmp < *min_psi_plus_diff )
      *min_psi_plus_diff = tmp;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerCorr3Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost,                    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Carry out a CORR3 dominance test whose first element is dt.              */
/*  Tables:  main_dom_table2                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerCorr3Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int v1, int v2, KHE_COST *avail_cost,
  int verbosity, int indent, FILE *fp)
{
  KHE_COST psi, psi0, sum_psi, sum_psi0, min_psi_diff;  bool differ3;
  KheDrsDebugCorr3Header(verbosity, indent, fp);
  KheDrsSignerPsiGet(dt, v1, v2, &psi, &psi0, NULL);
  KheDrsSignerPsiCumInit(psi, psi0, 0, &sum_psi, &sum_psi0, &min_psi_diff,NULL);
  KheDrsDebugCorr34Line(dt, dt_index, v1, v2, psi, psi0, min_psi_diff,
    sum_psi0, indent, fp);
  do
  {
    KheDrsSignerNextCorrelated(dsg, sig1, sig2, &dt_index, &dt, &v1, &v2);
    KheDrsSignerPsiGet(dt, v1, v2, &psi, &psi0, NULL);
    KheDrsSignerPsiCumAdd(psi, psi0, 0, &sum_psi, &sum_psi0, &min_psi_diff,
      NULL);
    KheDrsDebugCorr34Line(dt, dt_index, v1, v2, psi, psi0, min_psi_diff,
      sum_psi0, indent, fp);
  } while( dt->type != KHE_DRS_DOM_TEST_CORR3_LAST );

  /* and there we are */
  *avail_cost += (min_psi_diff + sum_psi0);
  if( fp != NULL )
  {
    differ3 = (min_psi_diff + sum_psi0 != sum_psi);
    fprintf(fp, "%*s] corr3 %.5f, uncorrelated %.5f%s\n\n", indent, "",
      KheCostShow(min_psi_diff + sum_psi0), KheCostShow(sum_psi),
      differ3 ? " (corr3 differs from uncorrelated)" : "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerCorr4Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost,                    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Like KheDrsSignerCorr4Dominates, only with debug output.                 */
/*  Tables:  main_dom_table2                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerCorr4Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int v1, int v2, KHE_COST *avail_cost,
  int verbosity, int indent, FILE *fp)
{
  KHE_COST psi, psi0, psi_plus, min_psi_diff;
  KHE_COST min_psi_plus_diff, sum_psi0, sum_psi;  bool differ3, differ4;
  KheDrsDebugCorr4Header(verbosity, indent, fp);
  KheDrsSignerPsiGet(dt, v1, v2, &psi, &psi0, &psi_plus);
  KheDrsSignerPsiCumInit(psi, psi0, psi_plus, &sum_psi, &sum_psi0,
    &min_psi_diff, &min_psi_plus_diff);
  KheDrsDebugCorr34Line(dt, dt_index, v1, v2, psi_plus, psi0,
    min_psi_plus_diff, sum_psi0, indent, fp);
  do
  {
    KheDrsSignerNextCorrelated(dsg, sig1, sig2, &dt_index, &dt, &v1, &v2);
    KheDrsSignerPsiGet(dt, v1, v2, &psi, &psi0, &psi_plus);
    KheDrsSignerPsiCumAdd(psi, psi0, psi_plus, &sum_psi, &sum_psi0,
      &min_psi_diff, &min_psi_plus_diff);
    KheDrsDebugCorr34Line(dt, dt_index, v1, v2, psi_plus, psi0,
      min_psi_plus_diff, sum_psi0, indent, fp);
  } while( dt->type != KHE_DRS_DOM_TEST_CORR4_LAST );

  /* and there we are */
  *avail_cost += (min_psi_plus_diff + sum_psi0);
  if( fp != NULL )
  {
    differ3 = (min_psi_diff + sum_psi0 != sum_psi);
    differ4 = (min_psi_plus_diff != min_psi_diff);
    fprintf(fp, "%*s] corr4 %.5f, corr3 %.5f, uncorrelated %.5f%s%s\n\n",
      indent, "", KheCostShow(min_psi_plus_diff + sum_psi0),
      KheCostShow(min_psi_diff + sum_psi0), KheCostShow(sum_psi),
      differ3 ? " (corr3 differs from uncorrelated)" : "",
      differ4 ? " (corr4 differs from corr3)" : "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDominanceDebugHeaderLine(char *desc, KHE_DRS_SIGNER dsg,      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,                        */
/*    KHE_DRS_SIGNER_TEST test, KHE_COST avail_cost, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print one header line of a dominance test, after a spacer line.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDominanceDebugHeaderLine(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_DRS_SIGNER_TEST test, KHE_COST avail_cost, int indent, FILE *fp)
{
  char *descr[3] = {"Hard", "Soft", "All"};
  if( fp != NULL )
  {
    KheDrsDominanceDebugSpacer(indent, fp);
    fprintf(fp, "%*s%s %-11s v1 v2 (avail %.5f)\n", indent, "",
      descr[(int) test], KheDrsSignerId(dsg), KheCostShow(avail_cost));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerDoDominates(KHE_DRS_SIGNER dsg,                         */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,                        */
/*    KHE_DRS_SIGNER_TEST test, int trie_start_depth, bool stop_on_neg,      */
/*    KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)             */
/*                                                                           */
/*  This function does most of the work of deciding whether sig1             */
/*  dominates sig2.  It has several extra parameters, allowing it            */
/*  to handle several variants of the idea, as follows.                      */
/*                                                                           */
/*  Parameter test says whether to compare only hard constraints, or only    */
/*  soft constraints, or all.  The individual tests can be done in any       */
/*  order, and doing all the hard constraints of several signers before      */
/*  all the soft constraints is known to save time.                          */
/*                                                                           */
/*  Parameter trie_start_depth is non-zero only when this function is        */
/*  called from within the trie data structure.  It specifies that some      */
/*  positions have already been tested and testing is to start after         */
/*  those.  Must be testing all, not hard or soft, else abort.               */
/*                                                                           */
/*  Parameter stop_on_neg, when true, specifies that the testing is to       */
/*  stop as soon as *avail_cost goes negative.                               */
/*                                                                           */
/*  Parameter *avail_cost is the available cost on entry and the             */
/*  remaining available cost on exit.  KheDrsSignerDoDominates does not      */
/*  consult the cost fields of sig1 and sig2; it assumes that they have      */
/*  already given rise to a suitable initial value of *avail_cost.           */
/*                                                                           */
/*  If fp != NULL, produce debug output while doing this.                    */
/*                                                                           */
/*  Implementation note.  I've added comments showing which tables are       */
/*  accessed by which tests.  The point is that we don't have tables for     */
/*  limit workload monitors, so we need to make sure that the dominance      */
/*  tests of expressions derived from limit workload monitors are not of     */
/*  the sort that expect tables.                                             */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsDomTestDominatesSeparateInt(KHE_DRS_DOM_TEST dt,
  int val1, int val2);
static bool KheDrsDomTestDominatesSeparateFloat(KHE_DRS_DOM_TEST dt,
  float val1, float val2);
static bool KheDrsDomTestDominatesTradeoff(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost);
static void KheDrsDomTestDominatesTabulated(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost);

static bool KheDrsSignerDoDominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_DRS_SIGNER_TEST test, int trie_start_depth, bool stop_on_neg,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_DOM_TEST dt;  int start_index, stop_index, dt_index, sig_len;
  KHE_DRS_VALUE v1, v2;

  /* consistency checks */
  sig_len = HaArrayCount(dsg->dom_tests);
  HnAssert(HaArrayCount(sig1->states) == sig_len,
    "KheDrsSignerDoDominates internal error 1 (count %d != count %d)\n",
    HaArrayCount(sig1->states), sig_len);
  HnAssert(HaArrayCount(sig2->states) == sig_len,
    "KheDrsSignerDoDominates internal error 2 (count %d != count %d)\n",
    HaArrayCount(sig2->states), sig_len);

  /* work out start_index and stop_index */
  switch( test )
  {
    case KHE_DRS_SIGNER_HARD:

      HnAssert(trie_start_depth == 0,
	"KheDrsSignerDoDominates internal error 1");
      start_index = 0;
      stop_index = dsg->last_hard_cost_index + 1;
      break;

    case KHE_DRS_SIGNER_SOFT:

      HnAssert(trie_start_depth == 0,
	"KheDrsSignerDoDominates internal error 2");
      start_index = dsg->last_hard_cost_index + 1;
      stop_index = sig_len;
      break;

    case KHE_DRS_SIGNER_ALL:

      start_index = trie_start_depth;
      stop_index = sig_len;
      break;

    default:

      HnAbort("KheDrsSignerDoDominates internal error 3");
      start_index = 0, stop_index = 0;  /* keep compiler happy */
  }

  /* print spacer and header line */
  KheDrsDominanceDebugHeaderLine(dsg, sig1, sig2, test, *avail_cost,
    indent, fp);

  /* quit immediately if no available cost */
  if( stop_on_neg && *avail_cost < 0 )
    return false;

  /* do the actual test from start_index inclusive to stop_index exclusive */
  for( dt_index = start_index;  dt_index < stop_index;  dt_index++ )
  {
    dt = HaArray(dsg->dom_tests, dt_index);
    v1 = HaArray(sig1->states, dt_index);
    v2 = HaArray(sig2->states, dt_index);
    switch( dt->type )
    {
      case KHE_DRS_DOM_TEST_UNUSED:

	/* unused test, should never happen */
	HnAbort("internal error in KheDrsSignerDoDominates (UNUSED)");
	break;

      case KHE_DRS_DOM_TEST_SEPARATE_GENERIC:

	/* unused test, should never happen */
	HnAbort("internal error in KheDrsSignerDoDominates (SEPARATE_GENERIC)");
	break;

      case KHE_DRS_DOM_TEST_SEPARATE_INT:

	/* separate dominance (int) */
	if( !KheDrsDomTestDominatesSeparateInt(dt, v1.i, v2.i) )
	  *avail_cost -= KheCost(1, 0);
	break;

      case KHE_DRS_DOM_TEST_SEPARATE_FLOAT:

	/* separate dominance (float) */
	if( !KheDrsDomTestDominatesSeparateFloat(dt, v1.f, v2.f) )
	  *avail_cost -= KheCost(1, 0);
	break;

      case KHE_DRS_DOM_TEST_TRADEOFF:

	/* tradeoff dominance */
	if( !KheDrsDomTestDominatesTradeoff(dt, v1.i, v2.i, avail_cost) )
	  *avail_cost -= KheCost(1, 0);
	break;

      case KHE_DRS_DOM_TEST_TABULATED:

	/* tabulated dominance */
	/* Tables:  main_dom_table2 */
	KheDrsDomTestDominatesTabulated(dt, v1.i, v2.i, avail_cost);
	break;

      case KHE_DRS_DOM_TEST_CORR1_PARENT:

	/* parent of correlated parent/child pair */
	/* nothing to do here (the child does it all) */
	break;

      case KHE_DRS_DOM_TEST_CORR1_CHILD:

	/* child of correlated parent/child pair */
	/* Tables:  corr_dom_table4 */
	KheDrsSignerCorr1Dominates(dsg, sig1, sig2, dt, dt_index,
	  v1.i, v2.i, avail_cost, verbosity, indent + 4, fp);
	break;

      case KHE_DRS_DOM_TEST_CORR2_CHILD:

	/* child of correlated parent/child pair (parent not in signature) */
	/* Tables:  corr_dom_table4 */
	KheDrsSignerCorr2Dominates(dsg, sig1, sig2, dt, dt_index,
	  v1.i, v2.i, avail_cost, verbosity, indent + 4, fp);
	break;

      case KHE_DRS_DOM_TEST_CORR3_FIRST:

	/* first element of a set of two or more corr3 dom tests */
	/* Tables:  main_dom_table2 */
	KheDrsSignerCorr3Dominates(dsg, sig1, sig2, dt, dt_index,
	  v1.i, v2.i, avail_cost, verbosity, indent + 4, fp);
	break;

      case KHE_DRS_DOM_TEST_CORR3_MID:
      case KHE_DRS_DOM_TEST_CORR3_LAST:

	/* non-first element of a set of two or more corr3 dom tests */
	/* nothing to do here (the first element does it all) */
	break;

      case KHE_DRS_DOM_TEST_CORR4_FIRST:

	/* first element of a set of two or more corr4 dom tests */
	/*  Tables:  main_dom_table2 */
	KheDrsSignerCorr4Dominates(dsg, sig1, sig2, dt, dt_index,
	  v1.i, v2.i, avail_cost, verbosity, indent + 4, fp);
	break;

      case KHE_DRS_DOM_TEST_CORR4_MID:
      case KHE_DRS_DOM_TEST_CORR4_LAST:

	/* non-first element of a set of two or more corr4 dom tests */
	/* nothing to do here (the first element does it all) */
	break;

      default:

	HnAbort("KheDrsSignerDoDominates internal error 5 (%d)", dt->type);
	break;
    }

    /* print debug line, and quit early if *avail_cost is now negative */
    KheDrsDominanceDebugCmpLine(*avail_cost, dt->type, dt_index, v1.i, v2.i,
      dt->monitor, indent, fp);
    if( stop_on_neg && *avail_cost < 0 )
      return false;
  }
  return *avail_cost >= 0;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerDominates(KHE_DRS_SIGNER dsg,                           */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,                        */
/*    KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)             */
/*                                                                           */
/*  Return true if sig1 dominates sig2, including comparing their costs.     */
/*  To allow this call to be integrated with others that affect *avail_cost, */
/*  this function assumes that *avail_cost is already initialized, and it    */
/*  carries on based on that initial value.                                  */
/*                                                                           */
/*  If fp != NULL, produce debug output while doing this.                    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignerDominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)
{
  *avail_cost += (sig2->cost - sig1->cost);
  return KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_ALL,
    0, true, avail_cost, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerDebug(KHE_DRS_SIGNER dsg,                               */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of dsg onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/
static void KheDrsDomTestDebug(KHE_DRS_DOM_TEST dt, int verbosity,
  int indent, FILE *fp);

static void KheDrsSignerDebug(KHE_DRS_SIGNER dsg,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_DOM_TEST dt;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ %s Signer %s\n", indent, "",
      KheDrsSignerTypeShow(dsg->type), KheDrsSignerId(dsg));
    HaArrayForEach(dsg->dom_tests, dt, i)
      KheDrsDomTestDebug(dt, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "Signer(%s, %s)",
      KheDrsSignerTypeShow(dsg->type), KheDrsSignerId(dsg));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SIGNER_SET"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNER_SET KheDrsSignerSetMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Make a new signer set.                                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNER_SET KheDrsSignerSetMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNER_SET res;
  if( HaArrayCount(drs->signer_set_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->signer_set_free_list);
    HaArrayClear(res->signers);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->signers, drs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerSetFree(KHE_DRS_SIGNER_SET signer_set,                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free signer_set.                                                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerSetFree(KHE_DRS_SIGNER_SET signer_set,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* individual signers do not need to be freed */
  HaArrayAddLast(drs->signer_set_free_list, signer_set);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerSetAddSigner(KHE_DRS_SIGNER_SET signer_set,             */
/*    KHE_DRS_SIGNER dsg)                                                    */
/*                                                                           */
/*  Add dsg to signer_set.                                                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerSetAddSigner(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_SIGNER dsg)
{
  HaArrayAddLast(signer_set->signers, dsg);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerSetDayOpen(KHE_DRS_SIGNER_SET signer_set,               */
/*    KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                      */
/*                                                                           */
/*  Open signer_set for use on day.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerSetDayOpen(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE dr;  int i;  KHE_DRS_RESOURCE_ON_DAY drd;
  KHE_DRS_SIGNER dsg;

  /* one signer for each open resource */
  HnAssert(HaArrayCount(signer_set->signers) == 0,
    "KheDrsSignerSetDayOpen internal error");
  KheDrsResourceSetForEach(drs->open_resources, dr, i)
  {
    drd = KheDrsResourceOnDay(dr, day);
    KheDrsSignerSetAddSigner(signer_set, drd->signer);
  }

  /* one additional signer for event resource expressions */
  dsg = KheDrsSignerMake(day, NULL, NULL, NULL, drs);
  KheDrsSignerSetAddSigner(signer_set, dsg);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerSetDayClose(KHE_DRS_SIGNER_SET signer_set,              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Close signer_set.  It was opened by KheDrsSignerSetDayOpen.              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerSetDayClose(KHE_DRS_SIGNER_SET signer_set,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNER dsg;

  /* delete and free the last signer (the others were not made here) */
  HnAssert(HaArrayCount(signer_set->signers) > 0,
    "KheDrsSignerSetDayClose internal error");
  dsg = HaArrayLast(signer_set->signers);
  KheDrsSignerFree(dsg, drs);

  /* clear the signers */
  HaArrayClear(signer_set->signers);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSignerSetPartialHashSignature(KHE_DRS_SIGNER_SET signer_set,   */
/*    KHE_DRS_SIGNATURE_SET sig_set)                                         */
/*                                                                           */
/*  Hash sig_set, but only the positions that have equality dom tests.       */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSignerSetPartialHashSignature(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_SIGNATURE_SET sig_set)
{
  int res, index, i, j;  KHE_DRS_SIGNER dsg;  KHE_DRS_SIGNATURE sig;
  KHE_DRS_VALUE val;
  HnAssert(HaArrayCount(signer_set->signers)==HaArrayCount(sig_set->signatures),
    "KheDrsSignerSetPartialHashSignature internal error");
  res = 0;
  HaArrayForEach(signer_set->signers, dsg, i)
  {
    sig = HaArray(sig_set->signatures, i);
    HaArrayForEach(dsg->eq_dom_test_indexes, index, j)
    {
      val = HaArray(sig->states, index);
      res = (res + val.i) * 3;
    }
  }
  if( res < 0 )
    res = - res;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerSetPartialEqualSignature(KHE_DRS_SIGNER_SET signer_set, */
/*    KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2)        */
/*                                                                           */
/*  Return true if sig1 and sig2 are equal at the positions that have        */
/*  equality dom tests.                                                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignerSetPartialEqualSignature(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2)
{
  int i, j, index;  KHE_DRS_SIGNATURE sig1, sig2;  KHE_DRS_SIGNER dsg;

  HnAssert(HaArrayCount(signer_set->signers) ==
    HaArrayCount(sig_set1->signatures),
    "KheDrsSignerSetPartialEqualSignature internal error 1");
  HnAssert(HaArrayCount(signer_set->signers) ==
    HaArrayCount(sig_set2->signatures),
    "KheDrsSignerSetPartialEqualSignature internal error 2");
  for( i = 0;  i < HaArrayCount(signer_set->signers);  i++ )
  {
    dsg = HaArray(signer_set->signers, i);
    sig1 = HaArray(sig_set1->signatures, i);
    sig2 = HaArray(sig_set2->signatures, i);
    HnAssert(HaArrayCount(sig1->states) == HaArrayCount(sig2->states),
     "KheDrsSignerPartialEqualSignature: signatures have different lengths");
    HaArrayForEach(dsg->eq_dom_test_indexes, index, j)
      if( HaArray(sig1->states, index).i != HaArray(sig2->states, index).i )
	return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerSetDominates(KHE_DRS_SIGNER_SET signer_set,             */
/*    KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2,        */
/*    KHE_COST trie_extra_cost, int trie_start_depth, bool use_caching,      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Return true if sig_set1 dominates sig_set2, including comparing their    */
/*  costs.  If fp != NULL, produce debug output while doing this.            */
/*                                                                           */
/*  When this function is called from the trie data structure, some of its   */
/*  work may already be done.  It starts comparing from the states at        */
/*  position trie_start_depth (counting across any number of signatures).    */
/*                                                                           */
/*  If trie_start_depth is 0 (as it usually will be) and use_caching is      */
/*  true, then dominance test caching is used to speed up the test.          */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignerSetDominates(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2,
  KHE_COST trie_extra_cost, int trie_start_depth, bool use_caching,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_COST *avail_cost,
  int verbosity, int indent, FILE *fp)
{
  int i, count, last_cache;  KHE_DRS_SIGNER dsg;
  KHE_DRS_SIGNATURE sig1, sig2;  KHE_DRS_RESOURCE dr;  KHE_DRS_COST_TUPLE ct;
  count = HaArrayCount(signer_set->signers);
  HnAssert(HaArrayCount(sig_set1->signatures) == count,
    "KheDrsSignerSetDominates internal error 1");
  HnAssert(HaArrayCount(sig_set2->signatures) == count,
    "KheDrsSignerSetDominates internal error 2");
  *avail_cost = sig_set2->cost - sig_set1->cost - trie_extra_cost;
  if( *avail_cost < 0 )
    return false;
  if( drs->solve_dom_approx > 0 )
    *avail_cost += (*avail_cost * drs->solve_dom_approx) / 10;
  if( trie_start_depth > 0 )
  {
    /* won't happen anyway but do it the easy way if it does */
    HaArrayForEach(signer_set->signers, dsg, i)
    {
      if( trie_start_depth < HaArrayCount(dsg->dom_tests) )
      {
	if( fp != NULL && i > 0 )
	  KheDrsDominanceDebugSpacer(indent, fp);
	sig1 = HaArray(sig_set1->signatures, i);
	sig2 = HaArray(sig_set2->signatures, i);
	if( !KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_ALL,
	      trie_start_depth, true, avail_cost, verbosity,indent,fp) )
	  return false;
	trie_start_depth = 0;
      }
      else
	trie_start_depth -= HaArrayCount(dsg->dom_tests);
    }
  }
  else if( USE_DOM_CACHING && use_caching )
  {
    /* used a cached value for all positions except the last */
    HnAbort("KheDrsSignerSetDominates - dom caching unavailable");
    last_cache = HaArrayCount(signer_set->signers) - 2;
    for( i = 0;  i <= last_cache;  i++ )
    {
      if( fp != NULL && i > 0 )
	KheDrsDominanceDebugSpacer(indent, fp);
      dr = KheDrsResourceSetResource(drs->open_resources, i);
      dsg = HaArray(signer_set->signers, i);
      sig1 = HaArray(sig_set1->signatures, i);
      HnAssert(sig1->asst_to_shift_index >= 0,
	"KheDrsSignerSetDominates internal error 3");
      sig2 = HaArray(sig_set2->signatures, i);
      HnAssert(sig2->asst_to_shift_index >= 0,
	"KheDrsSignerSetDominates internal error 4");
      ct = KheDrsDim2TableGet2(dr->expand_dom_test_cache,
	sig1->asst_to_shift_index, sig2->asst_to_shift_index);
      *avail_cost += ct.unweighted_psi;
      if( fp != NULL )
	fprintf(fp, "%*s%8.5f  (cache[%d, %d] = %d)\n", indent, "",
	  KheCostShow(*avail_cost), sig1->asst_to_shift_index,
	  sig2->asst_to_shift_index, ct.unweighted_psi);
      if( *avail_cost < 0 )
	return false;
    }

    /* regular test for the last position */
    if( fp != NULL && i > 0 )
      KheDrsDominanceDebugSpacer(indent, fp);
    dsg = HaArrayLast(signer_set->signers);
    sig1 = HaArrayLast(sig_set1->signatures);
    sig2 = HaArrayLast(sig_set2->signatures);
    if( !KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_ALL,
	  0, true, avail_cost, verbosity, indent, fp) )
      return false;
  }
  else
  {
    /* visit hard constraints first; they often end the test quickly */
    HaArrayForEach(signer_set->signers, dsg, i)
    {
      sig1 = HaArray(sig_set1->signatures, i);
      sig2 = HaArray(sig_set2->signatures, i);
      if( !KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_HARD,
	    0, true, avail_cost, verbosity, indent, fp) )
	return false;
    }

    /* visit soft constraints */
    HaArrayForEach(signer_set->signers, dsg, i)
    {
      sig1 = HaArray(sig_set1->signatures, i);
      sig2 = HaArray(sig_set2->signatures, i);
      if( !KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_SOFT,
	  0, true, avail_cost, verbosity, indent, fp) )
	return false;
    }
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerSetDebug(KHE_DRS_SIGNER_SET signer_set,                 */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of signer_set onto fp with the given verbosity and indent.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerSetDebug(KHE_DRS_SIGNER_SET signer_set,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SIGNER dsg;  int i;
  fprintf(fp, "%*s[ SignerSet\n", indent, "");
  HaArrayForEach(signer_set->signers, dsg, i)
  {
    if( i > 0 )
      KheDrsDominanceDebugSpacer(indent + 2, fp);
    KheDrsSignerDebug(dsg, verbosity, indent + 2, fp);
  }
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_KIND"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE KheDomKindToDomTestType(KHE_DRS_DOM_KIND dom_kind) */
/*                                                                           */
/*  Return the dom test type of dom_kind.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST_TYPE KheDomKindToDomTestType(KHE_DRS_DOM_KIND dom_kind)
{
  switch( dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:		return KHE_DRS_DOM_TEST_UNUSED;
    case KHE_DRS_DOM_LIST_SEPARATE:    return KHE_DRS_DOM_TEST_SEPARATE_GENERIC;
    case KHE_DRS_DOM_LIST_TRADEOFF:	return KHE_DRS_DOM_TEST_TRADEOFF;
    case KHE_DRS_DOM_LIST_TABULATED:	return KHE_DRS_DOM_TEST_TABULATED;
    case KHE_DRS_DOM_HASH_EQUALITY:	return KHE_DRS_DOM_TEST_UNUSED;
    case KHE_DRS_DOM_HASH_MEDIUM:	return KHE_DRS_DOM_TEST_SEPARATE_INT;
    /* ***
    case KHE_DRS_DOM_TRIE_SEPARATE:	return KHE_DRS_DOM_TEST_SEPARATE_INT;
    case KHE_DRS_DOM_TRIE_TRADEOFF:	return KHE_DRS_DOM_TEST_TRADEOFF;
    *** */
    case KHE_DRS_DOM_INDEXED_TRADEOFF:	return KHE_DRS_DOM_TEST_TRADEOFF;
    case KHE_DRS_DOM_INDEXED_TABULATED:	return KHE_DRS_DOM_TEST_TABULATED;

    default:

      HnAbort("KheDomKindToDomTestType: unknown dom_kind (%d)", dom_kind);
      return 0;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE KheDomKindCheckConsistency(                        */
/*    KHE_DRS_DOM_KIND main_dom_kind, bool cache,                            */
/*    KHE_DRS_DOM_KIND cache_dom_kind)                                       */
/*                                                                           */
/*  Check that main_dom_kind, cache, and cache_dom_kind are consistent,      */
/*  and return the consistent dom test type.                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST_TYPE KheDomKindCheckConsistency(
  KHE_DRS_DOM_KIND main_dom_kind, bool cache, KHE_DRS_DOM_KIND cache_dom_kind)
{
  KHE_DRS_DOM_TEST_TYPE main_dom_test_type, cache_dom_test_type;
  main_dom_test_type = KheDomKindToDomTestType(main_dom_kind);
  if( cache )
  {
    cache_dom_test_type = KheDomKindToDomTestType(cache_dom_kind);
    HnAssert(main_dom_test_type == KHE_DRS_DOM_TEST_UNUSED ||
      cache_dom_test_type == KHE_DRS_DOM_TEST_UNUSED ||
      main_dom_test_type == cache_dom_test_type, "KheDomKindCheckConsistency:"
      "inconsistent main_dom_kind and cache_dom_kind arguments");
  }
  return main_dom_test_type;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDomKindShow(KHE_DRS_DOM_KIND dom_kind)                          */
/*                                                                           */
/*  Return a suitable label for dom_kind.                                    */
/*                                                                           */
/*****************************************************************************/

static char *KheDomKindShow(KHE_DRS_DOM_KIND dom_kind)
{
  switch( dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:		return "ListNone";
    case KHE_DRS_DOM_LIST_SEPARATE:	return "ListSeparate";
    case KHE_DRS_DOM_LIST_TRADEOFF:	return "ListTradeoff";
    case KHE_DRS_DOM_LIST_TABULATED:	return "ListTabulated";
    case KHE_DRS_DOM_HASH_EQUALITY:	return "HashEquality";
    case KHE_DRS_DOM_HASH_MEDIUM:	return "HashMedium";
    /* ***
    case KHE_DRS_DOM_TRIE_SEPARATE:	return "TrieSeparate";
    case KHE_DRS_DOM_TRIE_TRADEOFF:	return "TrieTradeoff";
    *** */
    case KHE_DRS_DOM_INDEXED_TRADEOFF:	return "IndexedTradeoff";
    case KHE_DRS_DOM_INDEXED_TABULATED:	return "IndexedTabulated";

    default:

      HnAbort("KheDomKindShow: unknown dom_kind (%d)", dom_kind);
      return NULL;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_COST_TUPLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsCostTupleMake(KHE_COST psi, KHE_COST psi0,      */
/*    bool psi_plus_defined, KHE_COST psi_plus)                              */
/*                                                                           */
/*  Make an unweighted cost tuple object with these attributes.              */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_COST_TUPLE KheDrsCostTupleMake(short unweighted_psi,
  short unweighted_psi0, /* bool psi_plus_defined, */ short unweighted_psi_plus)
{
  KHE_DRS_COST_TUPLE res;
  res.unweighted_psi = unweighted_psi;
  res.unweighted_psi0 = unweighted_psi0;
  /* res.psi_plus_defined = psi_plus_defined; */
  res.unweighted_psi_plus = unweighted_psi_plus;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCostTupleDebug(KHE_DRS_COST_TUPLE ct, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of ct onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCostTupleDebug(KHE_DRS_COST_TUPLE ct, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "(%d, %d, %d)", ct.unweighted_psi, ct.unweighted_psi0,
    ct.unweighted_psi_plus);
  /* ***
  if( ct.psi_plus_defined )
    fprintf(fp, "(%.5f, %.5f, %.5f)", KheCostShow(ct.psi),
      KheCostShow(ct.psi0), KheCostShow(ct.psi_plus));
  else
    fprintf(fp, "(%.5f, %.5f)", KheCostShow(ct.psi), KheCostShow(ct.psi0));
  *** */
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM1_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM1_TABLE KheDrsDim1TableMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Make a new, empty dim1 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM1_TABLE KheDrsDim1TableMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM1_TABLE res;
  if( HaArrayCount(drs->table1_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->table1_free_list);
    HaArrayClear(res->children);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->children, drs->arena);
  }
  res->offset = 0;  /* actually undefined when the child array is empty */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim1TablePut(KHE_DRS_DIM1_TABLE d1, int index,                */
/*    KHE_DRS_COST_TUPLE val)                                                */
/*                                                                           */
/*  Add val to d1 at index.  If this is the first child, set the offset.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim1TablePut(KHE_DRS_DIM1_TABLE d1, int index,
  KHE_DRS_COST_TUPLE val)
{
  if( HaArrayCount(d1->children) == 0 )
  {
    /* first child, set offset */
    d1->offset = index;
  }
  else
  {
    /* not first child, must be next though */
    HnAssert(index == d1->offset + HaArrayCount(d1->children),
      "KheDrsDim1TablePut: index is %d when %d expected", index,
      d1->offset + HaArrayCount(d1->children));
  }
  HaArrayAddLast(d1->children, val);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDim1TableGetCheck(KHE_DRS_DIM1_TABLE d1, int index)           */
/*                                                                           */
/*  Return true if a call on KheDrsDim1TableGet(d1, index) would succeed.    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDim1TableGetCheck(KHE_DRS_DIM1_TABLE d1, int index)
{
  int pos;
  pos = index - d1->offset;
  return 0 <= pos && pos < HaArrayCount(d1->children);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsDim1TableGet(KHE_DRS_DIM1_TABLE d1, int index)  */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_COST_TUPLE KheDrsDim1TableGet(KHE_DRS_DIM1_TABLE d1, int index)
{
  int pos;
  pos = index - d1->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d1->children),
    "KheDrsDim1TableGet: index %d out of range %d .. %d", index,
    d1->offset, HaArrayCount(d1->children) + d1->offset - 1);
  return HaArray(d1->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim1TableFree(KHE_DRS_DIM1_TABLE d1,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free d1.                                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim1TableFree(KHE_DRS_DIM1_TABLE d1,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->table1_free_list, d1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim1TableDebug(KHE_DRS_DIM1_TABLE d1, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d1 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim1TableDebug(KHE_DRS_DIM1_TABLE d1, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_COST_TUPLE ct;  int i;
  fprintf(fp, "%*s[ Dim1 (%d .. %d)\n", indent, "", d1->offset,
    d1->offset + HaArrayCount(d1->children) - 1);
  HaArrayForEach(d1->children, ct, i)
  {
    if( i == 0 )
      fprintf(fp, "%*s", indent + 2, "");
    else if( i % 4 == 0 )
      fprintf(fp, "\n%*s", indent + 2, "");
    else
      fprintf(fp, ", ");
    KheDrsCostTupleDebug(ct, verbosity, -1, fp);
  }
  fprintf(fp, "\n%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM2_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM2_TABLE KheDrsDim2TableMake(char *debug_str, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty dim2 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM2_TABLE KheDrsDim2TableMake(/* char *debug_str, */ HA_ARENA a)
{
  KHE_DRS_DIM2_TABLE res;
  HaMake(res, a);
  HaArrayInit(res->children, a);
  res->offset = 0;  /* actually undefined when the child array is empty */
  /* res->debug_str = debug_str; */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim2TablePut(KHE_DRS_DIM2_TABLE d2, int index,                */
/*    KHE_DRS_DIM1_TABLE val)                                                */
/*                                                                           */
/*  Add val to d2 at index.  If this is the first child, set the offset.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim2TablePut(KHE_DRS_DIM2_TABLE d2, int index2,
  int index1, KHE_DRS_COST_TUPLE ct, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM1_TABLE d1;  int pos;

  /* set offset if this is the first insertion */
  if( HaArrayCount(d2->children) == 0 )
    d2->offset = index2;

  /* make sure index2 is within, or one step beyond, the current range */
  pos = index2 - d2->offset;
  HnAssert(0 <= pos && pos <= HaArrayCount(d2->children),
    "KheDrsDim2TablePut: index2 %d is out of range %d .. %d", index2,
    d2->offset, d2->offset + HaArrayCount(d2->children));

  /* find or make-and-add d1, the sub-array to insert into */
  if( pos < HaArrayCount(d2->children) )
    d1 = HaArray(d2->children, pos);
  else
  {
    d1 = KheDrsDim1TableMake(drs);
    HaArrayAddLast(d2->children, d1);
  }
  
  /* do the insertion */
  KheDrsDim1TablePut(d1, index1, ct);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM1_TABLE KheDrsDim2TableGet(KHE_DRS_DIM2_TABLE d2, int index2) */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM1_TABLE KheDrsDim2TableGet(KHE_DRS_DIM2_TABLE d2, int index2)
{
  int pos;
  pos = index2 - d2->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d2->children),
    "KheDrsDim2TableGet: index2 %d out of range %d .. %d", index2,
    d2->offset, HaArrayCount(d2->children) + d2->offset - 1);
  return HaArray(d2->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDim2TableGet2Check(KHE_DRS_DIM2_TABLE d2,                     */
/*    int index2, int index1)                                                */
/*                                                                           */
/*  Return true if a call on KheDrsDim2TableGet2(d2, index2, index1)         */
/*  would succeed.                                                           */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDim2TableGet2Check(KHE_DRS_DIM2_TABLE d2,
  int index2, int index1)
{
  int pos;  KHE_DRS_DIM1_TABLE d1;
  pos = index2 - d2->offset;
  if( 0 <= pos && pos < HaArrayCount(d2->children) )
  {
    d1 = HaArray(d2->children, pos);
    return KheDrsDim1TableGetCheck(d1, index1);
  }
  else
    return false;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsDim2TableGet2(KHE_DRS_DIM2_TABLE d2,            */
/*    int index2, int index1)                                                */
/*                                                                           */
/*  Get the cost tuple at d2[index2, index1].                                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_COST_TUPLE KheDrsDim2TableGet2(KHE_DRS_DIM2_TABLE d2,
  int index2, int index1)
{
  return KheDrsDim1TableGet(KheDrsDim2TableGet(d2, index2), index1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim2TableClear(KHE_DRS_DIM2_TABLE d2,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Clear d2, placing all its d1's onto a free list.                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim2TableClear(KHE_DRS_DIM2_TABLE d2,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM1_TABLE d1;  int i;
  HaArrayForEach(d2->children, d1, i)
    KheDrsDim1TableFree(d1, drs);
  HaArrayClear(d2->children);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim2TableDebug(KHE_DRS_DIM2_TABLE d2, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d2 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim2TableDebug(KHE_DRS_DIM2_TABLE d2, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_DIM1_TABLE d1;  int i;
  fprintf(fp, "%*s[ Dim2 (%d .. %d)\n", indent, "",
    /* d2->debug_str == NULL ? "" : d2->debug_str, */
    d2->offset, d2->offset + HaArrayCount(d2->children) - 1);
  HaArrayForEach(d2->children, d1, i)
    KheDrsDim1TableDebug(d1, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM3_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM3_TABLE KheDrsDim3TableMake(char *debug_str, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty dim3 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM3_TABLE KheDrsDim3TableMake(/*char *debug_str,*/ HA_ARENA a)
{
  KHE_DRS_DIM3_TABLE res;
  HaMake(res, a);
  HaArrayInit(res->children, a);
  res->offset = 0;  /* actually undefined when the child array is empty */
  /* res->debug_str = debug_str; */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim3TablePut(KHE_DRS_DIM3_TABLE d3, int index3, int index2,   */
/*    int index1, KHE_DRS_COST_TUPLE ct, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Add c3 to d3.  If this is the first insertion, set the offset.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim3TablePut(KHE_DRS_DIM3_TABLE d3, int index3, int index2,
  int index1, KHE_DRS_COST_TUPLE ct, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM2_TABLE d2;  int pos;  HA_ARENA a;  /* char *debug_str; */

  /* set offset if this is the first insertion */
  if( HaArrayCount(d3->children) == 0 )
    d3->offset = index3;

  /* make sure index3 is within, or one step beyond, the current range */
  pos = index3 - d3->offset;
  HnAssert(0 <= pos && pos <= HaArrayCount(d3->children),
    "KheDrsDim3TablePut: index3 %d is out of range %d .. %d", index3,
    d3->offset, d3->offset + HaArrayCount(d3->children));

  /* find or make-and-add d2, the sub-array to insert into */
  if( pos < HaArrayCount(d3->children) )
    d2 = HaArray(d3->children, pos);
  else
  {
    a = HaArrayArena(d3->children);
    /* debug_str = HnStringMake(a, "%s:%d", d3->debug_str, index3); */
    d2 = KheDrsDim2TableMake(/* debug_str, */ a);
    HaArrayAddLast(d3->children, d2);
  }
  
  /* do the insertion */
  KheDrsDim2TablePut(d2, index2, index1, ct, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM2_TABLE KheDrsDim3TableGet(KHE_DRS_DIM3_TABLE d3, int index3) */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM2_TABLE KheDrsDim3TableGet(KHE_DRS_DIM3_TABLE d3, int index3)
{
  int pos;
  pos = index3 - d3->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d3->children),
    "KheDrsDim3TableGet: index %d out of range %d .. %d", index3,
    d3->offset, HaArrayCount(d3->children) + d3->offset - 1);
  return HaArray(d3->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDim3TableGetCheck(KHE_DRS_DIM3_TABLE d3, int index3)          */
/*                                                                           */
/*  Return true if a call to KheDrsDim3TableGet with these arguments         */
/*  would succeed.                                                           */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDim3TableGetCheck(KHE_DRS_DIM3_TABLE d3, int index3)
{
  int pos;
  pos = index3 - d3->offset;
  return 0 <= pos && pos < HaArrayCount(d3->children);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim3TableDebug(KHE_DRS_DIM3_TABLE d3, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d3 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim3TableDebug(KHE_DRS_DIM3_TABLE d3, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_DIM2_TABLE d2;  int i;
  fprintf(fp, "%*s[ Dim3 (%d .. %d)\n", indent, "",
    /* d3->debug_str == NULL ? "" : d3->debug_str, */
    d3->offset, d3->offset + HaArrayCount(d3->children) - 1);
  HaArrayForEach(d3->children, d2, i)
    KheDrsDim2TableDebug(d2, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM4_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM4_TABLE KheDrsDim4TableMake(char *debug_str, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty Dim4 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM4_TABLE KheDrsDim4TableMake(/*char *debug_str,*/ HA_ARENA a)
{
  KHE_DRS_DIM4_TABLE res;
  HaMake(res, a);
  HaArrayInit(res->children, a);
  res->offset = 0;  /* actually undefined when the child array is empty */
  /* res->debug_str = debug_str; */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim4TablePut(KHE_DRS_DIM4_TABLE d4, int index4, int index3,   */
/*    int index2, int index1, KHE_DRS_COST_TUPLE ct,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add ct to d4.  If this is the first insertion, set the offset.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim4TablePut(KHE_DRS_DIM4_TABLE d4, int index4, int index3,
  int index2, int index1, KHE_DRS_COST_TUPLE ct,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM3_TABLE d3;  int pos;  HA_ARENA a;  /* char *debug_str; */

  /* set offset if this is the first insertion */
  if( HaArrayCount(d4->children) == 0 )
    d4->offset = index4;

  /* make sure index4 is within, or one step beyond, the current range */
  pos = index4 - d4->offset;
  HnAssert(0 <= pos && pos <= HaArrayCount(d4->children),
    "KheDrsDim4TablePut: index4 %d is out of range %d .. %d", index4,
    d4->offset, d4->offset + HaArrayCount(d4->children));

  /* find or make-and-add d3, the sub-array to insert into */
  if( pos < HaArrayCount(d4->children) )
    d3 = HaArray(d4->children, pos);
  else
  {
    a = HaArrayArena(d4->children);
    /* debug_str = HnStringMake(a, "%s:%d", d4->debug_str, index4); */
    d3 = KheDrsDim3TableMake(/* debug_str, */ a);
    HaArrayAddLast(d4->children, d3);
  }
  
  /* do the insertion */
  KheDrsDim3TablePut(d3, index3, index2, index1, ct, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM3_TABLE KheDrsDim4TableGet(KHE_DRS_DIM4_TABLE d4, int index4) */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM3_TABLE KheDrsDim4TableGet(KHE_DRS_DIM4_TABLE d4, int index4)
{
  int pos;
  pos = index4 - d4->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d4->children),
    "KheDrsDim4TableGet: index %d out of range %d .. %d", index4,
    d4->offset, HaArrayCount(d4->children) + d4->offset - 1);
  return HaArray(d4->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsDim4TableGet4(KHE_DRS_DIM4_TABLE d4,            */
/*    int index4, int index3, int index2, int index1)                        */
/*                                                                           */
/*  Get the cost tuple at d4[index4, index3, index2, index1].                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_COST_TUPLE KheDrsDim4TableGet4(KHE_DRS_DIM4_TABLE d4,
  int index4, int index3, int index2, int index1)
{
  return KheDrsDim2TableGet2(KheDrsDim3TableGet(KheDrsDim4TableGet(d4, index4),
    index3), index2, index1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim4TableDebug(KHE_DRS_DIM4_TABLE d4, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d4 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim4TableDebug(KHE_DRS_DIM4_TABLE d4, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_DIM3_TABLE d3;  int i;
  fprintf(fp, "%*s[ Dim4 (%d .. %d)\n", indent, "",
    /* d4->debug_str == NULL ? "" : d4->debug_str, */
    d4->offset, d4->offset + HaArrayCount(d4->children) - 1);
  HaArrayForEach(d4->children, d3, i)
    KheDrsDim3TableDebug(d3, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM5_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM5_TABLE KheDrsDim5TableMake(char *debug_str, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty Dim5 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM5_TABLE KheDrsDim5TableMake(/*char *debug_str,*/ HA_ARENA a)
{
  KHE_DRS_DIM5_TABLE res;
  HaMake(res, a);
  HaArrayInit(res->children, a);
  res->offset = 0;  /* actually undefined when the child array is empty */
  /* res->debug_str = debug_str; */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim5TablePut(KHE_DRS_DIM5_TABLE d5, int index5, int index4,   */
/*    int index3, int index2, int index1, KHE_DRS_COST_TUPLE ct,             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add ct to d5.  If this is the first insertion, set the offset.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim5TablePut(KHE_DRS_DIM5_TABLE d5, int index5, int index4,
  int index3, int index2, int index1, KHE_DRS_COST_TUPLE ct,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM4_TABLE d4;  int pos;  /* char *debug_str; */  HA_ARENA a;

  /* set offset if this is the first insertion */
  if( HaArrayCount(d5->children) == 0 )
    d5->offset = index5;

  /* make sure index5 is within, or one step beyond, the current range */
  pos = index5 - d5->offset;
  HnAssert(0 <= pos && pos <= HaArrayCount(d5->children),
    "KheDrsDim5TablePut: index5 %d is out of range %d .. %d", index5,
    d5->offset, d5->offset + HaArrayCount(d5->children));

  /* find or make-and-add d4, the sub-array to insert into */
  if( pos < HaArrayCount(d5->children) )
    d4 = HaArray(d5->children, pos);
  else
  {
    a = HaArrayArena(d5->children);
    /* debug_str = HnStringMake(a, "%s:%d", d5->debug_str, index5); */
    d4 = KheDrsDim4TableMake(/* debug_str, */ a);
    HaArrayAddLast(d5->children, d4);
  }
  
  /* do the insertion */
  KheDrsDim4TablePut(d4, index4, index3, index2, index1, ct, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM4_TABLE KheDrsDim5TableGet(KHE_DRS_DIM5_TABLE d5, int index5) */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM4_TABLE KheDrsDim5TableGet(KHE_DRS_DIM5_TABLE d5, int index5)
{
  int pos;
  pos = index5 - d5->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d5->children),
    "KheDrsDim5TableGet: index %d out of range %d .. %d", index5,
    d5->offset, HaArrayCount(d5->children) + d5->offset - 1);
  return HaArray(d5->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM2_TABLE KheDrsDim5TableGet3(KHE_DRS_DIM5_TABLE d5,            */
/*    int index5, int index4, int index3)                                    */
/*                                                                           */
/*  Get the two-dimensional table that results from applying three indexes   */
/*  to five-dimensional table d5.                                            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM2_TABLE KheDrsDim5TableGet3(KHE_DRS_DIM5_TABLE d5,
  int index5, int index4, int index3)
{
  return KheDrsDim3TableGet(KheDrsDim4TableGet(KheDrsDim5TableGet(d5, index5),
    index4), index3);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim5TableDebug(KHE_DRS_DIM5_TABLE d5, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d5 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim5TableDebug(KHE_DRS_DIM5_TABLE d5, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_DIM4_TABLE d4;  int i;
  fprintf(fp, "%*s[ Dim5 (%d .. %d)\n", indent, "",
    /* d5->debug_str == NULL ? "" : d5->debug_str, */
    d5->offset, d5->offset + HaArrayCount(d5->children) - 1);
  HaArrayForEach(d5->children, d4, i)
    KheDrsDim4TableDebug(d4, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_TEST_TYPE"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsDomTestTypeShow(KHE_DRS_DOM_TEST_TYPE type)                  */
/*                                                                           */
/*  Return a two-character string representation of type.                    */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsDomTestTypeShow(KHE_DRS_DOM_TEST_TYPE type)
{
  switch( type )
  {
    case KHE_DRS_DOM_TEST_UNUSED:		return "!!";
    case KHE_DRS_DOM_TEST_SEPARATE_GENERIC:	return "SG";
    case KHE_DRS_DOM_TEST_SEPARATE_INT:		return "SI";
    case KHE_DRS_DOM_TEST_SEPARATE_FLOAT:	return "SF";
    case KHE_DRS_DOM_TEST_TRADEOFF:		return "TR";
    case KHE_DRS_DOM_TEST_TABULATED:		return "TB";
    case KHE_DRS_DOM_TEST_CORR1_PARENT:		return "1P";
    case KHE_DRS_DOM_TEST_CORR1_CHILD:		return "1C";
    case KHE_DRS_DOM_TEST_CORR2_CHILD:		return "2C";
    case KHE_DRS_DOM_TEST_CORR3_FIRST:		return "3F";
    case KHE_DRS_DOM_TEST_CORR3_MID:		return "3M";
    case KHE_DRS_DOM_TEST_CORR3_LAST:		return "3L";
    case KHE_DRS_DOM_TEST_CORR4_FIRST:		return "4F";
    case KHE_DRS_DOM_TEST_CORR4_MID:		return "4M";
    case KHE_DRS_DOM_TEST_CORR4_LAST:		return "4L";
    default:					return "??";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE KheDomTestTypeActual(KHE_DRS_DOM_TEST_TYPE dt,     */
/*    bool tradeoff_allowed, bool float_vals)                                */
/*                                                                           */
/*  Return the dom test type to actually use, given than the requested       */
/*  type is dt, tradoff_allowed says whether tradeoff (and tabulated)        */
/*  dominance is allowed, and float_vals says whether the values to          */
/*  compare are floats.                                                      */
/*                                                                           */
/*  It is inconsistent for tradeoff_allowed to be true and float_vals to     */
/*  be true; at least one of them must be false.                             */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST_TYPE KheDomTestTypeActual(KHE_DRS_DOM_TEST_TYPE dt,
  bool tradeoff_allowed, bool float_vals)
{
  switch( dt )
  {
    case KHE_DRS_DOM_TEST_UNUSED:

      HnAbort("KheDomTestTypeActual unexpected dom type (%s)",
        KheDrsDomTestTypeShow(dt));
      break;

    case KHE_DRS_DOM_TEST_SEPARATE_GENERIC:

      dt = float_vals ? KHE_DRS_DOM_TEST_SEPARATE_FLOAT :
	KHE_DRS_DOM_TEST_SEPARATE_INT;
      break;

    case KHE_DRS_DOM_TEST_SEPARATE_INT:

      /* OK as is, just check */
      HnAssert(!float_vals, "KheDomTestTypeActual internal error 1");
      break;

    case KHE_DRS_DOM_TEST_SEPARATE_FLOAT:

      /* OK as is, just check */
      HnAssert(float_vals, "KheDomTestTypeActual internal error 2");
      break;

    case KHE_DRS_DOM_TEST_TRADEOFF:
    case KHE_DRS_DOM_TEST_TABULATED:

      if( !tradeoff_allowed )
	dt = float_vals ? KHE_DRS_DOM_TEST_SEPARATE_FLOAT :
	  KHE_DRS_DOM_TEST_SEPARATE_INT;
      break;

    case KHE_DRS_DOM_TEST_CORR1_PARENT:
    case KHE_DRS_DOM_TEST_CORR1_CHILD:
    case KHE_DRS_DOM_TEST_CORR2_CHILD:
    case KHE_DRS_DOM_TEST_CORR3_FIRST:
    case KHE_DRS_DOM_TEST_CORR3_MID:
    case KHE_DRS_DOM_TEST_CORR3_LAST:
    case KHE_DRS_DOM_TEST_CORR4_FIRST:
    case KHE_DRS_DOM_TEST_CORR4_MID:
    case KHE_DRS_DOM_TEST_CORR4_LAST:

      HnAbort("KheDomTestTypeActual unexpected dom type (%s)",
        KheDrsDomTestTypeShow(dt));
      break;

    default:

      HnAbort("KheDomTestTypeActual unknown dom type (%d)", dt);
      break;
  }
  return dt;

  /* ***
  if( dt == KHE_DRS_DOM_TEST_TRADEOFF && !tradeoff_allowed )
    return KHE_DRS_DOM_TEST_SEPARATE_INT;
  else
  {
    if( dt != KHE_DRS_DOM_TEST_SEPARATE_GENERIC )
      return dt;
    else if( float_vals )
      return KHE_DRS_DOM_TEST_SEPARATE_FLOAT;
    else
     return KHE_DRS_DOM_TEST_SEPARATE_INT;
  }
  *** */
}




/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_TEST"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsDomTestMake(KHE_DRS_DOM_TEST_TYPE type,           */
/*    KHE_DRS_EXPR e, bool allow_zero, KHE_DRS_VALUE a, KHE_DRS_VALUE b,     */
/*    KHE_COST combined_weight, KHE_DRS_DIM2_TABLE main_dom_table2,          */
/*    KHE_DRS_DIM4_TABLE corr_dom_table4, KHE_DRS_MONITOR m,                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make and return a new int dominance test object with these attributes.   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsDomTestMake(KHE_DRS_DOM_TEST_TYPE type,
  KHE_DRS_EXPR e, bool allow_zero, KHE_DRS_VALUE a, KHE_DRS_VALUE b,
  KHE_COST combined_weight, KHE_DRS_DIM2_TABLE main_dom_table2,
  KHE_DRS_DIM4_TABLE corr_dom_table4, KHE_DRS_MONITOR m,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST res;
  if( HaArrayCount(drs->dom_test_free_list) > 0 )
    res = HaArrayLastAndDelete(drs->dom_test_free_list);
  else
    HaMake(res, drs->arena);
  res->type = type;
  res->correlated_delta = 0;
  res->expr = e;
  res->allow_zero = allow_zero;
  res->a = a;
  res->b = b;
  res->combined_weight = combined_weight;
  res->main_dom_table2 = main_dom_table2;
  res->corr_dom_table4 = corr_dom_table4;
  res->monitor = m;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsDomTestMakeInt(KHE_DRS_DOM_TEST_TYPE type,        */
/*    KHE_DRS_EXPR e, bool allow_zero, int a, int b,                         */
/*    KHE_COST combined_weight, KHE_DRS_DIM2_TABLE main_dom_table2,          */
/*    KHE_DRS_DIM4_TABLE corr_dom_table4, KHE_MONITOR m,                     */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make and return a new int dominance test object with these attributes.   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsDomTestMakeInt(KHE_DRS_DOM_TEST_TYPE type,
  KHE_DRS_EXPR e, bool allow_zero, int a, int b,
  KHE_COST combined_weight, KHE_DRS_DIM2_TABLE main_dom_table2,
  KHE_DRS_DIM4_TABLE corr_dom_table4, KHE_DRS_MONITOR m,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_VALUE aval, bval;
  aval.i = a;
  bval.i = b;
  return KheDrsDomTestMake(type, e, allow_zero, aval, bval,
    combined_weight, main_dom_table2, corr_dom_table4, m, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsDomTestMakeFloat(KHE_DRS_DOM_TEST_TYPE type,      */
/*    KHE_DRS_EXPR e, bool allow_zero, float a, float b,                     */
/*    KHE_COST combined_weight, KHE_DRS_DIM2_TABLE main_dom_table2,          */
/*    KHE_DRS_DIM4_TABLE corr_dom_table4, KHE_MONITOR m,                     */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make and return a new int dominance test object with these attributes.   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsDomTestMakeFloat(KHE_DRS_DOM_TEST_TYPE type,
  KHE_DRS_EXPR e, bool allow_zero, float a, float b,
  KHE_COST combined_weight, KHE_DRS_DIM2_TABLE main_dom_table2,
  KHE_DRS_DIM4_TABLE corr_dom_table4, KHE_DRS_MONITOR m,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_VALUE aval, bval;
  aval.f = a;
  bval.f = b;
  return KheDrsDomTestMake(type, e, allow_zero, aval, bval,
    combined_weight, main_dom_table2, corr_dom_table4, m, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestDominatesSeparateInt(KHE_DRS_DOM_TEST dt,              */
/*    int val1, int val2)                                                    */
/*                                                                           */
/*  Return true if val1 dominates val2, according to dt (int version).       */
/*                                                                           */
/*  Implementation note.  This formula is derived in the Appendix to the     */
/*  User's Guide, where it is called dom(val1, val2).                        */
/*                                                                           */
/*****************************************************************************/

static bool TildeInt(KHE_DRS_DOM_TEST dt, int val1, int val2)
{
  return (dt->allow_zero ? (val1 == 0) == (val2 == 0) : true);
}

static bool KheDrsDomMaxInt(KHE_DRS_DOM_TEST dt, int val1, int val2)
{
  return val1 <= dt->a.i || val1 <= val2;
}

static bool KheDrsDomMinInt(KHE_DRS_DOM_TEST dt, int val1, int val2)
{
  return val1 >= dt->b.i || (TildeInt(dt, val1, val2) && val1 >= val2);
}

static bool KheDrsDomTestDominatesSeparateInt(KHE_DRS_DOM_TEST dt,
  int val1, int val2)
{
  return KheDrsDomMaxInt(dt, val1, val2) && KheDrsDomMinInt(dt, val1, val2);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestDominatesSeparateFloat(KHE_DRS_DOM_TEST dt,            */
/*    float val1, float val2)                                                */
/*                                                                           */
/*  Return true if val1 dominates val2, according to dt (float version).     */
/*                                                                           */
/*  Implementation note.  This formula is derived in the Appendix to the     */
/*  User's Guide, where it is called dom(val1, val2).                        */
/*                                                                           */
/*****************************************************************************/

static bool TildeFloat(KHE_DRS_DOM_TEST dt, float val1, float val2)
{
  return (dt->allow_zero ? (val1 == 0) == (val2 == 0) : true);
}

static bool KheDrsDomMaxFloat(KHE_DRS_DOM_TEST dt, float val1, float val2)
{
  return val1 <= dt->a.f || val1 <= val2;
}

static bool KheDrsDomMinFloat(KHE_DRS_DOM_TEST dt, float val1, float val2)
{
  return val1 >= dt->b.f || (TildeFloat(dt, val1, val2) && val1 >= val2);
}

static bool KheDrsDomTestDominatesSeparateFloat(KHE_DRS_DOM_TEST dt,
  float val1, float val2)
{
  return KheDrsDomMaxFloat(dt, val1, val2) && KheDrsDomMinFloat(dt, val1, val2);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTryTradeoff(KHE_DRS_DOM_TEST dt, int delta_val,               */
/*    KHE_COST *cost1, KHE_COST cost2)                                       */
/*                                                                           */
/*  Try tradeoff dominance at dt with value change delta_val, returning      */
/*  true if successful.                                                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTryTradeoff(KHE_DRS_DOM_TEST dt, int delta_val,
  KHE_COST *avail_cost)
{
  HnAssert(delta_val >= 0, "KheDrsTryTradeoff internal error");
  *avail_cost -= dt->combined_weight * delta_val;
  return *avail_cost >= 0;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestDominatesTradeoff(KHE_DRS_DOM_TEST dt,                 */
/*    int val1, int val2, KHE_COST *avail_cost)                              */
/*                                                                           */
/*  Return true if val1 dominates val2, according to dt, allowing for a      */
/*  tradeoff.                                                                */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDomTestDominatesTradeoff(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost)
{
  if( KheDrsDomMaxInt(dt, val1, val2) )
  {
    if( KheDrsDomMinInt(dt, val1, val2) )
      return true;
    else if( dt->allow_zero )
      return false;
    else
      return KheDrsTryTradeoff(dt, val2 - val1, avail_cost);
  }
  else
  {
    if( KheDrsDomMinInt(dt, val1, val2) )
      return KheDrsTryTradeoff(dt, val1 - val2, avail_cost);
    else
      return false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDominatesTabulated(KHE_DRS_DOM_TEST dt,                */
/*    int val1, int val2, KHE_COST *avail_cost)                              */
/*                                                                           */
/*  Check whether val1 dominates val2 using tabulated dominance, updating    */
/*  *avail_cost by the change in available cost.                             */
/*  Tables:  main_dom_table2                                                 */
/*                                                                           */
/*****************************************************************************/
static char *KheDrsMonitorId(KHE_DRS_MONITOR dm);

static void KheDrsDomTestDominatesTabulated(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost)
{
  KHE_DRS_COST_TUPLE ct;
  if( DEBUG60 || DEBUG44_MONITOR(dt->monitor) )
    fprintf(stderr, "  [ KheDrsDomTestDominatesTabulated(%s, %d, %d)\n",
      KheDrsMonitorId(dt->monitor), val1, val2);
  if( DEBUG90 && !KheDrsDim2TableGet2Check(dt->main_dom_table2, val1, val2) )
  {
    fprintf(stderr, "KheDrsDomTestDominatesTabulated(dt, %d, %d) failing:\n",
      val1, val2);
    KheDrsDomTestDebug(dt, 2, 2, stderr);
  }
  ct = KheDrsDim2TableGet2(dt->main_dom_table2, val1, val2);
  *avail_cost += ct.unweighted_psi * dt->combined_weight;
  if( DEBUG60 || DEBUG44_MONITOR(dt->monitor) )
    fprintf(stderr, "  ]\n");
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestIsStrictEquality(KHE_DRS_DOM_TEST dom_test)            */
/*                                                                           */
/*  Return true if dom_test amounts to a strict equality test.  As shown     */
/*  in the documentation, the allow_zero flag does not affect the result.    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDomTestIsStrictEquality(KHE_DRS_DOM_TEST dom_test)
{
  return dom_test->a.i < 0 && dom_test->b.i >= INT_MAX;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDominatingSet(KHE_DRS_DOM_TEST dt, int x,              */
/*    int *from_val, int *to_val, bool *also_zero)                           */
/*                                                                           */
/*  Set [*from_val, *to_val] and *also_zero to the set of non-negative       */
/*  values that dominate x under dt.  That is, set it to                     */
/*                                                                           */
/*    { v | v >= 0 && KheDrsDomTestDominatesSeparate(dt, v, x) }             */
/*                                                                           */
/*  The formula for this is derived in the User's Guide.                     */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDomTestDominatingSetInt(KHE_DRS_DOM_TEST dt, int x,
  int *from_val, int *to_val, bool *also_zero)
{
  if( dt->allow_zero && x == 0 )
  {
    *from_val = max(0, dt->b.i);
    *to_val = max(dt->a.i, 0);
    *also_zero = (*from_val > 0);
  }
  else
  {
    *from_val = min(dt->b.i, x);
    if( *from_val < 0 )  *from_val = 0;
    *to_val = max(dt->a.i, x);
    *also_zero = false;
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDominatedSet(KHE_DRS_DOM_TEST dt, int x,               */
/*    int *from_val, int *to_val, bool *also_zero)                           */
/*                                                                           */
/*  Set [*from_val, *to_val] and *also_zero to the set of non-negative       */
/*  values that x dominates under dt.  That is, set it to                    */
/*                                                                           */
/*    { v | v >= 0 && KheDrsDomTestDominatesSeparate(dt, x, val) }           */
/*                                                                           */
/*  This formula is derived in the Appendix to the User's Guide.             */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDomTestDominatedSetInt(KHE_DRS_DOM_TEST dt, int x,
  int *from_val, int *to_val, bool *also_zero)
{
  if( dt->allow_zero && x >= 1 )
  {
    *from_val = (x <= dt->a.i ? 1 : max(1, x));
    *to_val = (x >= dt->b.i ? INT_MAX : x);
    if( x <= max(dt->a.i, 0) && x >= dt->b.i )
    {
      ** make sure 0 is in the set **
      if( *from_val == 1 )
	*from_val = 0;
      *also_zero = (*from_val > 0);
    }
    else
      *also_zero = false;
  }
  else
  {
    *from_val = (x <= dt->a.i ? 0 : x);
    *to_val = (x >= dt->b.i ? INT_MAX : x);
    *also_zero = false;
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDebug(KHE_DRS_DOM_TEST dt, int verbosity,              */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dt onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDomTestDebug(KHE_DRS_DOM_TEST dt, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "DomTest %p(%s %s, table2 %p)", (void *) dt,
    KheDrsDomTestTypeShow(dt->type), KheDrsMonitorId(dt->monitor),
    (void *) dt->main_dom_table2);
    /* dt->main_dom_table2 != NULL ? dt->main_dom_table2->debug_str : "-", */
    /* dt->corr_dom_table4 != NULL ? dt->corr_dom_table4->debug_str : "-"); */
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "constraints"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_CONSTRAINT"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprIsSingleDay(KHE_DRS_EXPR e, KHE_FRAME days_frame,         */
/*    KHE_TIME_GROUP *time_group)                                            */
/*                                                                           */
/*  If e covers a single day of days_frame, return true and set *time_group  */
/*  to the time group of that day in days_frame.  Otherwise return false     */
/*  with *time_group set to NULL.                                            */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprIsSingleDay(KHE_DRS_EXPR e, KHE_FRAME days_frame,
  KHE_TIME_GROUP *time_group)
{
  KHE_TIME_GROUP tg;  KHE_DRS_EXPR child_e;  int i;
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      /* shortcut; we aren't expecting this case to ever occur */
      return *time_group = NULL, true;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      { KHE_DRS_EXPR_BUSY_TIME ebt;
	ebt = (KHE_DRS_EXPR_BUSY_TIME) e;
	*time_group = KheFrameTimeTimeGroup(days_frame, ebt->time);
	return true;
      }

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      { KHE_DRS_EXPR_FREE_TIME eft;
	eft = (KHE_DRS_EXPR_FREE_TIME) e;
	*time_group = KheFrameTimeTimeGroup(days_frame, eft->time);
	return true;
      }

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      { KHE_DRS_EXPR_WORK_TIME ewt;
	ewt = (KHE_DRS_EXPR_WORK_TIME) e;
	*time_group = KheFrameTimeTimeGroup(days_frame, ewt->time);
	return true;
      }

    case KHE_DRS_EXPR_BUSY_DAY_TAG:
      
      { KHE_DRS_EXPR_BUSY_DAY ebd;
	ebd = (KHE_DRS_EXPR_BUSY_DAY) e;
        *time_group = ebd->time_group;
	return true;
      }

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      { KHE_DRS_EXPR_FREE_DAY efd;
	efd = (KHE_DRS_EXPR_FREE_DAY) e;
        *time_group = efd->time_group;
	return true;
      }

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      { KHE_DRS_EXPR_WORK_DAY ewd;
	ewd = (KHE_DRS_EXPR_WORK_DAY) e;
        *time_group = ewd->time_group;
	return true;
      }

    case KHE_DRS_EXPR_OR_TAG:
    case KHE_DRS_EXPR_AND_TAG:	

      *time_group = NULL;
      HaArrayForEach(e->children, child_e, i)
      {
	if( !KheDrsExprIsSingleDay(child_e, days_frame, &tg) )
	  return false;
	else if( *time_group == NULL )
	  *time_group = tg;
	else if( !KheTimeGroupEqual(*time_group, tg) )
	  return false;
      }
      return (*time_group != NULL);
      
    /* ***
    case KHE_DRS_EXPR_INT_SUM_TAG:
    case KHE_DRS_EXPR_FLOAT_SUM_TAG:
    case KHE_DRS_EXPR_INT_DEV_TAG:
    case KHE_DRS_EXPR_FLOAT_DEV_TAG:
    *** */
    case KHE_DRS_EXPR_COUNTER_TAG:
    case KHE_DRS_EXPR_SUM_INT_TAG:
    case KHE_DRS_EXPR_SUM_FLOAT_TAG:
    case KHE_DRS_EXPR_SEQUENCE_TAG:

      /* shortcut; we aren't expecting these cases to ever occur */
      return *time_group = NULL, false;

    default:

      HnAbort("KheDrsExprIsSingleDay internal error (tag %d)", e->tag);
      return *time_group = NULL, false;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprNeedsCorrTable(KHE_DRS_EXPR e, KHE_FRAME days_frame)      */
/*                                                                           */
/*  Return true if e needs a corr table - if it is a counter expression,     */
/*  and at least one of its children spans more than one day of days_frame.  */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprNeedsCorrTable(KHE_DRS_EXPR e, KHE_FRAME days_frame)
{
  KHE_DRS_EXPR child_e;  KHE_TIME_GROUP tg;  int i;
  if( e->tag == KHE_DRS_EXPR_COUNTER_TAG )
  {
    HaArrayForEach(e->children, child_e, i)
      if( !KheDrsExprIsSingleDay(child_e, days_frame, &tg) )
	return true;
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CONSTRAINT KheDrsConstraintMake(KHE_CONSTRAINT c,                */
/*    int init_history, KHE_DYNAMIC_RESOURCE_SOLVER drs)                     */
/*                                                                           */
/*  Make a constraint object for c.                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CONSTRAINT KheDrsConstraintMake(KHE_CONSTRAINT c,
  int init_history, KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CONSTRAINT res;
  HaMake(res, drs->arena);
  res->constraint = c;
  res->min_history = init_history;
  res->max_history = init_history;
  res->max_child_count = HaArrayCount(e->children);
  res->needs_corr_table = KheDrsExprNeedsCorrTable(e, drs->days_frame);
  res->sample_expr = e;
  res->counter_main_dom_table3 = NULL;
  res->counter_corr_dom_table5 = NULL;
  res->sequence_main_dom_table5 = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsConstraintId(KHE_DRS_CONSTRAINT dc)                          */
/*                                                                           */
/*  Return the Id of dc.                                                     */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsConstraintId(KHE_DRS_CONSTRAINT dc)
{
  return KheConstraintId(dc->constraint);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprCounterExtraUnweightedCost(KHE_DRS_EXPR_COUNTER ec,        */
/*    int e, int l, int y, bool debug)                                       */
/*                                                                           */
/*  Return the extra unweighted cost in the dominance testing formula with   */
/*  these attributes.  This is function Gamma in the documentation.          */
/*                                                                           */
/*****************************************************************************/
static int KheDrsExprCounterDelta(KHE_DRS_EXPR_COUNTER ec,
  int lower_det, int upper_det);
static KHE_COST KheDrsExprCostCost(KHE_DRS_EXPR_COST ec, int dev);
#define f(e, d) KheDrsExprCostCost((KHE_DRS_EXPR_COST) (e), (d))
static int KheDrsExprCostUnweightedCost(KHE_DRS_EXPR_COST ec, int dev);
#define uf(e, d) KheDrsExprCostUnweightedCost((KHE_DRS_EXPR_COST) (e), (d))

static int KheDrsExprCounterExtraUnweightedCost(KHE_DRS_EXPR_COUNTER ec,
  int e, int l, int y, bool debug)
{
  int delta1, delta2, res;
  delta1 = KheDrsExprCounterDelta(ec, l + y, l + y + ec->history_after);
  delta2 = KheDrsExprCounterDelta(ec, l, l + e + ec->history_after);
  res = uf(ec, delta1) - uf(ec, delta2);
  if( debug )
    fprintf(stderr, "    Gamma(e %d, l %d, y %d) = f(%d) %d - f(%d) %d = %d\n",
      e, l, y, delta1, uf(ec, delta1), delta2, uf(ec, delta2), res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprCounterExtraCost(KHE_DRS_EXPR_COUNTER ec,             */
/*    int e, int l, int x, bool debug)                                       */
/*                                                                           */
/*  Return the extra cost in the dominance testing formula with these        */
/*  attributes.  This is function Gamma in the documentation.                */
/*                                                                           */
/*****************************************************************************/
/* *** replaced by unweighted version
static int KheDrsExprCounterDelta(KHE_DRS_EXPR_COUNTER ec,
  int lower_det, int upper_det);
static KHE_COST KheDrsExprCostCost(KHE_DRS_EXPR_COST ec, int dev);
#define f(e, d) KheDrsExprCostCost((KHE_DRS_EXPR_COST) (e), (d))

static KHE_COST KheDrsExprCounterExtraCost(KHE_DRS_EXPR_COUNTER ec,
  int e, int l, int y, bool debug)
{
  int delta1, delta2;  KHE_COST res;
  delta1 = KheDrsExprCounterDelta(ec, l + y, l + y + ec->history_after);
  delta2 = KheDrsExprCounterDelta(ec, l, l + e + ec->history_after);
  res = f(ec, delta1) - f(ec, delta2);
  if( debug )
    fprintf(stderr, "    Gamma(e %d, l %d, y %d) = f(%d) %.5f - f(%d) "
      "%.5f = %.5f\n", e, l, y, delta1, KheCostShow(f(ec, delta1)),
      delta2, KheCostShow(f(ec, delta2)), KheCostShow(res));
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprCounterFindAvailCost(                                 */
/*    KHE_DRS_EXPR_COUNTER ec, int e, int l1, int l2)                        */
/*                                                                           */
/*  Find the available cost for ec given e, l1, and l2.                      */
/*  This implements the outer "min" of the formula in the documentation.     */
/*  KheDrsExprCounterExtraCost is Gamma.                                     */
/*                                                                           */
/*****************************************************************************/
/* *** replaced by unweighted version
#define DEBUG_INT_SUM_DOM(e, d1, d2)					\
  (DEBUG37 && (e) == 10 && (d1) == 10 && (d2) == 5)

static KHE_COST KheDrsExprCounterFindAvailCost(
  KHE_DRS_EXPR_COUNTER ec, int e, int l1, int l2)
{
  int y;  KHE_COST res, extra_cost1, extra_cost2, cost;  bool debug;
  debug = DEBUG_INT_SUM_DOM(e, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprCounterFindAvailCost(%s, e %d, l1 %d,"
      " l2 %d)\n", KheDrsMonitorId(ec->monitor), e, l1, l2);
  res = KheCost(INT_MAX, INT_MAX);
  for( y = 0;  y <= e;  y++ )
  {
    extra_cost2 = KheDrsExprCounterExtraCost(ec, e, l2, y, debug);
    extra_cost1 = KheDrsExprCounterExtraCost(ec, e, l1, y, debug);
    cost = extra_cost2 - extra_cost1;
    if( debug )
      fprintf(stderr, "  y = %2d: %8.5f - %8.5f = %8.5f\n", y,
        KheCostShow(extra_cost2), KheCostShow(extra_cost1), KheCostShow(cost));
    if( cost < res )
      res = cost;
  }
  HnAssert(res <= 0, "KheDrsExprCounterFindAvailCost: res > 0");
  if( debug )
    fprintf(stderr, "] KheDrsExprCounterFindAvailCost returning %.5f\n",
      KheCostShow(res));
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/* int KheDrsExprCounterFindAvailUnweightedCost(KHE_DRS_EXPR_COUNTER ec,     */
/*   int e, int l1, int l2)                                                  */
/*                                                                           */
/*  Find the available cost for ec given e, l1, and l2.                      */
/*  This implements the outer "min" of the formula in the documentation.     */
/*  KheDrsExprCounterExtraCost is Gamma.                                     */
/*                                                                           */
/*****************************************************************************/
#define DEBUG_INT_SUM_DOM(e, d1, d2)					\
  (DEBUG37 && (e) == 10 && (d1) == 10 && (d2) == 5)

static int KheDrsExprCounterFindAvailUnweightedCost(KHE_DRS_EXPR_COUNTER ec,
  int e, int l1, int l2)
{
  int y;  int res, extra_cost1, extra_cost2, cost;  bool debug;
  debug = DEBUG_INT_SUM_DOM(e, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprCounterFindAvailUnweightedCost(%s, e %d,"
      "l1 %d, l2 %d)\n", KheDrsMonitorId(ec->monitor), e, l1, l2);
  res = INT_MAX;
  for( y = 0;  y <= e;  y++ )
  {
    extra_cost2 = KheDrsExprCounterExtraUnweightedCost(ec, e, l2, y,debug);
    extra_cost1 = KheDrsExprCounterExtraUnweightedCost(ec, e, l1, y,debug);
    cost = extra_cost2 - extra_cost1;
    if( debug )
      fprintf(stderr, "  y = %2d: %d - %d = %d\n", y, extra_cost2,
	extra_cost1, cost);
    if( cost < res )
      res = cost;
  }
  HnAssert(res <= 0, "KheDrsExprCounterFindAvailUnweightedCost: res > 0");
  if( debug )
    fprintf(stderr,
      "] KheDrsExprCounterFindAvailUnweightedCost returning %d\n", res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprCounterFindCorrAvailCost(                             */
/*    KHE_DRS_EXPR_COUNTER ec, int e, int a1, int a2, int l1, int l2)        */
/*                                                                           */
/*  Find the correlated available cost for ec given e, a1, a2, l1, and l2.   */
/*                                                                           */
/*****************************************************************************/
/* *** replaced by unweighted version

static KHE_COST KheDrsExprCounterFindCorrAvailCost(
  KHE_DRS_EXPR_COUNTER ec, int e, int a1, int a2, int l1, int l2)
{
  KHE_COST res, extra_cost1, extra_cost2, cost;
  int y1, y2, D_first, D_last, i;  bool debug;
  debug = DEBUG_INT_SUM_CORR(e, a1, a2, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprCounterFindCorrAvailCost(%s, e %d, a1 %d,"
      " a2 %d, l1 %d, l2 %d)\n", KheDrsMonitorId(ec->monitor),
      e, a1, a2, l1, l2);

  ** work out D, in the form D_first .. D_last **
  if( a1 == 0 )
  {
    if( a2 == 0 )
      D_first = D_last = 0;
    else
      D_first = -1, D_last = 0;
  }
  else
  {
    if( a2 == 0 )
      D_first = 0, D_last = 1;
    else
      D_first = D_last = 0;
  }

  ** do the rest **
  res = KheCost(INT_MAX, INT_MAX);
  for( y1 = a1;  y1 <= e;  y1++ )
    for( i = D_first;  i <= D_last;  i++ )
    {
      y2 = y1 - i;
      if( a2 <= y2 && y2 <= e )
      {
	extra_cost2 = KheDrsExprCounterExtraCost(ec, e, l2, y2, debug);
	extra_cost1 = KheDrsExprCounterExtraCost(ec, e, l1, y1, debug);
	cost = extra_cost2 - extra_cost1;
	if( debug )
	  fprintf(stderr, "  y1 = %2d, y2 = %2d: %8.5f - %8.5f = %8.5f\n", y1,
	    y2, KheCostShow(extra_cost2), KheCostShow(extra_cost1),
	    KheCostShow(cost));
	if( cost < res )
	  res = cost;
      }
    }
  if( debug )
    fprintf(stderr, "] KheDrsExprCounterFindCorrAvailCost returning %.5f\n",
      KheCostShow(res));
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprCounterFindCorrAvailCost(                             */
/*    KHE_DRS_EXPR_COUNTER ec, int e, int a1, int a2, int l1, int l2)        */
/*                                                                           */
/*  Find the correlated available cost for ec given e, a1, a2, l1, and l2.   */
/*                                                                           */
/*****************************************************************************/
#define DEBUG_INT_SUM_CORR(e, a1, a2, l1, l2)				\
  (DEBUG56 && (e) == 1 && (a1) == 0 && (a2) == 1 && (l1) == 0 && (l2) == 2)

static int KheDrsExprCounterFindCorrAvailUnweightedCost(
  KHE_DRS_EXPR_COUNTER ec, int e, int a1, int a2, int l1, int l2)
{
  int res, extra_cost1, extra_cost2, cost;
  int y1, y2, D_first, D_last, i;  bool debug;
  debug = DEBUG_INT_SUM_CORR(e, a1, a2, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprCounterFindCorrAvailUnweightedCost(%s, "
      "e %d, a1 %d, a2 %d, l1 %d, l2 %d)\n",
      KheDrsMonitorId(ec->monitor), e, a1, a2, l1, l2);

  /* work out D, in the form D_first .. D_last */
  if( a1 == 0 )
  {
    if( a2 == 0 )
      D_first = D_last = 0;
    else
      D_first = -1, D_last = 0;
  }
  else
  {
    if( a2 == 0 )
      D_first = 0, D_last = 1;
    else
      D_first = D_last = 0;
  }

  /* do the rest */
  res = INT_MAX;
  for( y1 = a1;  y1 <= e;  y1++ )
    for( i = D_first;  i <= D_last;  i++ )
    {
      y2 = y1 - i;
      if( a2 <= y2 && y2 <= e )
      {
	extra_cost2 = KheDrsExprCounterExtraUnweightedCost(ec,
	  e, l2, y2, debug);
	extra_cost1 = KheDrsExprCounterExtraUnweightedCost(ec,
	  e, l1, y1, debug);
	cost = extra_cost2 - extra_cost1;
	if( debug )
	  fprintf(stderr, "  y1 = %2d, y2 = %2d: %d - %d = %d\n", y1,
	    y2, extra_cost2, extra_cost1, cost);
	if( cost < res )
	  res = cost;
      }
    }
  if( debug )
    fprintf(stderr,
      "] KheDrsExprCounterFindCorrAvailUnweightedCost returning %d\n", res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsConstraintSetCounterDomTables(KHE_DRS_CONSTRAINT dc,          */
/*    KHE_DRS_EXPR_COUNTER ec, KHE_DYNAMIC_RESOURCE_SOLVER drs)              */
/*                                                                           */
/*  Set the main_dom_table3 and corr_dom_table5 fields of dc, using          */
/*  ec as an example of the kind of expression they have to handle.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsConstraintSetCounterDomTables(KHE_DRS_CONSTRAINT dc,
  KHE_DRS_EXPR_COUNTER ec, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int e, l1, l2, a1, a2, lim, ucost;  KHE_DRS_COST_TUPLE ct;

  /* ***
  if( KheConstraintTag(dc->constraint) != KHE_LIMIT_WORKLOAD_CONSTRAINT_TAG )
  {
  *** */

  /* build dc->counter_main_dom_table3 */
  if( DEBUG94(dc) )
    fprintf(stderr, "[ KheDrsConstraintSetCounterDomTables(%s, -, drs):\n",
      KheDrsConstraintId(dc));
  /* child_count = HaArrayCount(ec->children); */
  /* debug_str = KheDrsMonitorId(ec->monitor); */
  dc->counter_main_dom_table3 = KheDrsDim3TableMake(/*debug_str,*/drs->arena);
  for( e = 0;  e <= dc->max_child_count;  e++ )
  {
    lim = dc->max_child_count + dc->max_history - e;
    if( DEBUG94(dc) )
      fprintf(stderr, "  e %d, lim %d\n", e, lim);
    for( l1 = dc->min_history;  l1 <= lim;  l1++ )
      for( l2 = dc->min_history;  l2 <= lim;  l2++ )
      {
	ucost = KheDrsExprCounterFindAvailUnweightedCost(ec, e, l1, l2);
	HnAssert(ucost <= SHRT_MAX,
	  "KheDrsConstraintSetCounterDomTables internal error 1");
	ct = KheDrsCostTupleMake((short) ucost, 0, /* false, */ 0);
	KheDrsDim3TablePut(dc->counter_main_dom_table3, e, l1, l2, ct, drs);
      }
  }
  if( DEBUG94(dc) )
    fprintf(stderr, "[ KheDrsConstraintSetCounterDomTables(dc, %s, drs):\n",
      KheDrsMonitorId(ec->monitor));

  /* build dc->counter_corr_dom_table5, if required */
  /* if( KheDrsExprNeedsCorrTable((KHE_DRS_EXPR) ec, drs->days_frame) ) */
  if( dc->needs_corr_table )
  {
    dc->counter_corr_dom_table5 = KheDrsDim5TableMake(
      /* HnStringMake(drs->arena, "Corr %s", debug_str), */ drs->arena);
    for( e = 1;  e <= dc->max_child_count;  e++ )
    {
      lim = dc->max_child_count + dc->max_history - e;
      for( a1 = 0;  a1 <= 1;  a1++ )
	for( a2 = 0;  a2 <= 1;  a2++ )
	  for( l1 = dc->min_history;  l1 <= lim;  l1++ )
	    for( l2 = dc->min_history;  l2 <= lim;  l2++ )
	    {
	      ucost = KheDrsExprCounterFindCorrAvailUnweightedCost(ec,
		e, a1, a2, l1, l2);
	      HnAssert(ucost <= SHRT_MAX,
		"KheDrsConstraintSetCounterDomTables internal error 2");
	      ct = KheDrsCostTupleMake((short) ucost, 0, /* false, */ 0);
	      KheDrsDim5TablePut(dc->counter_corr_dom_table5, e, a1, a2,
		l1, l2, ct, drs);
	    }
    }
  }
  /* } */
}


/*****************************************************************************/
/*                                                                           */
/*  int ci_bar(KHE_DRS_EXPR_SEQUENCE es, int p, int x)                       */
/*                                                                           */
/*  Return the value of ci overbar as defined in the documentation.          */
/*                                                                           */
/*****************************************************************************/

static int ci_bar(KHE_DRS_EXPR_SEQUENCE es, int p, int x)
{
  return x < p ? 0 : es->history_after;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprSequenceExtraCost(KHE_DRS_EXPR_SEQUENCE es,           */
/*    int p, int q, int r, int l, int y, bool debug)                         */
/*                                                                           */
/*  Return the extra cost in the dominance testing formula with these        */
/*  attributes.                                                              */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by unweighted version
static KHE_COST KheDrsExprSequenceExtraCost(KHE_DRS_EXPR_SEQUENCE es,
  int p, int q, int r, int l, int y, bool debug)
{
  int d1, d2;  KHE_COST res;
  d1 = KheDrsExprSequenceDelta(es, l+y, l+y, l+y + ci_bar(es, p, y));
  d2 = KheDrsExprSequenceDelta(es, l+r, l, l+q+r + ci_bar(es, p, q+r));
  res = f(es, d1) - f(es, d2);
  if( debug )
    fprintf(stderr, "    extra(p %d, q %d, r %d, l %d, y %d) = f(%d) %8.5f - "
      "f(%d) %8.5f = %8.5f\n", p, q, r, l, y, d1, KheCostShow(f(es, d1)),
      d2, KheCostShow(f(es, d2)), KheCostShow(res));
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSequenceExtraUnweightedCost(KHE_DRS_EXPR_SEQUENCE es,      */
/*    int p, int q, int r, int l, int y, bool debug)                         */
/*                                                                           */
/*  Return the extra unweighted cost in the dominance testing formula with   */
/*  these attributes.                                                        */
/*                                                                           */
/*****************************************************************************/
static int KheDrsExprSequenceDelta(KHE_DRS_EXPR_SEQUENCE es,
  int z, int lower_det, int upper_det);

static int KheDrsExprSequenceExtraUnweightedCost(KHE_DRS_EXPR_SEQUENCE es,
  int p, int q, int r, int l, int y, bool debug)
{
  int d1, d2, res;
  d1 = KheDrsExprSequenceDelta(es, l+y, l+y, l+y + ci_bar(es, p, y));
  d2 = KheDrsExprSequenceDelta(es, l+r, l, l+q+r + ci_bar(es, p, q+r));
  res = uf(es, d1) - uf(es, d2);
  if( debug )
    fprintf(stderr, "    extra(p %d, q %d, r %d, l %d, y %d) = f(%d) %d - "
      "f(%d) %d = %d\n", p, q, r, l, y, d1, uf(es, d1),
      d2, uf(es, d2), res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprSequenceFindPsi(KHE_DRS_EXPR_SEQUENCE es,             */
/*    int p, int q, int r, int l1, int l2)                                   */
/*                                                                           */
/*  Find psi, the available cost for es given p, q, r, l1, and l2.           */
/*  This implements the outer "min" of the formula in the documentation.     */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by unweighted version
static KHE_COST KheDrsExprSequenceFindPsi(KHE_DRS_EXPR_SEQUENCE es,
  int lower, int p, int q, int r, int l1, int l2)
{
  int y;  KHE_COST res, extra_cost1, extra_cost2, cost;  bool debug;
  HnAssert(lower <= p + q, "KheDrsExprSequenceFindPsi internal error");
  debug = DEBUG_INT_SEQ_DOM(p, q, r, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprSequenceFindPsi(%s, lower %d, p %d, q %d,"
      " r %d, l1 %d, l2 %d)\n", KheDrsMonitorId(es->monitor),
      lower, p, q, r, l1, l2);
  res = KheCost(INT_MAX, INT_MAX);
  for( y = lower;  y <= q + r;  y++ )
  {
    extra_cost2 = KheDrsExprSequenceExtraCost(es, p, q, r, l2, y, debug);
    extra_cost1 = KheDrsExprSequenceExtraCost(es, p, q, r, l1, y, debug);
    cost = extra_cost2 - extra_cost1;
    if( debug )
      fprintf(stderr, "  y = %2d: %8.5f - %8.5f = %8.5f\n", y,
        KheCostShow(extra_cost2), KheCostShow(extra_cost1), KheCostShow(cost));
    if( cost < res )
      res = cost;
  }
  if( debug )
    fprintf(stderr, "] KheDrsExprSequenceFindPsi returning %.5f\n",
      KheCostShow(res));
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSequenceFindUnweightedPsi(KHE_DRS_EXPR_SEQUENCE es,        */
/*    int lower, int p, int q, int r, int l1, int l2)                        */
/*                                                                           */
/*  Find psi, the available cost for es given p, q, r, l1, and l2.           */
/*  This implements the outer "min" of the formula in the documentation.     */
/*                                                                           */
/*****************************************************************************/
#define DEBUG_INT_SEQ_DOM(p, q, r, l1, l2)				\
  (DEBUG37 && (p) == 10 && (l1) == 10 && (l2) == 5)

static int KheDrsExprSequenceFindUnweightedPsi(KHE_DRS_EXPR_SEQUENCE es,
  int lower, int p, int q, int r, int l1, int l2)
{
  int y;  int res, extra_cost1, extra_cost2, cost;  bool debug;
  HnAssert(lower <= p + q, "KheDrsExprSequenceFindPsi internal error");
  debug = DEBUG_INT_SEQ_DOM(p, q, r, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprSequenceFindPsi(%s, lower %d, p %d, q %d,"
      " r %d, l1 %d, l2 %d)\n", KheDrsMonitorId(es->monitor),
      lower, p, q, r, l1, l2);
  res = INT_MAX;
  for( y = lower;  y <= q + r;  y++ )
  {
    extra_cost2 = KheDrsExprSequenceExtraUnweightedCost(es,
      p, q, r, l2, y, debug);
    extra_cost1 = KheDrsExprSequenceExtraUnweightedCost(es,
      p, q, r, l1, y, debug);
    cost = extra_cost2 - extra_cost1;
    if( debug )
      fprintf(stderr, "  y = %2d: %d - %d = %d\n", y,
        extra_cost2, extra_cost1, cost);
    if( cost < res )
      res = cost;
  }
  if( debug )
    fprintf(stderr, "] KheDrsExprSequenceFindPsi returning %d\n", res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprSequenceFindPsi0(KHE_DRS_EXPR_SEQUENCE es,            */
/*    int p, int q, int r, int l1, int l2)                                   */
/*                                                                           */
/*  Find psi0, part of the available cost for es given p, q, r, l1, and      */
/*  This implements the outer "min" of the formula in the documentation.     */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by unweighted version
static KHE_COST KheDrsExprSequenceFindPsi0(KHE_DRS_EXPR_SEQUENCE es,
  int p, int q, int r, int l1, int l2)
{
  int y;  KHE_COST res, extra_cost1, extra_cost2;  bool debug;
  debug = DEBUG_INT_SEQ_DOM(p, q, r, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprSequenceFindPsi0(%s, p %d, q %d,"
      " r %d, l1 %d, l2 %d)\n", KheDrsMonitorId(es->monitor),
      p, q, r, l1, l2);
  res = KheCost(INT_MAX, INT_MAX);
  y = 0;
  extra_cost2 = KheDrsExprSequenceExtraCost(es, p, q, r, l2, y, debug);
  extra_cost1 = KheDrsExprSequenceExtraCost(es, p, q, r, l1, y, debug);
  res = extra_cost2 - extra_cost1;
  if( debug )
  {
    fprintf(stderr, "  y = %2d: %8.5f - %8.5f = %8.5f\n", y,
      KheCostShow(extra_cost2), KheCostShow(extra_cost1), KheCostShow(res));
    fprintf(stderr, "] KheDrsExprSequenceFindPsi0 returning %.5f\n",
      KheCostShow(res));
  }
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSequenceFindUnweightedPsi0(KHE_DRS_EXPR_SEQUENCE es,       */
/*    int p, int q, int r, int l1, int l2)                                   */
/*                                                                           */
/*  Find psi0, part of the available cost for es given p, q, r, l1, and l2.  */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprSequenceFindUnweightedPsi0(KHE_DRS_EXPR_SEQUENCE es,
  int p, int q, int r, int l1, int l2)
{
  int y;  int res, extra_cost1, extra_cost2;  bool debug;
  debug = DEBUG_INT_SEQ_DOM(p, q, r, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprSequenceFindUnweightedPsi0(%s, p %d, q %d,"
      " r %d, l1 %d, l2 %d)\n", KheDrsMonitorId(es->monitor), p, q, r, l1, l2);
  res = INT_MAX;
  y = 0;
  extra_cost2 = KheDrsExprSequenceExtraUnweightedCost(es,
    p, q, r, l2, y, debug);
  extra_cost1 = KheDrsExprSequenceExtraUnweightedCost(es,
    p, q, r, l1, y, debug);
  res = extra_cost2 - extra_cost1;
  if( debug )
  {
    fprintf(stderr, "  y = %2d: %d - %d = %d\n", y,
      extra_cost2, extra_cost1, res);
    fprintf(stderr, "] KheDrsExprSequenceFindUnweightedPsi0 returning %d\n",
      res);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsConstraintSetSequenceDomTables(KHE_DRS_CONSTRAINT dc,         */
/*    KHE_DRS_EXPR_SEQUENCE es, KHE_DYNAMIC_RESOURCE_SOLVER drs)             */
/*                                                                           */
/*  Set the dom_table field of es.                                           */
/*                                                                           */
/*****************************************************************************/
/* ***
static void KheDrsExprSequenceDebugDomTable(KHE_DRS_EXPR_SEQUENCE es,
  int verbosity, int indent, FILE *fp);
*** */

static void KheDrsConstraintSetSequenceDomTables(KHE_DRS_CONSTRAINT dc,
  KHE_DRS_EXPR_SEQUENCE es, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int p, q, r, l1, l2, child_count;  /* KHE_COST psi, psi0, psi_plus; */
  int upsi, upsi0, upsi_plus;
  KHE_DRS_COST_TUPLE ct;  char *debug_str;  HA_ARENA a;
  debug_str = KheDrsMonitorId(es->monitor);
  if( DEBUG42 )
    fprintf(stderr, "[ KheDrsConstrantSetSequenceDomTable(%s)\n", debug_str);
  child_count = HaArrayCount(es->children);
  a = drs->arena;
  dc->sequence_main_dom_table5 =
    KheDrsDim5TableMake( /* HnStringMake(a, "Seq %s",debug_str), */ a);
  for( p = 0;  p <= child_count;  p++ )
    for( q = 0;  q <= p;  q++ )
      for( r = 0;  r <= p - q;  r++ )
	for( l1 = 0;  l1 <= child_count - p + dc->max_history;  l1++ )
	  for( l2 = 0;  l2 <= child_count - p + dc->max_history;  l2++ )
	  {
	    upsi = KheDrsExprSequenceFindUnweightedPsi(es,
	      0, p, q, r, l1, l2);
	    upsi0 = KheDrsExprSequenceFindUnweightedPsi0(es,
	      p, q, r, l1, l2);
	    if( q == 0 )
	      ct = KheDrsCostTupleMake((short) upsi, (short) upsi0,
		/* false, */ 0);
	    else
	    {
	      upsi_plus = KheDrsExprSequenceFindUnweightedPsi(es,
		1, p, q, r, l1, l2);
	      ct = KheDrsCostTupleMake((short) upsi, (short) upsi0,
		/* true, */ (short) upsi_plus);
	    }
	    /* ***
	    psi = KheDrsExprSequenceFindPsi(es, 0, p, q, r, l1, l2);
	    psi0 = KheDrsExprSequenceFindPsi0(es, p, q, r, l1, l2);
	    if( q == 0 )
	      ct = KheDrsCostTupleMake(psi, psi0, ** false, ** 0);
	    else
	    {
	      psi_plus = KheDrsExprSequenceFindPsi(es, 1, p, q, r, l1, l2);
	      ct = KheDrsCostTupleMake(psi, psi0, ** true, ** psi_plus);
	    }
	    *** */
	    KheDrsDim5TablePut(dc->sequence_main_dom_table5, p, q, r,
	      l1, l2, ct, drs);
	  }
  /* ***
  if( DEBUG36 || DEBUG44_MONITOR(es->monitor->monitor) )
    KheDrsExprSequenceDebugDomTable(es, 1, 0, stderr);
  *** */
  if( DEBUG42 )
    fprintf(stderr, "] KheDrsConstrantSetSequenceDomTable\n");
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprSequenceWantDebugDomTable(                                */
/*    KHE_DRS_EXPR_SEQUENCE es)                                              */
/*                                                                           */
/*  Return true if we want to debug es's dom table.                         */
/*                                                                           */
/*  At present we are doing this is ec's monitor is a resource monitor       */
/*  and the resource it monitors is the first resource of its type.          */
/*  This is to limit the debug output to a manageable amount.                */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsExprSequenceWantDebugDomTable(
  KHE_DRS_EXPR_SEQUENCE es)
{
  KHE_MONITOR m;  KHE_RESOURCE r;  KHE_RESOURCE_TYPE rt;

  ** find the resource **
  m = es->monitor->monitor;
  HnAssert(KheMonitorTag(m) == KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG,
    "KheDrsExprSequenceWantDebugDomTable internal error");
  r = KheLimitActiveIntervalsMonitorResource(
    (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) m);

  ** return true if r is the first resource of its type **
  rt = KheResourceResourceType(r);
  return KheResourceTypeResourceCount(rt) > 0 &&
    KheResourceTypeResource(rt, 0) == r;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSequenceDebugDomTable(KHE_DRS_EXPR_SEQUENCE es,           */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of es's dom table.                                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprSequenceDebugDomTable(KHE_DRS_EXPR_SEQUENCE es,
  int verbosity, int indent, FILE *fp)
{
  if( KheDrsExprSequenceWantDebugDomTable(es) )
  {
    fprintf(fp, "%*s[ ExprCost for %s\n", indent, "",
      KheDrsMonitorId(es->monitor));
    fprintf(fp, "%*s  cost_fn %s, combined_weight %.5f, min_limit %d,"
      " max_limit %d\n", indent, "",
      KheCostFunctionShow(es->cost_fn), KheCostShow(es->combined_weight),
      es->min_limit, es->max_limit);
    fprintf(fp, "%*s  history_before %d, history_after %d, history %d\n",
      indent, "", es->history_before, es->history_after, es->history);
    KheDrsDim5TableDebug(es->dom_table, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsConstraintSetTables(KHE_DRS_CONSTRAINT dc,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Set the tables of dc.                                                    */
/*                                                                           */
/*****************************************************************************/
static void KheDrsConstraintDebug(KHE_DRS_CONSTRAINT dc,
  int verbosity, int indent, FILE *fp);

static void KheDrsConstraintSetTables(KHE_DRS_CONSTRAINT dc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(dc != NULL, "KheDrsConstraintSetTables internal error 1");
  HnAssert(dc->sample_expr!=NULL, "KheDrsConstraintSetTables internal error 2");
  switch( dc->sample_expr->tag )
  {
    case KHE_DRS_EXPR_COUNTER_TAG:

      /* set counter dom tables */
      KheDrsConstraintSetCounterDomTables(dc,
	(KHE_DRS_EXPR_COUNTER) dc->sample_expr, drs);
      break;

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      /* set sequence dom tables */
      KheDrsConstraintSetSequenceDomTables(dc,
	(KHE_DRS_EXPR_SEQUENCE) dc->sample_expr, drs);
      break;

    default:

      HnAbort("KheDrsConstraintSetTables internal error (tag %d)",
	dc->sample_expr->tag);
      break;
  }
  if( DEBUG36 )
    KheDrsConstraintDebug(dc, 1, 0, stderr);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprCounterWantDebugDomTable(                                 */
/*    KHE_DRS_EXPR_COUNTER ec)                                               */
/*                                                                           */
/*  Return true if we want to debug ec's dom table.                          */
/*                                                                           */
/*  At present we are doing this is ec's monitor is a resource monitor       */
/*  and the resource it monitors is the first resource of its type.          */
/*  This is to limit the debug output to a manageable amount.                */
/*                                                                           */
/*****************************************************************************/

/* *** irrelevant now we have one table per constraint, not per monitor
static bool KheDrsExprCounterWantDebugDomTable(
  KHE_DRS_EXPR_COUNTER ec)
{
  KHE_MONITOR m;  KHE_RESOURCE r;  KHE_RESOURCE_TYPE rt;

  ** find the resource ec monitors; return false if not resource monitor **
  m = ec->monitor->monitor;
  switch( KheMonitorTag(m) )
  {
    case KHE_AVOID_CLASHES_MONITOR_TAG:

      r = KheAvoidClashesMonitorResource((KHE_AVOID_CLASHES_MONITOR) m);
      break;

    case KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG:

      r = KheAvoidUnavailableTimesMonitorResource(
	(KHE_AVOID_UNAVAILABLE_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_IDLE_TIMES_MONITOR_TAG:

      r = KheLimitIdleTimesMonitorResource((KHE_LIMIT_IDLE_TIMES_MONITOR) m);
      break;

    case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

      r = KheClusterBusyTimesMonitorResource(
	(KHE_CLUSTER_BUSY_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

      r = KheLimitBusyTimesMonitorResource((KHE_LIMIT_BUSY_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_WORKLOAD_MONITOR_TAG:

      r = KheLimitWorkloadMonitorResource((KHE_LIMIT_WORKLOAD_MONITOR) m);
      break;

    default:

      return false;
  }

  ** return true if r is the first resource of its type **
  rt = KheResourceResourceType(r);
  return KheResourceTypeResourceCount(rt) > 0 &&
    KheResourceTypeResource(rt, 0) == r;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsConstraintDebug(KHE_DRS_CONSTRAINT dc,                        */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of ec's dom table.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsConstraintDebug(KHE_DRS_CONSTRAINT dc,
  int verbosity, int indent, FILE *fp)
{
  fprintf(fp, "%*s[ KheDrsConstraint %s (min_history %d, max_history %d)\n",
    indent, "", KheConstraintId(dc->constraint), dc->min_history,
    dc->max_history);
  if( dc->counter_main_dom_table3 != NULL )
  {
    fprintf(fp, "%*s  counter_main_dom_table3:\n", indent, "");
    KheDrsDim3TableDebug(dc->counter_main_dom_table3, verbosity, indent+2, fp);
  }
  if( dc->counter_corr_dom_table5 != NULL )
  {
    fprintf(fp, "%*s  counter_corr_dom_table5:\n", indent, "");
    KheDrsDim5TableDebug(dc->counter_corr_dom_table5, verbosity, indent+2, fp);
  }
  if( dc->sequence_main_dom_table5 != NULL )
  {
    fprintf(fp, "%*s  sequence_main_dom_table5:\n", indent, "");
    KheDrsDim5TableDebug(dc->sequence_main_dom_table5, verbosity, indent+2, fp);
  }
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_MONITOR"                                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_MONITOR KheDrsMonitorMakeAndAdd(KHE_MONITOR m,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new monitor object for m, add it to drs->all_monitors, and        */
/*  return it.                                                               */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_MONITOR KheDrsMonitorMakeAndAdd(KHE_MONITOR m,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_MONITOR res;
  HaMake(res, drs->arena);
  res->monitor = m;
  res->rerun_open_and_search_cost = 0;
  res->rerun_open_and_close_cost = 0;
  res->sample_expr = NULL;
  HaArrayAddLast(drs->all_monitors, res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorSetSampleExpr(KHE_DRS_MONITOR dm, KHE_DRS_EXPR e)      */
/*                                                                           */
/*  Set the sample_expr field of dm.  This is used for debugging.            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorSetSampleExpr(KHE_DRS_MONITOR dm, KHE_DRS_EXPR e)
{
  dm->sample_expr = e;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsMonitorId(KHE_DRS_MONITOR dm)                                */
/*                                                                           */
/*  Monitor info id.                                                         */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsMonitorId(KHE_DRS_MONITOR dm)
{
  return dm==NULL ? "-" : dm->monitor == NULL ? "?" : KheMonitorId(dm->monitor);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorDoUpdateRerunCost(KHE_DRS_MONITOR dm,                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, char *field, char *operation,         */
/*    int child_index, KHE_COST *cost_field, char *fmt, va_list ap)          */
/*                                                                           */
/*  Do the actual work of KheDrsMonitorUpdateRerunCost.                      */
/*                                                                           */
/*  Implementation note.  If dsg->type == KHE_DRS_SIGNER_SHIFT, we do        */
/*  nothing here, because this call is part of an evaluation whose           */
/*  purpose is only for dominance testing, not for adding to solutions.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorDoUpdateRerunCost(KHE_DRS_MONITOR dm,
  KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_SIGNER dsg,
  char *field, char *operation, int child_index, KHE_COST *cost_field,
  char *fmt, va_list ap)
{
  char field_buff[3], op_buff[8], buff[50];  int count;  KHE_COST cost;
  bool print;  KHE_SOLN soln;
  print = RERUN_PRINT(drs, dm);
  if( print )
  {
    snprintf(field_buff, 3, "%s", field);
    snprintf(op_buff, 8, "%s", operation);
    if( child_index >= 0 )
      snprintf(buff, 50, "%s %s %s %d",
	dsg != NULL ? KheDrsSignerId(dsg) : "-",
	field_buff, op_buff, child_index);
    else
      snprintf(buff, 50, "%s %s %s", dsg != NULL ? KheDrsSignerId(dsg) : "-",
	field_buff, op_buff);
    soln = KheMonitorSoln(dm->monitor);
    fprintf(stderr, "%d# %15s: %.5f", KheSolnDiversifier(soln),
      buff, KheCostShow(*cost_field));
  }
  for( count = 0;  fmt[count] != '\0';  count++ )
  {
    cost = va_arg(ap, KHE_COST);
    if( print )
      fprintf(stderr, " %c %.5f", fmt[count], KheCostShow(cost));
    if( fmt[count] == '+' )
      *cost_field += cost;
    else
      *cost_field -= cost;
  }
  if( print )
  {
    while( count < 4 )
    {
      fprintf(stderr, "          ");
      count++;
    }
    fprintf(stderr, "  (%s)\n", KheMonitorId(dm->monitor));
    /* ***
    if( e != NULL )
      KheDrsExprDebug(e, drs, 2, 2, stderr);
    *** */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorUpdateRerunCost(KHE_DRS_MONITOR dm,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_SIGNER dsg,                   */
/*    KHE_DRS_OP op, char *operation, int child_index, char *fmt, ...)       */
/*                                                                           */
/*  Update mi.  Also produce a debug print about it, if mi's monitor is      */
/*  RERUN_MONITOR_ID.                                                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorUpdateRerunCost(KHE_DRS_MONITOR dm,
  KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_SIGNER dsg,
  KHE_DRS_OP op, char *operation, int child_index, char *fmt, ...)
{
  va_list ap;
  /* if( drs->rerun_soln != NULL && (dsg==NULL || dsg->debug_eval_if_rerun) ) */
  if( drs->rerun_soln != NULL && KheDrsSignerMonitorUpdateIfRerun(dsg) )
  {
    if( op == KHE_DRS_OPEN || op == KHE_DRS_SEARCH )
    {
      va_start(ap, fmt);
      KheDrsMonitorDoUpdateRerunCost(dm, e, drs, dsg, "os", operation,
	child_index, &dm->rerun_open_and_search_cost, fmt, ap);
      va_end(ap);
    }
    if( op == KHE_DRS_OPEN || op == KHE_DRS_CLOSE )
    {
      va_start(ap, fmt);
      KheDrsMonitorDoUpdateRerunCost(dm, e, drs, dsg, "oc", operation,
	child_index, &dm->rerun_open_and_close_cost, fmt, ap);
      va_end(ap);
    }
    if( ALL_PRINT(drs, dm) && dm->sample_expr != NULL )
      KheDrsExprDebug(dm->sample_expr, drs, 2, 6, stderr);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorInitRerunCost(KHE_DRS_MONITOR dm)                      */
/*                                                                           */
/*  Initialize the rerun cost fields of mi to its monitor's cost.            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorInitRerunCost(KHE_DRS_MONITOR dm,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  dm->rerun_open_and_search_cost = dm->rerun_open_and_close_cost = 0;
  KheDrsMonitorUpdateRerunCost(dm, NULL, drs, NULL, KHE_DRS_OPEN, "init", -1,
    "+", KheMonitorCost(dm->monitor));
  if( RERUN_PRINT(drs, dm) )
  {
    fprintf(stderr, "%d#   ", KheSolnDiversifier(drs->soln));
    KheMonitorDebug(dm->monitor, 2, 0, stderr);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorCheckRerunCost(KHE_DRS_MONITOR dm,                     */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Check the rerun costs of mi, and do some printing if there is a problem. */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorCheckRerunCost(KHE_DRS_MONITOR dm,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_SOLN soln;
  if( dm->rerun_open_and_search_cost != KheMonitorCost(dm->monitor) ||
      dm->rerun_open_and_close_cost != KheMonitorCost(dm->monitor) )
  {
    soln = KheMonitorSoln(dm->monitor);
    fprintf(stderr, "%d# [ KheDrsRerun inconsistent monitor%s\n",
      KheSolnDiversifier(soln), RERUN_PRINT(drs, dm) ? " (+++)" : "");
    fprintf(stderr, "%d#   ", KheSolnDiversifier(soln));
    KheMonitorDebug(dm->monitor, 2, 0, stderr);
    if( dm->sample_expr != NULL )
      KheDrsExprDebug(dm->sample_expr, drs, 2, 2, stderr);
    fprintf(stderr, "%d#  %s (%s)\n", KheSolnDiversifier(soln),
      KheMonitorId(dm->monitor),
      KheMonitorAttachedToSoln(dm->monitor) ? "attached" : "detached");
    fprintf(stderr, "%d#  monitor cost         %.5f\n",
      KheSolnDiversifier(soln),
      KheCostShow(KheMonitorCost(dm->monitor)));
    if( dm->rerun_open_and_search_cost != KheMonitorCost(dm->monitor) )
      fprintf(stderr, "%d#  open_and_search_cost %.5f\n",
	KheSolnDiversifier(soln),
	KheCostShow(dm->rerun_open_and_search_cost));
    if( dm->rerun_open_and_close_cost != KheMonitorCost(dm->monitor) )
      fprintf(stderr, "%d#  open_and_close_cost  %.5f\n",
	KheSolnDiversifier(soln),
	KheCostShow(dm->rerun_open_and_close_cost));
    fprintf(stderr, "%d# ]\n", KheSolnDiversifier(soln));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsMonitorSolnIndexTypedCmp(KHE_DRS_MONITOR dm1,                  */
/*    KHE_DRS_MONITOR dm2)                                                   */
/*                                                                           */
/*  Typed comparison function to sort monitors by increasing soln index.     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsMonitorSolnIndexTypedCmp(KHE_DRS_MONITOR dm1,
  KHE_DRS_MONITOR dm2)
{
  return KheMonitorSolnIndex(dm1->monitor) - KheMonitorSolnIndex(dm2->monitor);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsMonitorSolnIndexCmp(const void *t1, const void *t2)            */
/*                                                                           */
/*  Untyped comparison function to sort monitors by increasing soln index.   */
/*                                                                           */
/*****************************************************************************/

static int KheDrsMonitorSolnIndexCmp(const void *t1, const void *t2)
{
  KHE_DRS_MONITOR dm1 = * (KHE_DRS_MONITOR *) t1;
  KHE_DRS_MONITOR dm2 = * (KHE_DRS_MONITOR *) t2;
  return KheDrsMonitorSolnIndexTypedCmp(dm1, dm2);
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "expressions"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_OPEN_CHILDREN"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_OPEN_CHILD KheDrsOpenChildMake(KHE_DRS_EXPR child_e,             */
/*    int open_index)                                                        */
/*                                                                           */
/*  Make an open child object with these attributes.  The reverse            */
/*  cumulative thing is done separately.                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_OPEN_CHILD KheDrsOpenChildMake(KHE_DRS_EXPR child_e,
  int open_index)
{
  KHE_DRS_OPEN_CHILD res;
  res.child_e = child_e;
  res.open_index = open_index;
  res.rev_cum_value_ub.i = 0;  /* placeholder, will be set separately */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenInit(KHE_DRS_OPEN_CHILDREN oc,                    */
/*    KHE_DRS_OPEN_CHILDREN_INDEX_TYPE index_type, bool float_ub,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Initialize oc.                                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenInit(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_OPEN_CHILDREN_INDEX_TYPE index_type, bool float_ub,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayInit(oc->open_children, drs->arena);
  oc->index_range = KheIntervalMake(1, 0);
  oc->index_type = index_type;
  oc->float_ub = float_ub;
  HaArrayInit(oc->child_indexes, drs->arena);

  /* ***
  HaArrayInit(oc->open_children, drs->arena);
  oc->range = KheIntervalMake(1, 0);
  HaArrayInit(oc->child_indexes, drs->arena);
  oc->float_ub = float_ub;
  if( float_ub )
  {
    HaArrayInit(oc->upper_bounds.f, drs->arena);
    HaArrayAddLast(oc->upper_bounds.f, 0.0);
  }
  else
  {
    HaArrayInit(oc->upper_bounds.i, drs->arena);
    HaArrayAddLast(oc->upper_bounds.i, 0);
  }
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenClear(KHE_DRS_OPEN_CHILDREN oc)                   */
/*                                                                           */
/*  Clear out oc to the state it takes when it is closed.                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenClear(KHE_DRS_OPEN_CHILDREN oc)
{
  HaArrayClear(oc->open_children);
  oc->index_range = KheIntervalMake(1, 0);
  HaArrayClear(oc->child_indexes);

  /* ***
  HaArrayClear(oc->open_children);
  oc->range = KheIntervalMake(1, 0);
  HaArrayClear(oc->child_indexes);
  if( oc->float_ub )
  {
    HaArrayClear(oc->upper_bounds.f);
    HaArrayAddLast(oc->upper_bounds.f, 0.0);
  }
  else
  {
    HaArrayClear(oc->upper_bounds.i);
    HaArrayAddLast(oc->upper_bounds.i, 0);
  }
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenCount(KHE_DRS_OPEN_CHILDREN oc)                    */
/*                                                                           */
/*  Return the number of open children in oc.                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenCount(KHE_DRS_OPEN_CHILDREN oc)
{
  return HaArrayCount(oc->open_children);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenUpdateIndexRange(KHE_DRS_OPEN_CHILDREN oc)        */
/*                                                                           */
/*  Update oc->index_range to be consistent with oc->open_children.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenUpdateIndexRange(KHE_DRS_OPEN_CHILDREN oc)
{
  int first_index, last_index;
  if( HaArrayCount(oc->open_children) == 0 )
    oc->index_range = KheIntervalMake(1, 0);
  else
  {
    first_index = HaArrayFirst(oc->open_children).open_index;
    last_index = HaArrayLast(oc->open_children).open_index;
    oc->index_range = KheIntervalMake(first_index, last_index);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenUpdateChildIndexes(KHE_DRS_OPEN_CHILDREN oc)      */
/*                                                                           */
/*  Update oc->child_indexes to be consistent with oc->open_children.        */
/*  This assumes that oc->index_range is up to date.                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenUpdateChildIndexes(KHE_DRS_OPEN_CHILDREN oc)
{
  int index, i;  KHE_DRS_OPEN_CHILD open_child;
  HaArrayClear(oc->child_indexes);
  index = oc->index_range.first - 1;
  HaArrayForEach(oc->open_children, open_child, i)
  {
    while( open_child.open_index > index )
    {
      index++;
      HaArrayAddLast(oc->child_indexes, i);

      /******************************************************************/
      /*                                                                */
      /*  Here we have just added i at position index of child_indexes, */
      /*  and the child at index i is the first to have a larger last   */
      /*  index than index.                                             */
      /*                                                                */
      /******************************************************************/
    }
  }
  HaArrayAddLast(oc->child_indexes, i);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenUpdateUpperBounds(KHE_DRS_OPEN_CHILDREN oc)       */
/*                                                                           */
/*  Update oc's upper bounds.                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenUpdateUpperBounds(KHE_DRS_OPEN_CHILDREN oc)
{
  int i, sum_i;  KHE_DRS_OPEN_CHILD open_child;  float sum_f;
  if( oc->float_ub )
  {
    /* float version */
    sum_f = 0.0;
    HaArrayForEachReverse(oc->open_children, open_child, i)
    {
      sum_f += open_child.child_e->value_ub.f;
      HaArray(oc->open_children, i).rev_cum_value_ub.f = sum_f;
    }
  }
  else
  {
    /* int version */
    sum_i = 0;
    HaArrayForEachReverse(oc->open_children, open_child, i)
    {
      sum_i += open_child.child_e->value_ub.i;
      HaArray(oc->open_children, i).rev_cum_value_ub.i = sum_i;
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprOpenShiftIndex(KHE_DRS_EXPR e)                             */
/*                                                                           */
/*  Return the open shift index of the shift containing the task monitored   */
/*  by e, which is known to be a KHE_DRS_EXPR_ASSIGNED_TASK expression.      */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprOpenShiftIndex(KHE_DRS_EXPR e)
{
  KHE_DRS_EXPR_ASSIGNED_TASK eat;  int res;
  eat = (KHE_DRS_EXPR_ASSIGNED_TASK) e;
  res = eat->task_on_day->encl_dt->encl_dmt->encl_shift->open_shift_index;
  HnAssert(res >= 0, "KheDrsExprOpenShiftIndex internal error 2");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenAddChild(KHE_DRS_OPEN_CHILDREN oc,                */
/*    KHE_DRS_EXPR child_e)                                                  */
/*                                                                           */
/*  Add child_e to oc->children in sorted open index position.               */
/*                                                                           */
/*  Implementation note.  This code includes an inefficient linear search    */
/*  and inefficient updates to the child indexes and upper bounds.  But it   */
/*  happens only during opening so it doesn't matter.  Simplicity matters.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenAddChild(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_EXPR child_e)
{
  int i, open_index, prev_open_index;  KHE_DRS_OPEN_CHILD tmp, open_child;

  /* make the child's open index */
  switch( oc->index_type )
  {
    case KHE_DRS_OPEN_CHILDREN_INDEX_DAY:

      /* open index is child_e's last day */
      open_index = child_e->open_children_by_day.index_range.last;
      break;

    case KHE_DRS_OPEN_CHILDREN_INDEX_DAY_ADJUSTED:

      /* open index is child_e's last day after adjustment */
      if( HaArrayCount(oc->open_children) > 0 )
      {
	prev_open_index = HaArrayLast(oc->open_children).open_index;
	if( child_e->open_children_by_day.index_range.last < prev_open_index )
	  child_e->open_children_by_day.index_range.last = prev_open_index;
      }
      open_index = child_e->open_children_by_day.index_range.last;
      break;

    case KHE_DRS_OPEN_CHILDREN_INDEX_SHIFT:

      /* open index is child_e's shift index */
      open_index = KheDrsExprOpenShiftIndex(child_e);
      break;

    default:
      HnAbort("KheDrsOpenChildrenAddChild internal error");
      open_index = 0;  /* keep compiler happy */
  }

  /* add child_e to oc->children in sorted position */
  tmp = KheDrsOpenChildMake(child_e, open_index);
  HaArrayAddLast(oc->open_children, tmp);  /* not really, just to make space */
  for( i = HaArrayCount(oc->open_children) - 2;  i >= 0;  i-- )
  {
    open_child = HaArray(oc->open_children, i);
    if( open_child.open_index <= open_index )
      break;
    HaArrayPut(oc->open_children, i + 1, open_child);
  }
  HaArrayPut(oc->open_children, i + 1, tmp);  /* for real this time */

  /* update oc->index_range, oc->child_indexes, and rev_cum_value_ub fields */
  KheDrsOpenChildrenUpdateIndexRange(oc);
  KheDrsOpenChildrenUpdateChildIndexes(oc);
  KheDrsOpenChildrenUpdateUpperBounds(oc);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenAddChildUpperBound(KHE_DRS_OPEN_CHILDREN oc,      */
/*    KHE_DRS_EXPR child_e)                                                  */
/*                                                                           */
/*  Add the upper bound of child_e to oc.                                    */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsOpenChildrenAddChildUpperBound(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_EXPR child_e)
{
  if( oc->float_ub )
    HaArrayFirst(oc->upper_bounds.f) += child_e->value_ub.f;
  else
    HaArrayFirst(oc->upper_bounds.i) += child_e->value_ub.i;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenAddChildInDayOrder(KHE_DRS_OPEN_CHILDREN oc,      */
/*    KHE_DRS_EXPR child_e)                                                  */
/*                                                                           */
/*  Add child_e to oc's list of open children, at a position that ensures    */
/*  that the children's day indexes are monotone non-decreasing.             */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsOpenChildrenAddChildInDayOrder(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_EXPR child_e)
{
  KHE_DRS_EXPR e;  int i, last;

  ** add child_e to oc->children in sorted position **
  HaArrayAddLast(oc->open_children, NULL);
  for( i = HaArrayCount(oc->open_children) - 2;  i >= 0;  i-- )
  {
    e = HaArray(oc->open_children, i);
    if( e->open_children_by_day.range.last <=
	child_e->open_children_by_day.range.last )
      break;
    HaArrayPut(oc->open_children, i + 1, e);
  }
  HaArrayPut(oc->open_children, i + 1, child_e);

  ** add child_e's last open day to oc->range **
  last = KheIntervalLast(child_e->open_children_by_day.range);
  oc->range = KheIntervalUnion(oc->range, KheIntervalMake(last, last));

  ** add in child_e's upper bound **
  KheDrsOpenChildrenAddChildUpperBound(oc, child_e);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenAddChildWithDayUpdate(KHE_DRS_OPEN_CHILDREN oc,   */
/*    KHE_DRS_EXPR child_e)                                                  */
/*                                                                           */
/*  Add child_e to the end of oc's list of open children, increasing         */
/*  child_e's open day index if required.                                    */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsOpenChildrenAddChildWithDayUpdate(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_EXPR child_e)
{
  KHE_DRS_EXPR e;  int last;

  ** increase child_e's last open day if required **
  if( HaArrayCount(oc->open_children) > 0 )
  {
    e = HaArrayLast(oc->open_children);
    if( e->open_children_by_day.range.last >
	child_e->open_children_by_day.range.last )
      child_e->open_children_by_day.range.last =
	e->open_children_by_day.range.last;
  }

  ** add child_e to the end of oc->children **
  HaArrayAddLast(oc->open_children, child_e);

  ** add child_e's last open day to e's open_day_range **
  last = KheIntervalLast(child_e->open_children_by_day.range);
  oc->range = KheIntervalUnion(oc->range, KheIntervalMake(last, last));

  ** add in child_e's upper bound **
  KheDrsOpenChildrenAddChildUpperBound(oc, child_e);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenAddChildInShiftOrder(KHE_DRS_OPEN_CHILDREN oc,    */
/*    KHE_DRS_EXPR child_e)                                                  */
/*                                                                           */
/*  Add child_e to oc's list of open children, at a position that ensures    */
/*  that the children's open shift indexes are monotone non-decreasing.      */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsOpenChildrenAddChildInShiftOrder(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_EXPR child_e)
{
  KHE_DRS_EXPR e;  int i, child_e_index;

  ** make sure child_e is an assigned task child **
  HnAssert(child_e->tag == KHE_DRS_EXPR_ASSIGNED_TASK_TAG,
    "KheDrsOpenChildrenAddChildInShiftOrder internal error (tag %d)",
    child_e->tag);

  ** add child_e to oc->children in sorted position **
  child_e_index = KheDrsExprOpenShiftIndex(child_e);
  HaArrayAddLast(oc->open_children, NULL);
  for( i = HaArrayCount(oc->open_children) - 2;  i >= 0;  i-- )
  {
    e = HaArray(oc->open_children, i);
    if( KheDrsExprOpenShiftIndex(e) <= child_e_index )
      break;
    HaArrayPut(oc->open_children, i + 1, e);
  }
  HaArrayPut(oc->open_children, i + 1, child_e);

  ** add child_e's shift index to oc->range **
  oc->range = KheIntervalUnion(oc->range,
    KheIntervalMake(child_e_index, child_e_index));

  ** add in child_e's upper bound *.
  KheDrsOpenChildrenAddChildUpperBound(oc, child_e);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenDeleteChild(KHE_DRS_OPEN_CHILDREN oc,             */
/*    KHE_DRS_EXPR child_e)                                                  */
/*                                                                           */
/*  Delete child_e from oc.                                                  */
/*                                                                           */
/*  Implementation note.  This code includes an inefficient linear search    */
/*  and inefficient updates to the child indexes and upper bounds.  But it   */
/*  happens only during closing so it doesn't matter.  Simplicity matters.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenDeleteChild(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_EXPR child_e)
{
  KHE_DRS_OPEN_CHILD open_child;  int i;

  HaArrayForEach(oc->open_children, open_child, i)
    if( open_child.child_e == child_e )
    {
      /* delete and shift here */
      HaArrayDeleteAndShift(oc->open_children, i);

      /* update oc->index_range, oc->child_indexes, and rev_cum_value_ub */
      KheDrsOpenChildrenUpdateIndexRange(oc);
      KheDrsOpenChildrenUpdateChildIndexes(oc);
      KheDrsOpenChildrenUpdateUpperBounds(oc);

      /* all done */
      return;
    }

  /* should never get here */
  HnAbort("KheDrsOpenChildrenDeleteChild internal error");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenDeleteLast(KHE_DRS_OPEN_CHILDREN oc)              */
/*                                                                           */
/*  Delete the last child of oc.                                             */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsOpenChildrenDeleteLast(KHE_DRS_OPEN_CHILDREN oc)
{
  HaArrayDeleteLast(oc->children);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenBuildDayChildIndexes(KHE_DRS_OPEN_CHILDREN oc)    */
/*                                                                           */
/*  Build the child indexes of oc based on the day ranges of the children.   */
/*  Also set the upper bounds array.                                         */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsOpenChildrenSetUpperBounds(KHE_DRS_OPEN_CHILDREN oc);

static void KheDrsOpenChildrenBuildDayChildIndexes(KHE_DRS_OPEN_CHILDREN oc)
{
  int index, i, open_index;  KHE_DRS_EXPR child_e;
  HaArrayClear(oc->child_indexes);
  index = oc->range.first - 1;
  HaArrayForEach(oc->open_children, child_e, i)
  {
    open_index = child_e->open_children_by_day.range.last;
    while( open_index > index )
    {
      index++;
      HaArrayAddLast(oc->child_indexes, i);

      ********************************************************************
      **                                                                **
      **  Here we have just added i at position index of child_indexes, **
      **  and the child at index i is the first to have a larger last   **
      **  index than index.                                             **
      **                                                                **
      ********************************************************************
    }
  }
  HaArrayAddLast(oc->child_indexes, i);

  ** also set upper bounds **
  KheDrsOpenChildrenSetUpperBounds(oc);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenBuildShiftChildIndexes(KHE_DRS_OPEN_CHILDREN oc)  */
/*                                                                           */
/*  Build the child indexes of oc based on the children's shift indexes.     */
/*  Also set the upper bounds array.                                         */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsOpenChildrenBuildShiftChildIndexes(KHE_DRS_OPEN_CHILDREN oc)
{
  int index, i, open_index;  KHE_DRS_EXPR child_e;
  HaArrayClear(oc->child_indexes);
  index = oc->range.first - 1;
  HaArrayForEach(oc->open_children, child_e, i)
  {
    open_index = KheDrsExprOpenShiftIndex(child_e);
    while( open_index > index )
    {
      index++;
      HaArrayAddLast(oc->child_indexes, i);

      ********************************************************************
      **                                                                **
      **  Here we have just added i at position index of child_indexes, **
      **  and the child at index i is the first to have a larger last   **
      **  index than index.                                             **
      **                                                                **
      ********************************************************************
    }
  }
  HaArrayAddLast(oc->child_indexes, i);

  ** also set upper bounds **
  KheDrsOpenChildrenSetUpperBounds(oc);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenBefore(KHE_DRS_OPEN_CHILDREN oc, int index)        */
/*                                                                           */
/*  Return the number of open children of oc whose index is less than the    */
/*  given index.  Here index is expected to satisfy                          */
/*                                                                           */
/*    oc->range.first <= index <= oc->range.last + 1                         */
/*                                                                           */
/*  If index = oc->range.last + 1, the number of open children is returned.  */
/*                                                                           */
/*  Implementation note.  We've removed the limits on index to help with     */
/*  evaluating shift assignment objects.  This is really work in progress.   */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenBefore(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  if( index < oc->index_range.first )
    return 0;
  else if( index > oc->index_range.last )
    return HaArrayCount(oc->open_children);
  else
    return HaArray(oc->child_indexes, index - oc->index_range.first);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenAtOrAfter(KHE_DRS_OPEN_CHILDREN oc, int index)     */
/*                                                                           */
/*  Return the number of open children of e whose index is index or more.    */
/*  Again, index is expected to satisfy                                      */
/*                                                                           */
/*    oc->range.first <= index <= oc->range.last + 1                         */
/*                                                                           */
/*  If index = oc->range.last + 1, 0 is returned.                            */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenAtOrAfter(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return HaArrayCount(oc->open_children) - KheDrsOpenChildrenBefore(oc, index);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenUpperBoundInt(KHE_DRS_OPEN_CHILDREN oc,            */
/*    int open_index)                                                        */
/*                                                                           */
/*  Return the reverse cumulative upper bound for the given open index       */
/*  (int version).  If it's off the end, return 0.                           */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenUpperBoundInt(KHE_DRS_OPEN_CHILDREN oc,
  int open_index)
{
  int i;
  HnAssert(!oc->float_ub, "KheDrsOpenChildrenUpperBoundInt internal error 1");
  if( open_index < oc->index_range.first )
    HnAbort("KheDrsOpenChildrenUpperBoundInt internal error 2");
  if( open_index > oc->index_range.last )
    return 0;
  else
  {
    i = HaArray(oc->child_indexes, open_index - oc->index_range.first);
    return HaArray(oc->open_children, i).rev_cum_value_ub.i;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenUpperBoundIntAll(KHE_DRS_OPEN_CHILDREN oc)         */
/*                                                                           */
/*  Return the sum of the upper bounds of all open children of oc.           */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenUpperBoundIntAll(KHE_DRS_OPEN_CHILDREN oc)
{
  HnAssert(!oc->float_ub,
    "KheDrsOpenChildrenUpperBoundIntAll internal error 1");
  if( HaArrayCount(oc->child_indexes) == 0 )
    return 0;
  else
    return HaArrayFirst(oc->open_children).rev_cum_value_ub.i;
}



/*****************************************************************************/
/*                                                                           */
/*  float KheDrsOpenChildrenUpperBoundFloat(KHE_DRS_OPEN_CHILDREN oc,        */
/*    int open_index)                                                        */
/*                                                                           */
/*  Return the reverse cumulative upper bound for the given open index       */
/*  (float version).  If it's off the end, return 0.0.                       */
/*                                                                           */
/*****************************************************************************/

static float KheDrsOpenChildrenUpperBoundFloat(KHE_DRS_OPEN_CHILDREN oc,
  int open_index)
{
  int i;
  HnAssert(oc->float_ub, "KheDrsOpenChildrenUpperBoundFloat internal error 1");
  if( open_index < oc->index_range.first )
    HnAbort("KheDrsOpenChildrenUpperBoundFloat internal error 2 (%d < %d)",
      open_index, oc->index_range.first);
  if( open_index > oc->index_range.last )
    return 0.0;
  else
  {
    i = HaArray(oc->child_indexes, open_index - oc->index_range.first);
    return HaArray(oc->open_children, i).rev_cum_value_ub.f;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  float KheDrsOpenChildrenUpperBoundFloatAll(KHE_DRS_OPEN_CHILDREN oc)     */
/*                                                                           */
/*  Return the sum of the upper bounds of all open children of oc.           */
/*                                                                           */
/*****************************************************************************/

static float KheDrsOpenChildrenUpperBoundFloatAll(KHE_DRS_OPEN_CHILDREN oc)
{
  HnAssert(oc->float_ub,
    "KheDrsOpenChildrenUpperBoundFloatAll internal error 1");
  if( HaArrayCount(oc->child_indexes) == 0 )
    return 0.0;
  else
    return HaArrayFirst(oc->open_children).rev_cum_value_ub.f;
}


/*****************************************************************************/
/*                                                                           */
/*  KheDrsOpenChildrenForEachIndex(oc, index)                                */
/*                                                                           */
/*  Iterator for visiting each index of oc.                                  */
/*                                                                           */
/*****************************************************************************/

#define KheDrsOpenChildrenForEachIndex(oc, i)			\
  for( i = (oc)->index_range.first;  i <= (oc)->index_range.last;  i++ )


/*****************************************************************************/
/*                                                                           */
/*  KheDrsOpenChildrenForEach(oc, index, x, i)                               */
/*                                                                           */
/*  Iterator over all the open children x of oc with the given index.        */
/*                                                                           */
/*****************************************************************************/

#define KheDrsOpenChildrenForEach(oc, index, x, i)			   \
  i1 = KheDrsOpenChildrenBefore((oc), (index));				   \
  i2 = KheDrsOpenChildrenBefore((oc), (index) + 1);			   \
  for( (i) = i1;							   \
   (i) < i2 ? ((x) = HaArray((oc)->open_children, (i)).child_e, true) : false;\
   (i)++ )


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOpenChildrenIndexInRange(KHE_DRS_OPEN_CHILDREN oc, int index) */
/*                                                                           */
/*  Return true if index is in the range of oc.                              */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOpenChildrenIndexInRange(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return index >= oc->index_range.first && index <= oc->index_range.last;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOpenChildrenIndexIsFirst(KHE_DRS_OPEN_CHILDREN oc, int index) */
/*                                                                           */
/*  Return true if index is the first open index of oc.                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOpenChildrenIndexIsFirst(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return index == oc->index_range.first;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOpenChildrenIndexIsFirstOrLess(KHE_DRS_OPEN_CHILDREN oc,      */
/*    int index)                                                             */
/*                                                                           */
/*  Return true if index is the first open index of oc, or less.             */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOpenChildrenIndexIsFirstOrLess(KHE_DRS_OPEN_CHILDREN oc,
  int index)
{
  return index <= oc->index_range.first;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOpenChildrenIndexIsLast(KHE_DRS_OPEN_CHILDREN oc, int index)  */
/*                                                                           */
/*  Return true if index is the last open index of oc.                       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOpenChildrenIndexIsLast(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return index == oc->index_range.last;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenWithIndex(KHE_DRS_OPEN_CHILDREN oc, int index)     */
/*                                                                           */
/*  Return the number of children in oc with this exact index.               */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenWithIndex(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return KheDrsOpenChildrenBefore(oc, index + 1) - 
    KheDrsOpenChildrenBefore(oc, index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenSetUpperBounds(KHE_DRS_OPEN_CHILDREN oc)          */
/*                                                                           */
/*  Set the upper bounds array of oc.                                        */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsOpenChildrenSetUpperBounds(KHE_DRS_OPEN_CHILDREN oc)
{
  int i, index, i1, i2;  KHE_DRS_EXPR child_e;
  if( oc->float_ub )
  {
    ** upper bounds are floats; total is in HaArrayFirst(oc->upper_bounds.f) **
    float val_f = HaArrayFirst(oc->upper_bounds.f);

    ** one value for each index **
    HaArrayClear(oc->upper_bounds.f);
    KheDrsOpenChildrenForEachIndex(oc, index)
    {
      ** add the value for index **
      HaArrayAddLast(oc->upper_bounds.f, val_f);

      ** take away the value upper bounds at this index **
      KheDrsOpenChildrenForEach(oc, index, child_e, i)
	val_f -= child_e->value_ub.f;
    }

    ** a sentinel value at the end **
    HaArrayAddLast(oc->upper_bounds.f, 0.0);
  }
  else
  {
    ** upper bounds are ints; total is in HaArrayFirst(oc->upper_bounds.i) **
    int val_i = HaArrayFirst(oc->upper_bounds.i);

    ** one value for each index **
    HaArrayClear(oc->upper_bounds.i);
    KheDrsOpenChildrenForEachIndex(oc, index)
    {
      ** add the value for index **
      HaArrayAddLast(oc->upper_bounds.i, val_i);

      ** take away the value upper bounds at this index **
      KheDrsOpenChildrenForEach(oc, index, child_e, i)
	val_i -= child_e->value_ub.i;
    }

    ** a sentinel value at the end **
    HaArrayAddLast(oc->upper_bounds.i, 0);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenUpperBoundsAtOrAfterInt(KHE_DRS_OPEN_CHILDREN oc,  */
/*    int index)                                                             */
/*                                                                           */
/*  Return the sum of the upper bounds of the open childen whose open child  */
/*  index is index or more.  This could be zero if index is off the end.     */
/*                                                                           */
/*  While children are being added, this is undefined for index > 0.  But    */
/*  it does work for index == 0:  it returns the total upper bound.          */
/*                                                                           */
/*****************************************************************************/

/* ***
static int KheDrsOpenChildrenUpperBoundsAtOrAfterInt(KHE_DRS_OPEN_CHILDREN oc,
  int index)
{
  HnAssert(!oc->float_ub,
    "KheDrsOpenChildrenUpperBoundsAtOrAfterInt internal error");
  HnAssert(0 <= index && index < HaArrayCount(oc->upper_bounds.i),
    "KheDrsOpenChildrenUpperBoundsAtOrAfterInt internal error:  index (%d)"
    "out of range (0 .. %d)", index, HaArrayCount(oc->upper_bounds.i) - 1);
  return HaArray(oc->upper_bounds.i, index);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  float KheDrsOpenChildrenUpperBoundsAtOrAfterFloat(                       */
/*    KHE_DRS_OPEN_CHILDREN oc, int index)                                   */
/*                                                                           */
/*  Return the sum of the upper bounds of the open childen whose open child  */
/*  index is index or more.  This could be zero if index is off the end.     */
/*                                                                           */
/*  While children are being added, this is undefined for index > 0.  But    */
/*  it does work for index == 0:  it returns the total upper bound.          */
/*                                                                           */
/*****************************************************************************/

/* ***
static float KheDrsOpenChildrenUpperBoundsAtOrAfterFloat(
  KHE_DRS_OPEN_CHILDREN oc, int index)
{
  HnAssert(oc->float_ub,
    "KheDrsOpenChildrenUpperBoundsAtOrAfterFloat internal error");
  HnAssert(0 <= index && index < HaArrayCount(oc->upper_bounds.f),
    "KheDrsOpenChildrenUpperBoundsAtOrAfterFloat internal error:  index (%d)"
    "out of range (0 .. %d)", index, HaArrayCount(oc->upper_bounds.f) - 1);
  return HaArray(oc->upper_bounds.f, index);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenDebug(KHE_DRS_OPEN_CHILDREN oc,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of oc.                                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenDebugIndexes(KHE_DRS_OPEN_CHILDREN oc, FILE *fp)
{
  int val, i;
  fprintf(fp, "[%d open children, open range %d-%d: ",
    HaArrayCount(oc->open_children), oc->index_range.first,
    oc->index_range.last);
  HaArrayForEach(oc->child_indexes, val, i)
  {
    if( i > 0 )
      fprintf(fp, ",");
    fprintf(fp, "%d", val);
  }
  fprintf(fp, "]");
}

static void KheDrsOpenChildrenDebug(KHE_DRS_OPEN_CHILDREN oc,
  int verbosity, int indent, FILE *fp)
{
  int i;  KHE_DRS_OPEN_CHILD open_child;
  if( indent >= 0 )
  {
    if( indent > 0 )
      fprintf(fp, "%*s", indent, "");
    fprintf(fp, "[ ");
    KheDrsOpenChildrenDebugIndexes(oc, fp);
    fprintf(fp, "\n");
    HaArrayForEach(oc->open_children, open_child, i)
      KheDrsExprDebug(open_child.child_e, NULL, verbosity, indent + 2, fp);
    if( indent > 0 )
      fprintf(fp, "%*s", indent, "");
    fprintf(fp, "]\n");
  }
  else
    KheDrsOpenChildrenDebugIndexes(oc, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - construction"                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprDebugWanted(KHE_DRS_EXPR e)                               */
/*                                                                           */
/*  Return true if e is wanted for debugging, according to DEBUG58, or       */
/*  is a child of a wanted expression.                                       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprDebugWanted(KHE_DRS_EXPR e)
{
  KHE_DRS_EXPR_ASSIGNED_TASK eat;  KHE_TASK task;
  KHE_DRS_EXPR_COUNTER ec;
  if( DEBUG80 && e->tag == KHE_DRS_EXPR_ASSIGNED_TASK_TAG )
  {
    eat = (KHE_DRS_EXPR_ASSIGNED_TASK) e;
    task = eat->task_on_day->task;
    return task != NULL && strcmp(KheTaskId(task), DEBUG80_TASK_ID) == 0;
  }
  if( DEBUG91_WANTED && e->tag == KHE_DRS_EXPR_COUNTER_TAG )
  {
    ec = (KHE_DRS_EXPR_COUNTER) e;
    return DEBUG91(ec->monitor);
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprInitBegin(KHE_DRS_EXPR e, KHE_DRS_EXPR_TAG tag,           */
/*    KHE_DRS_RESOURCE dr, KHE_DRS_VALUE value_ub,                           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Initialize those fields of e common to all expressions.                  */
/*                                                                           */
/*  Except e->value and e->value_ub are left uninitialized; e->value is      */
/*  initialized by KheDrsExprInitEnd below, and e->value_ub is initialized   */
/*  by type-specific code alongside its call to KheDrsExprInitEnd.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprInitBegin(KHE_DRS_EXPR e, KHE_DRS_EXPR_TAG tag,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_OPEN_CHILDREN_INDEX_TYPE index_type;  float float_ub;
  e->tag = tag;
  e->gathered = false;
  e->open = false;
  e->postorder_index = -1;
  e->resource = dr;
  /* e->value and e->value_ub are not initialized here */
  HaArrayInit(e->parents, drs->arena);
  HaArrayInit(e->children, drs->arena);
  index_type = (tag == KHE_DRS_EXPR_SEQUENCE_TAG ?
    KHE_DRS_OPEN_CHILDREN_INDEX_DAY_ADJUSTED : KHE_DRS_OPEN_CHILDREN_INDEX_DAY);
  float_ub = (tag == KHE_DRS_EXPR_SUM_FLOAT_TAG);
  KheDrsOpenChildrenInit(&e->open_children_by_day, index_type, float_ub, drs);
  HaArrayInit(e->sig_indexes, drs->arena);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAddChild(KHE_DRS_EXPR parent, KHE_DRS_EXPR child,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make child a child of parent.                                            */
/*                                                                           */
/*  This function includes a switch on the expression type and then calls a  */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAddChild(KHE_DRS_EXPR parent, KHE_DRS_EXPR child,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprInitEnd(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Carry out the parts of finalizing the initialization that are common to  */
/*  all expression types:  setting the postorder index and closed value.     */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExprInitEnd(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* postorder index */
  e->postorder_index = drs->postorder_count++;

  /* closed value; e->value is initialized here */
  KheDrsExprSetClosedValue(e, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsParentIsFirstCounterParent(KHE_DRS_EXPR child,                */
/*    KHE_DRS_EXPR parent)                                                   */
/*                                                                           */
/*  Return true if parent is the first int sum cost parent of child.         */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsParentIsFirstCounterParent(KHE_DRS_EXPR child,
  KHE_DRS_EXPR parent)
{
  KHE_DRS_PARENT prnt;  int i;
  HaArrayForEach(child->parents, prnt, i)
    if( prnt.expr->tag == KHE_DRS_EXPR_COUNTER_TAG )
      return prnt.expr == parent;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprHasParent(KHE_DRS_EXPR child, KHE_DRS_EXPR_TAG tag)       */
/*                                                                           */
/*  Return true if child has a parent of the type given by tag.              */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprHasParent(KHE_DRS_EXPR child, KHE_DRS_EXPR_TAG tag)
{
  KHE_DRS_PARENT prnt;  int i;
  HaArrayForEach(child->parents, prnt, i)
    if( prnt.expr->tag == tag )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - opening"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCheckClosed(KHE_DRS_EXPR e,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Check e's closed state.  This function was written to help with          */
/*  tracking down a nasty bug.                                               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprCounterCheckClosed(KHE_DRS_EXPR_COUNTER ec,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExprCheckClosed(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( DEBUG79 )
  {
    switch( e->tag )
    {
      case KHE_DRS_EXPR_COUNTER_TAG:

	KheDrsExprCounterCheckClosed((KHE_DRS_EXPR_COUNTER) e, drs);
	break;

      default:

	/* not checking other kinds */
	break;
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprGatherForOpening(KHE_DRS_EXPR e,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Gather e into drs->open_exprs, the list of expressions that are to be    */
/*  opened.  This does not actually open e; that comes later.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprGatherForOpening(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;  int i;
  if( !e->gathered )
  {
    e->gathered = true;
    HaArrayAddLast(drs->open_exprs, e);
    HaArrayForEach(e->parents, prnt, i)
      KheDrsExprGatherForOpening(prnt.expr, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprPostorderCmp(const void *t1, const void *t2)               */
/*                                                                           */
/*  Comparison function for sorting an array of expressions by increasing    */
/*  postorder index.                                                         */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprPostorderCmp(const void *t1, const void *t2)
{
  KHE_DRS_EXPR e1 = * (KHE_DRS_EXPR *) t1;
  KHE_DRS_EXPR e2 = * (KHE_DRS_EXPR *) t2;
  return e1->postorder_index - e2->postorder_index;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprChildHasOpened(KHE_DRS_EXPR e, KHE_DRS_EXPR child_e,      */
/*    int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                      */
/*                                                                           */
/*  As part of opening e for solving, inform e that its child_index'th       */
/*  child, child_e, has opened.                                              */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprChildHasOpened(KHE_DRS_EXPR e, KHE_DRS_EXPR child_e,
  int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOpen(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)     */
/*                                                                           */
/*  Open e for solving.                                                      */
/*                                                                           */
/*  By the time this is called, all of e's open children will have           */
/*  called KheDrsExprChildHasOpened.                                         */
/*                                                                           */
/*  Implementation note - how e->open_day_range is set                       */
/*  --------------------------------------------------                       */
/*  In external expressions, e->open_day_range receives its final value      */
/*  when KheDrsExprGatherForOpening is called.  In internal expressions,     */
/*  its value is initially empty and is updated by each call to              */
/*  KheDrsExprChildHasOpened until it reaches its final value.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOpen(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;  int i;  bool debug;

  debug = KheDrsExprDebugWanted(e);
  if( debug )
  {
    fprintf(stderr, "[ KheDrsExprOpen\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
  }

  /* inform e's parents that e is now open */
  e->gathered = false;
  HnAssert(!e->open, "KheDrsExprOpen internal error");
  e->open = true;
  HaArrayForEach(e->parents, prnt, i)
    KheDrsExprChildHasOpened(prnt.expr, e, prnt.index, drs);

  /* what to do now depends on whether e is external or internal, and if */
  /* internal whether from an event resource or resource constraint */
  if( e->tag <= KHE_DRS_EXPR_WORK_DAY_TAG )
  {
    /* external expression; clear its value */
    KheDrsExprLeafClear(e, drs);
  }
  else
  {
    /* internal expression, build day child indexes */
    /* *** done on the fly now
    KheDrsOpenChildrenBuildDayChildIndexes(&e->open_children_by_day);
    *** */
  }

  if( debug )
  {
    fprintf(stderr, "  KheDrsExprOpen returning; e is now:\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
    fprintf(stderr, "] KheDrsExprOpen\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - dom tests and notifying signers"               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprDomTest(KHE_DRS_EXPR e,                       */
/*    int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return a dom test suited to e, taking account not just of e itself       */
/*  but also of its context.                                                 */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprDomTest(KHE_DRS_EXPR e,
  int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprChildDomTest(KHE_DRS_EXPR e,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return a dom test suited to a child of e.                                */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsExprDomTest
static KHE_DRS_DOM_TEST KheDrsExprChildDomTest(KHE_DRS_EXPR e,
  int open_day_index, KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs);
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e,                    */
/*    int open_child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return a dom test suited to e on its open_child_index'th open day.       */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsExprDomTest
static KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e,
  int open_child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs);
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprNeedsDayEval(KHE_DRS_EXPR e, KHE_DRS_DAY day,             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_DOM_TEST *dom_test)           */
/*                                                                           */
/*  If e needs to be evaluated by the signer for day, return true and set    */
/*  *dom_test to the dominance test the signer needs, or to NULL if no       */
/*  dominance test is needed.                                                */
/*                                                                           */
/*  Implementation note.  The values of day for which this function is       */
/*  called are such that the result is always true, so only the dominance    */
/*  test part needs actual work.  We do it this way for consistency with     */
/*  similar functions for other signers.                                     */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsExprNeedsDayEval(KHE_DRS_EXPR e, KHE_DRS_DAY day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_DOM_TEST *dom_test)
{
  int di;
  di = day->open_day_index;
  if( KheDrsOpenChildrenIndexIsLast(&e->open_children_by_day, di) )
    *dom_test = NULL;
  else
    *dom_test = KheDrsExprDomTest(e, di, drs);
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprNotifyResourceSigners(KHE_DRS_EXPR e,                     */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Do the work of KheDrsExprNotifySigners (just below) for the case         */
/*  where e is an internal expression derived from a resource monitor.       */
/*  In this case e needs to be added to the resource on day signer for       */
/*  its resource on each of its open days.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprNotifyResourceSigners(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY day;  KHE_DRS_RESOURCE_ON_DAY drd;  int di, sig_index;
  /* KHE_DRS_DOM_TEST dom_test; */  KHE_DRS_SIGNER dsg;  bool debug;
  KHE_DRS_EXPR_COUNTER ec;
  if( DEBUG91_WANTED && e->tag == KHE_DRS_EXPR_COUNTER_TAG )
  {
    ec = (KHE_DRS_EXPR_COUNTER) e;
    debug = DEBUG91(ec->monitor);
  }
  else
  {
    ec = NULL;
    debug = false;
  }
  HaArrayClear(e->sig_indexes);
  KheDrsOpenChildrenForEachIndex(&e->open_children_by_day, di)
  {
    day = HaArray(drs->open_days, di);
    drd = KheDrsResourceOnDay(e->resource, day);
    dsg = KheDrsResourceOnDaySigner(drd);
    if( KheDrsSignerAddExpr(dsg, e, drs, &sig_index) )
    {
      if( debug )
	fprintf(stderr, "  %s receiving %s dom test (index %d)\n",
	  KheDrsMonitorId(ec->monitor), KheDrsDayId(day), sig_index);
      HaArrayAddLast(e->sig_indexes, sig_index);
    }
    /* ***
    if( KheDrsExprNeedsDayEval(e, day, drs, &dom_test) )
    {
      if( KheDrsSignerAddExpr(dsg, e, drs, &sig_index) )
	HaArrayAddLast(e->sig_indexes, sig_index);
      KheDrsResourceOnDayAddOpenExpr(drd, e);
      if( dom_test != NULL )
	HaArrayAddLast(e->sig_indexes,
	  KheDrsResourceOnDayAddDomTest(drd, dom_test, drs));
    }
    *** */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprNotifySigners(KHE_DRS_EXPR e,                             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Notify all relevant signers that e has opened; they will then enrol e    */
/*  in their lists of expressions, and optionally add a dom test for e.      */
/*                                                                           */
/*  By the time this is called, every expression that is going to open       */
/*  will be open, including knowing what all of its open children are.       */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprCounterNotifyCoverSigners(
  KHE_DRS_EXPR_COUNTER ec, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExprNotifySigners(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  bool debug;

  debug = KheDrsExprDebugWanted(e);
  if( debug )
  {
    fprintf(stderr, "[ KheDrsExprNotifySigners\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
  }

  if( e->tag <= KHE_DRS_EXPR_WORK_DAY_TAG )
  {
    /* external expression; nothing to do */
  }
  else if( e->resource != NULL )
  {
    /* add e to its resource on day signers */
    KheDrsExprNotifyResourceSigners(e, drs);
  }
  else
  {
    /* add e to its cover signers */
    HnAssert(e->tag == KHE_DRS_EXPR_COUNTER_TAG,
      "KheDrsExprNotifySigners: internal error");
    KheDrsExprCounterNotifyCoverSigners((KHE_DRS_EXPR_COUNTER) e, drs);
  }
  if( debug )
  {
    fprintf(stderr, "  KheDrsExprNotifySigners returning; e is now:\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
    fprintf(stderr, "] KheDrsExprNotifySigners\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - accessing signatures"                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_VALUE KheDrsExprDaySigVal(KHE_DRS_EXPR e, int open_day_index,    */
/*    KHE_DRS_SOLN soln)                                                     */
/*                                                                           */
/*  Retrieve from soln the signature of e on open day open_day_index.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_VALUE KheDrsExprDaySigVal(KHE_DRS_EXPR e, int open_day_index,
  KHE_DRS_SIGNATURE sig)
{
  int pos;
  pos = HaArray(e->sig_indexes, open_day_index -
    e->open_children_by_day.index_range.first);
  return HaArray(sig->states, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - closing"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprChildHasClosed(KHE_DRS_EXPR e,                            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform internal expression e that its child_index'th child, child_e,     */
/*  has closed.  The child will have its correct closed_state and value      */
/*  fields when this is called.                                              */
/*                                                                           */
/*  Although KheDrsExprChildHasOpened adds child_e to e->open_children,      */
/*  KheDrsExprChildHasClosed does not remove child_e from e->open_children.  */
/*  This is because children are closed in the same order they were opened,  */
/*  but there is no efficient way to remove them in this order, short of     */
/*  something too complicated such as a circular list.                       */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprChildHasClosed(KHE_DRS_EXPR e,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing e, set its value suitably for its closed state.       */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprClose(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Close e, which may be internal or external.  Whatever it needs to work   */
/*  out its closed value (an assignment, closed children, or closed state)   */
/*  is up to date.  At the end, inform e's parents that e has closed.        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprClose(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;  int i;  KHE_DRS_EXPR_COUNTER ec;

  /* set e's closed value */
  if( KheDrsExprDebugWanted(e) )
  {
    fprintf(stderr, "  KheDrsExprClose(e), e =\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
  }
  HnAssert(e->open, "KheDrsExprClose internal error");
  KheDrsExprSetClosedValue(e, drs);

  /* clear fields that are used only when e is open */
  KheDrsOpenChildrenClear(&e->open_children_by_day);
  HaArrayClear(e->sig_indexes);
  if( e->tag == KHE_DRS_EXPR_COUNTER_TAG )
  {
    ec = (KHE_DRS_EXPR_COUNTER) e;
    KheDrsOpenChildrenClear(&ec->open_children_by_shift);
  }

  /* close e and inform e's parents that e has closed */
  e->open = false;
  HaArrayForEach(e->parents, prnt, i)
    KheDrsExprChildHasClosed(prnt.expr, e, prnt.index, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - searching"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,          */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Inform leaf expression e that dtd has been assigned drd.                 */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprLeafClear(KHE_DRS_EXPR e)                                 */
/*                                                                           */
/*  Inform leaf expression e that the assignment of drd to dtd carried out   */
/*  by KheDrsExprLeafSet above has been removed.                             */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprLeafClear(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,         */
/*    KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE sig,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)                           */
/*                                                                           */
/*  Evaluate expression e on day next_di, which is one of e's open days.     */
/*  The results go into sig.                                                 */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, /* int next_di, */ KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug);


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - debug"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp);                            */
/*                                                                           */
/*  Unindented debug print of that part of e which is specific to it.        */
/*  If soln != NULL, also print e's signature (if any) in soln.              */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDebug(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs,    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of e onto fp with the given verbosity and indent.            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprDebug(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_EXPR child_e;  int i, oc_count, div;
  if( indent >= 0 )
  {
    div = (drs == NULL ? -1 : KheSolnDiversifier(drs->soln));
    fprintf(fp, "%d# %*s[ ", div, indent, "");
    KheDrsExprDoDebug(e, NULL, drs, fp);
    fprintf(fp, " p %p pi %d %s", (void *) e, e->postorder_index,
      e->open ? "open" : "closed");
    oc_count = KheDrsOpenChildrenCount(&e->open_children_by_day);
    if( oc_count > 0 )
      fprintf(fp, " (oc %d)", oc_count);
    if( verbosity >= 2 && HaArrayCount(e->children) > 0 )
    {
      fprintf(fp, "\n");
      HaArrayForEach(e->children, child_e, i)
	KheDrsExprDebug(child_e, drs, verbosity, indent + 2, fp);
      fprintf(fp, "%d# %*s]\n", div, indent, "");
    }
    else
      fprintf(fp, " ]\n");
  }
  else
    KheDrsExprDoDebug(e, NULL, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDebugSignature(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,         */
/*    bool is_float, FILE *fp)                                               */
/*                                                                           */
/*  If soln != NULL and e has a signature entry in soln, debug that entry.   */
/*                                                                           */
/*  NB soln == NULL does not mean that there is a valid solution whose       */
/*  value is NULL, it just means that we are doing this debug without the    */
/*  benefit of a current solution, and so this function has nothing to do.   */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_DAY KheDrsSolnDay(KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExprDebugSignature(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,
  bool is_float, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  if( soln != NULL )
  {
    fprintf(fp, "  KheDrsExprDebugSignature decommissioned");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_ASSIGNED_TASK"                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_ASSIGNED_TASK KheDrsExprAssignedTaskMake(                   */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_RESOURCE_GROUP rg,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make an assigned task expression object with these attributes.           */
/*  Or if there is already such an expression in dtd's list, use that.       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_ASSIGNED_TASK KheDrsExprAssignedTaskMake(
  KHE_DRS_TASK_ON_DAY dtd, KHE_RESOURCE_GROUP rg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_ASSIGNED_TASK res;

  /* MakeBegin part */
  if( KheDrsTaskOnDayHasAssignedTaskExpr(dtd, rg, &res) )
    return res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_ASSIGNED_TASK_TAG,
    NULL, /* KheDrsValueInt(1), */ drs);
  res->task_on_day = dtd;
  res->resource_group = rg;

  /* MakeEnd part */
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  res->value_ub.i = 1;
  KheDrsTaskOnDayAddAssignedTaskExpr(dtd, res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprAssignedTaskValue(KHE_DRS_EXPR_ASSIGNED_TASK eat,          */
/*    KHE_RESOURCE r)                                                        */
/*                                                                           */
/*  Return the value of eat when its task is assigned r (possibly NULL).     */
/*  This will be 1 when r is non-NULL and lies in eat's resource group.      */
/*  For efficiency, eat's resource group may be NULL, meaning all resources. */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprAssignedTaskValue(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_RESOURCE r)
{
  KHE_RESOURCE_GROUP rg;
  if( r == NULL )
    return 0;
  rg = eat->resource_group;
  return (rg == NULL || KheResourceGroupContains(rg, r)) ? 1 : 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskCheckClosed(KHE_DRS_EXPR_ASSIGNED_TASK eat,   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Check closed value in parent.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskCheckClosed(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;  int i;
  if( DEBUG79 )
    HaArrayForEach(eat->parents, prnt, i)
      KheDrsExprCheckClosed(prnt.expr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskLeafSet(KHE_DRS_EXPR_ASSIGNED_TASK eat,       */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Update the value of eat to reflect the assignment of dr to dtd.  Here    */
/*  dtd is known to be the relevant task (which is why it is unused), and    */
/*  dr is known to represent a non-NULL resource, not a non-assignment.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskLeafSet(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eat->value.i = KheDrsExprAssignedTaskValue(eat, dr->resource);
  KheDrsExprAssignedTaskCheckClosed(eat, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskLeafClear(KHE_DRS_EXPR_ASSIGNED_TASK eat,     */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Inform leaf expression ebt that the assignment of drd to dtd carried     */
/*  out by KheDrsExprAssignedTaskLeafSet above has been removed.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskLeafClear(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eat->value.i = 0;
  KheDrsExprAssignedTaskCheckClosed(eat, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskSetClosedValue(KHE_DRS_EXPR_ASSIGNED_TASK eat,*/
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eat, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskSetClosedValue(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_ON_DAY drd;  KHE_RESOURCE r;
  drd = eat->task_on_day->closed_drd;
  r = (drd == NULL ? NULL : drd->encl_dr->resource);
  eat->value.i = KheDrsExprAssignedTaskValue(eat, r);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskDoDebug(KHE_DRS_EXPR_ASSIGNED_TASK eat,       */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eat which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskDoDebug(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  KHE_RESOURCE r;
  fprintf(fp, "ASSIGNED_TASK(");
  HnAssert(eat->task_on_day->task != NULL,
    "KheDrsExprAssignedTaskDoDebug internal error");
  KheTaskDebug(eat->task_on_day->task, 1, -1, fp);
  r = KheTaskAsstResource(eat->task_on_day->task);
  fprintf(fp, " := %s, %s, val %d)", r == NULL ? "-" : KheResourceId(r),
    eat->resource_group == NULL ? "-" :
    KheResourceGroupId(eat->resource_group), eat->value.i);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eat, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_BUSY_TIME"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_BUSY_TIME KheDrsExprBusyTimeMake(                           */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_BUSY_TIME object with these attributes.          */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_BUSY_TIME KheDrsExprBusyTimeMake(
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_BUSY_TIME res;  KHE_DRS_EXPR e;

  /* MakeBegin part */
  if( KheDrsResourceOnDayHasTimeExpr(drd, KHE_DRS_EXPR_BUSY_TIME_TAG,
	time, drs, &e) )
    return (KHE_DRS_EXPR_BUSY_TIME) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_BUSY_TIME_TAG,
    drd->encl_dr, /* KheDrsValueInt(1), */ drs);
  res->resource_on_day = drd;
  res->time = time;

  /* MakeEnd part */
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  res->value_ub.i = 1;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyTimeLeafSet(KHE_DRS_EXPR_BUSY_TIME ebt,               */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of ebt, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as ebt is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyTimeLeafSet(KHE_DRS_EXPR_BUSY_TIME ebt,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  ebt->value.i = (dtd->time == ebt->time ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyTimeLeafClear(KHE_DRS_EXPR_BUSY_TIME ebt)             */
/*                                                                           */
/*  Inform leaf expression ebt that the assignment of drd to dtd carried     */
/*  out by KheDrsExprBusyTimeLeafSet above has been removed.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyTimeLeafClear(KHE_DRS_EXPR_BUSY_TIME ebt)
{
  ebt->value.i = 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyTimeSetClosedValue(KHE_DRS_EXPR_BUSY_TIME ebt,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ebt, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyTimeSetClosedValue(KHE_DRS_EXPR_BUSY_TIME ebt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = ebt->resource_on_day->closed_dtd;
  ebt->value.i = (dtd != NULL && dtd->time == ebt->time ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyTimeDoDebug(KHE_DRS_EXPR_BUSY_TIME ebt,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ebt which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyTimeDoDebug(KHE_DRS_EXPR_BUSY_TIME ebt,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "BUSY_TIME(%s, %s, iv %d)",
    KheDrsResourceId(ebt->resource_on_day->encl_dr),
    KheTimeId(ebt->time), ebt->value.i);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ebt, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_FREE_TIME"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_FREE_TIME KheDrsExprFreeTimeMake(                           */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_FREE_TIME object with these attributes.          */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_FREE_TIME KheDrsExprFreeTimeMake(
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_FREE_TIME res;  KHE_DRS_EXPR e;

  /* MakeBegin part */
  if( KheDrsResourceOnDayHasTimeExpr(drd, KHE_DRS_EXPR_FREE_TIME_TAG,
      time, drs, &e) )
    return (KHE_DRS_EXPR_FREE_TIME) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_FREE_TIME_TAG,
    drd->encl_dr, /* KheDrsValueInt(1), */ drs);
  res->resource_on_day = drd;
  res->time = time;

  /* MakeEnd part */
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  res->value_ub.i = 1;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeTimeLeafSet(KHE_DRS_EXPR_FREE_TIME eft,               */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of eft, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as eft is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeTimeLeafSet(KHE_DRS_EXPR_FREE_TIME eft,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  eft->value.i = (dtd->time == eft->time ? 0 : 1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeTimeLeafClear(KHE_DRS_EXPR_FREE_TIME eft)             */
/*                                                                           */
/*  Inform leaf expression eft that the assignment of drd to dtd carried     */
/*  out by KheDrsExprFreeTimeLeafSet above has been removed.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeTimeLeafClear(KHE_DRS_EXPR_FREE_TIME eft)
{
  eft->value.i = 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeTimeSetClosedValue(KHE_DRS_EXPR_FREE_TIME eft,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eft, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeTimeSetClosedValue(KHE_DRS_EXPR_FREE_TIME eft,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = eft->resource_on_day->closed_dtd;
  eft->value.i = (dtd != NULL && dtd->time == eft->time ? 0 : 1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeTimeDoDebug(KHE_DRS_EXPR_FREE_TIME eft,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eft which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeTimeDoDebug(KHE_DRS_EXPR_FREE_TIME eft,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "FREE_TIME(%s, %s, iv %d)",
    KheDrsResourceId(eft->resource_on_day->encl_dr),
    KheTimeId(eft->time), eft->value.i);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eft, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_WORK_TIME"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  float KheDrsMaxWorkloadAtTime(KHE_DYNAMIC_RESOURCE_SOLVER drs,           */
/*    KHE_TIME t)                                                            */
/*                                                                           */
/*  Return the maximum workload available to a resource at time t.           */
/*                                                                           */
/*****************************************************************************/

static float KheDrsMaxWorkloadAtTime(KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_TIME t)
{
  return HaArray(drs->max_workload_per_time, KheTimeIndex(t));
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_WORK_TIME KheDrsExprWorkTimeMake(                           */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_WORK_TIME object with these attributes.          */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_WORK_TIME KheDrsExprWorkTimeMake(
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_WORK_TIME res;  KHE_DRS_EXPR e;

  /* MakeBegin part */
  if( KheDrsResourceOnDayHasTimeExpr(drd, KHE_DRS_EXPR_WORK_TIME_TAG,
	time, drs, &e) )
    return (KHE_DRS_EXPR_WORK_TIME) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_WORK_TIME_TAG,
    drd->encl_dr, drs);
  res->resource_on_day = drd;
  res->time = time;

  /* MakeEnd part */
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  res->value_ub.f = KheDrsMaxWorkloadAtTime(drs, time);
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkTimeLeafSet(KHE_DRS_EXPR_WORK_TIME ewt,               */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of ewt, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as ewt is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkTimeLeafSet(KHE_DRS_EXPR_WORK_TIME ewt,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  ewt->value.f =
    (dtd->time == ewt->time ? KheTaskWorkloadPerTime(dtd->task) : 0.0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkTimeLeafClear(KHE_DRS_EXPR_WORK_TIME ewt)             */
/*                                                                           */
/*  Inform leaf expression ebt that the assignment of drd to dtd carried     */
/*  out by KheDrsExprWorkTimeLeafSet above has been removed.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkTimeLeafClear(KHE_DRS_EXPR_WORK_TIME ewt)
{
  ewt->value.f = 0.0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkTimeSetClosedValue(KHE_DRS_EXPR_WORK_TIME ewt,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ewt, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkTimeSetClosedValue(KHE_DRS_EXPR_WORK_TIME ewt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = ewt->resource_on_day->closed_dtd;
  ewt->value.f = (dtd != NULL && dtd->time == ewt->time ?
    KheTaskWorkloadPerTime(dtd->task) : 0.0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkTimeDoDebug(KHE_DRS_EXPR_WORK_TIME ewt,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ewt which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkTimeDoDebug(KHE_DRS_EXPR_WORK_TIME ewt,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "WORK_TIME(%s, %s)",
    KheDrsResourceId(ewt->resource_on_day->encl_dr), KheTimeId(ewt->time));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ewt, soln, true, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_BUSY_DAY"                                        */
/*                                                                           */
/*****************************************************************************/

#define BUSY_DAY_DEBUG 0
#define BUSY_DAY_TIME_GROUP_ID "1Sat"
#define BUSY_DAY_RESOURCE "TR_25"


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsBusyDayDebug(KHE_DRS_EXPR_BUSY_DAY ebd)                       */
/*                                                                           */
/*  Return true if we want to debug ebd.                                     */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheDrsBusyDayDebug(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_RESOURCE r;  KHE_TIME_GROUP tg;
  if( RERUN && drs->rerun != NULL && BUSY_DAY_DEBUG )
  {
    r = ebd->resource_on_day->encl_dr->resource;
    tg = ebd->time_group;
    return strcmp(KheResourceId(r), BUSY_DAY_RESOURCE) == 0 &&
      strcmp(KheTimeGroupId(tg), BUSY_DAY_TIME_GROUP_ID) == 0;
  }
  else
    return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_BUSY_DAY KheDrsExprBusyDayMake(KHE_DRS_RESOURCE_ON_DAY drd, */
/*    KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)                    */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_BUSY_DAY object with these attributes.           */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_BUSY_DAY KheDrsExprBusyDayMake(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_BUSY_DAY res;  KHE_DRS_EXPR e;

  /* MakeBegin part */
  if( KheDrsResourceOnDayHasDayExpr(drd, KHE_DRS_EXPR_BUSY_DAY_TAG, tg,drs,&e) )
    return (KHE_DRS_EXPR_BUSY_DAY) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_BUSY_DAY_TAG,
    drd->encl_dr, /* KheDrsValueInt(1), */ drs);
  res->resource_on_day = drd;
  res->time_group = tg;

  /* MakeEnd part */
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  res->value_ub.i = 1;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyDayLeafSet(KHE_DRS_EXPR_BUSY_DAY ebd,                 */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of ebd, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as ebt is.               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprBusyDayDoDebug(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp);

static void KheDrsExprBusyDayLeafSet(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  ebd->value.i = 1;
  /* ***
  if( KheDrsBusyDayDebug(ebd, drs) )
  {
    fprintf(stderr, "  KheDrsExprBusyDayLeafSet(");
    KheDrsExprBusyDayDoDebug(ebd, NULL, drs, stderr);
    fprintf(stderr, ")\n");
  }
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyDayLeafClear(KHE_DRS_EXPR_BUSY_DAY ebd)               */
/*                                                                           */
/*  Inform leaf soln ebd that the assignment of drd to dtd carried out by    */
/*  KheDrsExprBusyDayLeafSet above has been removed.                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyDayLeafClear(KHE_DRS_EXPR_BUSY_DAY ebd)
{
  ebd->value.i = 0;
  /* ***
  if( KheDrsBusyDayDebug(ebd, drs) )
  {
    fprintf(stderr, "  KheDrsExprBusyDayLeafClear(");
    KheDrsExprBusyDayDoDebug(ebd, NULL, drs, stderr);
    fprintf(stderr, ")\n");
  }
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyDaySetClosedValue(KHE_DRS_EXPR_BUSY_DAY ebd,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ebd, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyDaySetClosedValue(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = ebd->resource_on_day->closed_dtd;
  ebd->value.i = (dtd != NULL ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyDayDoDebug(KHE_DRS_EXPR_BUSY_DAY ebd,                 */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ebd which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyDayDoDebug(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "BUSY_DAY(%s, %s, iv %d)",
    KheDrsResourceId(ebd->resource_on_day->encl_dr),
    KheTimeGroupId(ebd->time_group), ebd->value.i);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ebd, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_FREE_DAY"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_FREE_DAY KheDrsExprFreeDayMake(KHE_DRS_RESOURCE_ON_DAY drd, */
/*    KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)                    */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_FREE_DAY object with these attributes.           */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_FREE_DAY KheDrsExprFreeDayMake(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_FREE_DAY res;  KHE_DRS_EXPR e;

  /* MakeBegin part */
  if( KheDrsResourceOnDayHasDayExpr(drd, KHE_DRS_EXPR_FREE_DAY_TAG, tg,drs,&e) )
    return (KHE_DRS_EXPR_FREE_DAY) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_FREE_DAY_TAG,
    drd->encl_dr, /* KheDrsValueInt(1), */ drs);
  res->resource_on_day = drd;
  res->time_group = tg;

  /* MakeEnd part */
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  res->value_ub.i = 1;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeDayLeafSet(KHE_DRS_EXPR_FREE_DAY efd,                 */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of efd, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as efd is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeDayLeafSet(KHE_DRS_EXPR_FREE_DAY efd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  efd->value.i = 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeDayLeafClear(KHE_DRS_EXPR_FREE_DAY efd)               */
/*                                                                           */
/*  Inform leaf expression eft that the assignment of drd to dtd carried     */
/*  out by KheDrsExprFreeDayLeafSet above has been removed.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeDayLeafClear(KHE_DRS_EXPR_FREE_DAY efd)
{
  efd->value.i = 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeDaySetClosedValue(KHE_DRS_EXPR_FREE_DAY efd,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing efd, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeDaySetClosedValue(KHE_DRS_EXPR_FREE_DAY efd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = efd->resource_on_day->closed_dtd;
  efd->value.i = (dtd != NULL ? 0 : 1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeDayDoDebug(KHE_DRS_EXPR_FREE_DAY efd,                 */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of efd which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeDayDoDebug(KHE_DRS_EXPR_FREE_DAY efd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "FREE_DAY(%s, %s, iv %d)",
    KheDrsResourceId(efd->resource_on_day->encl_dr),
    KheTimeGroupId(efd->time_group), efd->value.i);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) efd, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_WORK_DAY"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  float KheDrsMaxWorkloadOnDay(KHE_DYNAMIC_RESOURCE_SOLVER drs,            */
/*    KHE_TIME_GROUP tg)                                                     */
/*                                                                           */
/*  Return the maximum workload that a resource can have on day tg.          */
/*                                                                           */
/*****************************************************************************/

static float KheDrsMaxWorkloadOnDay(KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_TIME_GROUP tg)
{
  float res, wk;  int i;  KHE_TIME t;
  res = 0.0;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    wk = KheDrsMaxWorkloadAtTime(drs, t);
    if( wk > res )
      res = wk;
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_WORK_DAY KheDrsExprWorkDayMake(KHE_TIME_GROUP tg,           */
/*    KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs)            */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_WORK_DAY object with these attributes.           */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_WORK_DAY KheDrsExprWorkDayMake(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_WORK_DAY res;  KHE_DRS_EXPR e;

  /* MakeBegin part */
  if( KheDrsResourceOnDayHasDayExpr(drd, KHE_DRS_EXPR_WORK_DAY_TAG, tg,drs,&e) )
    return (KHE_DRS_EXPR_WORK_DAY) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_WORK_DAY_TAG,
    drd->encl_dr, drs);
  res->resource_on_day = drd;
  res->time_group = tg;

  /* MakeEnd part */
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  res->value_ub.f = KheDrsMaxWorkloadOnDay(drs, tg);
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkDayLeafSet(KHE_DRS_EXPR_WORK_DAY ewd,                 */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of ewd, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as ewd is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkDayLeafSet(KHE_DRS_EXPR_WORK_DAY ewd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  ewd->value.f = KheTaskWorkloadPerTime(dtd->task);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkDayLeafClear(KHE_DRS_EXPR_WORK_DAY ewd)               */
/*                                                                           */
/*  Inform leaf expression ewt that the assignment of drd to dtd carried     */
/*  out by KheDrsExprWorkDayLeafSet above has been removed.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkDayLeafClear(KHE_DRS_EXPR_WORK_DAY ewd)
{
  ewd->value.f = 0.0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkDaySetClosedValue(KHE_DRS_EXPR_WORK_DAY ewd,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ewd, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkDaySetClosedValue(KHE_DRS_EXPR_WORK_DAY ewd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = ewd->resource_on_day->closed_dtd;
  ewd->value.f = (dtd != NULL ? KheTaskWorkloadPerTime(dtd->task) : 0.0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkDayDoDebug(KHE_DRS_EXPR_WORK_DAY ewd,                 */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ewd which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkDayDoDebug(KHE_DRS_EXPR_WORK_DAY ewd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "WORK_DAY(%s, %s)",
    KheDrsResourceId(ewd->resource_on_day->encl_dr),
    KheTimeGroupId(ewd->time_group));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ewd, soln, true, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_OR - dominance testing"                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprOrCounterDomTest(KHE_DRS_EXPR_OR eo,          */
/*    KHE_DRS_EXPR_COUNTER ec, int open_day_index,                           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return a dom test suited to child eo of ec.                              */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_CONSTRAINT KheDrsExprCostConstraint(KHE_DRS_EXPR_COST ec,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static KHE_DRS_DOM_TEST KheDrsExprOrCounterDomTest(KHE_DRS_EXPR_OR eo,
  KHE_DRS_EXPR_COUNTER ec, int open_day_index, 
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b, unassigned_child_count;  KHE_DRS_DIM4_TABLE corr_dom_table4;
  KHE_DRS_CONSTRAINT dc;

  /* calculate a */
  a = (ec->max_limit < INT_MAX ? -1 : INT_MAX);

  /* calculate b */
  b = (ec->min_limit > 0 ? INT_MAX : 0);

  /* make and return the dom test */
  unassigned_child_count = KheDrsOpenChildrenAtOrAfter(
    &ec->open_children_by_day, open_day_index + 1);
  dc = KheDrsExprCostConstraint((KHE_DRS_EXPR_COST) ec, drs);
  if( DEBUG57 )
  {
    fprintf(stderr, "[ calling KheDrsDim5TableGet(%p, %d), open_day_index "
      "%d, ec:\n", (void *) dc->counter_corr_dom_table5,
      unassigned_child_count, open_day_index);
    KheDrsExprDebug((KHE_DRS_EXPR) ec, drs, 2, 2, stderr);
    KheDrsOpenChildrenDebug(&ec->open_children_by_day, 2, 2, stderr);
    fprintf(stderr, "]\n");
  }
  corr_dom_table4 = KheDrsDim5TableGet(dc->counter_corr_dom_table5,
    unassigned_child_count);
  return KheDrsDomTestMakeInt(KHE_DRS_DOM_TEST_SEPARATE_INT, (KHE_DRS_EXPR) eo,
    false, /* 0, INT_MAX, */ a, b, 0, NULL, corr_dom_table4, NULL, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprOrSequenceDomTest(KHE_DRS_EXPR_OR eo,         */
/*    KHE_DRS_EXPR_SEQUENCE es, int open_day_index,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return a dom test suited to child eo of es.                              */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprOrSequenceDomTest(KHE_DRS_EXPR_OR eo,
  KHE_DRS_EXPR_SEQUENCE es, int open_day_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b;

  /* calculate a */
  a = (es->max_limit < INT_MAX ? -1 : INT_MAX);

  /* calculate b */
  b = (es->min_limit > 0 ? INT_MAX : 0);

  /* make and return the dom test */
  return KheDrsDomTestMakeInt(KHE_DRS_DOM_TEST_SEPARATE_INT, (KHE_DRS_EXPR) eo,
    false, /* 0, INT_MAX, */ a, b, 0, NULL, NULL, NULL, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprOrDomTest(KHE_DRS_EXPR_OR eo,                 */
/*    int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return a dom test suited to eo on day open_day_index, taking the         */
/*  context of eo into account.  It is always a child of an COUNTER          */
/*  or SEQUENCE.                                                             */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprOrDomTest(KHE_DRS_EXPR_OR eo,
  int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR prnt_e;

  HnAssert(HaArrayCount(eo->parents)==1, "KheDrsExprOrDomTest internal error");
  prnt_e = HaArrayFirst(eo->parents).expr;
  switch( prnt_e->tag )
  {
    case KHE_DRS_EXPR_COUNTER_TAG:

      return KheDrsExprOrCounterDomTest(eo,
	(KHE_DRS_EXPR_COUNTER) prnt_e, open_day_index, drs);

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      return KheDrsExprOrSequenceDomTest(eo,
	(KHE_DRS_EXPR_SEQUENCE) prnt_e, open_day_index, drs);

    default:

      HnAbort("KheDrsExprOrDomTest internal error (prnt tag %d)", prnt_e->tag);
      return NULL;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_OR"                                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_OR KheDrsExprOrMakeBegin(KHE_DRS_RESOURCE dr,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Begin making a new KHE_DRS_EXPR_OR object with these attributes.         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_OR KheDrsExprOrMakeBegin(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_OR res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_OR_TAG, dr, drs);
  res->closed_state = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrAddChild(KHE_DRS_EXPR_OR eo, KHE_DRS_EXPR child_e,      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  New child child_e has just been added to eo.  Update eo's closed         */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrAddChild(KHE_DRS_EXPR_OR eo, KHE_DRS_EXPR child_e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->value.i == 1 )
    eo->closed_state += 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrMakeEnd(KHE_DRS_EXPR_OR eo,                             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  End the initialization of eo.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrMakeEnd(KHE_DRS_EXPR_OR eo,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsExprInitEnd((KHE_DRS_EXPR) eo, drs);
  eo->value_ub.i = 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrChildHasOpened(KHE_DRS_EXPR_OR eo,                      */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eo for solving, eo's child_index'th child, child_e,   */
/*  has been opened.  Update eo's closed state to reflect this.              */
/*                                                                           */
/*  The closed state of eo is the number of eo's closed children whose       */
/*  value is 1, so this update subtracts 1 from the closed state if          */
/*  child_e's value (which preserves its value before opening) is 1.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrChildHasOpened(KHE_DRS_EXPR_OR eo,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsOpenChildrenAddChild(&eo->open_children_by_day, child_e);
  /* ***
  KheDrsOpenChildrenAddChildInDayOrder(&eo->open_children_by_day, child_e);
  *** */
  if( child_e->value.i == 1 )
    eo->closed_state -= 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrChildHasClosed(KHE_DRS_EXPR_OR eo,                      */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing eo, eo's child_index'th child, child_e, has been      */
/*  closed.  Update eo's closed state to reflect this.                       */
/*                                                                           */
/*  The closed state of eo is the number of eo's closed children whose       */
/*  value is 1, so this update adds 1 to the closed state if child_e's       */
/*  value (which is up to date because it is closed first) is 1.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrChildHasClosed(KHE_DRS_EXPR_OR eo,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->value.i == 1 )
    eo->closed_state += 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrSetClosedValue(KHE_DRS_EXPR_OR eo,                      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eo, set its value suitably for its closed state.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrSetClosedValue(KHE_DRS_EXPR_OR eo,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eo->value.i = (eo->closed_state > 0 ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrEvalSignature(KHE_DRS_EXPR_OR eo,                       */
/*    KHE_DRS_SOLN prev_soln, int next_di, KHE_DRS_SIGNATURE sig,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Evaluate expression eo within sig.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrEvalSignature(KHE_DRS_EXPR_OR eo, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, /* int next_di, */ KHE_DRS_SIGNATURE next_sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, next_di;  KHE_DRS_EXPR child_e;  KHE_DRS_VALUE val;

  next_di = KheDrsSignerOpenDayIndex(dsg);
  if( KheDrsOpenChildrenIndexIsFirst(&eo->open_children_by_day, next_di) )
  {
    /*  no previous day, so we have a 0 (false) value here */
    val.i = 0;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) eo, next_di - 1, prev_sig);
  }

  /* accumulate the values of the children of eo that finalized today */
  KheDrsOpenChildrenForEach(&eo->open_children_by_day, next_di, child_e, i)
    if( child_e->value.i == 1 )
      val.i = 1;

  if( KheDrsOpenChildrenIndexIsLast(&eo->open_children_by_day, next_di) )
  {
    /* last day; incorporate closed state and set value */
    if( eo->closed_state > 0 )
      val.i = 1;
    eo->value = val;
  }
  else
  {
    /* not last day; store val in next_soln */
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) eo);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrDoDebug(KHE_DRS_EXPR_OR eo,                             */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eo which is specific to it.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrDoDebug(KHE_DRS_EXPR_OR eo,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "OR(cs %d, iv %d)", eo->closed_state, eo->value.i);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eo, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_AND - dominance testing"                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprAndCounterDomTest(KHE_DRS_EXPR_AND ea,        */
/*    KHE_DRS_EXPR_COUNTER ec, int open_day_index,                           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return a dom test suited to child ea of ec.                              */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprAndCounterDomTest(KHE_DRS_EXPR_AND ea,
  KHE_DRS_EXPR_COUNTER ec, int open_day_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b;

  /* calculate a */
  a = (ec->max_limit < INT_MAX ? -1 : INT_MAX);

  /* calculate b */
  b = (ec->min_limit > 0 ? INT_MAX : 0);

  /* make and return the dom test */
  return KheDrsDomTestMakeInt(KHE_DRS_DOM_TEST_SEPARATE_INT, (KHE_DRS_EXPR) ea,
    false, /* 0, INT_MAX, */ a, b, 0, NULL, NULL, NULL, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprAndSequenceDomTest(KHE_DRS_EXPR_AND ea,       */
/*    KHE_DRS_EXPR_SEQUENCE es, int open_day_index,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return a dom test suited to child ea of es.                              */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprAndSequenceDomTest(KHE_DRS_EXPR_AND ea,
  KHE_DRS_EXPR_SEQUENCE es, int open_day_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b;

  /* calculate a */
  a = (es->max_limit < INT_MAX ? -1 : INT_MAX);

  /* calculate b */
  b = (es->min_limit > 0 ? INT_MAX : 0);

  /* make and return the dom test */
  return KheDrsDomTestMakeInt(KHE_DRS_DOM_TEST_SEPARATE_INT, (KHE_DRS_EXPR) ea,
    false, /* 0, INT_MAX, */ a, b, 0, NULL, NULL, NULL, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprAndDomTest(KHE_DRS_EXPR_AND ea,               */
/*    int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return a dom test suited to ea on day open_day_index, taking the         */
/*  context of eo into account.                                              */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprAndDomTest(KHE_DRS_EXPR_AND ea,
  int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR prnt_e;

  HnAssert(HaArrayCount(ea->parents)==1, "KheDrsExprAndDomTest internal error");
  prnt_e = HaArrayFirst(ea->parents).expr;
  switch( prnt_e->tag )
  {
    case KHE_DRS_EXPR_COUNTER_TAG:

      return KheDrsExprAndCounterDomTest(ea,
	(KHE_DRS_EXPR_COUNTER) prnt_e, open_day_index, drs);

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      return KheDrsExprAndSequenceDomTest(ea,
	(KHE_DRS_EXPR_SEQUENCE) prnt_e, open_day_index, drs);

    default:

      HnAbort("KheDrsExprAndDomTest internal error (prnt tag %d)", prnt_e->tag);
      return NULL;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_AND"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_AND KheDrsExprAndMakeBegin(KHE_DRS_RESOURCE dr,             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_AND object with these attributes.                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_AND KheDrsExprAndMakeBegin(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_AND res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_AND_TAG, dr, drs);
  res->closed_state = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndAddChild(KHE_DRS_EXPR_AND ea, KHE_DRS_EXPR child_e,    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  New child child_e has just been added to ea.  Update ea's closed         */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndAddChild(KHE_DRS_EXPR_AND ea, KHE_DRS_EXPR child_e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->value.i == 0 )
    ea->closed_state += 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndMakeEnd(KHE_DRS_EXPR_AND ea,                           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  End the initialization of ea.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndMakeEnd(KHE_DRS_EXPR_AND ea,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsExprInitEnd((KHE_DRS_EXPR) ea, drs);
  ea->value_ub.i = 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndChildHasOpened(KHE_DRS_EXPR_AND ea,                    */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening ea for solving, ea's child_index'th child, child_e,   */
/*  has been opened.  Update ea's closed state to reflect this.              */
/*                                                                           */
/*  The closed state of ea is the number of ea's closed children whose       */
/*  value is 0, so this update subtracts 1 from the closed state if          */
/*  child_e's value (which preserves its value before opening) is 0.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndChildHasOpened(KHE_DRS_EXPR_AND ea,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsOpenChildrenAddChild(&ea->open_children_by_day, child_e);
  /* ***
  KheDrsOpenChildrenAddChildInDayOrder(&ea->open_children_by_day, child_e);
  *** */
  if( child_e->value.i == 0 )
    ea->closed_state -= 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndChildHasClosed(KHE_DRS_EXPR_AND ea,                    */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing ea, ea's child_index'th child, child_e, has been      */
/*  closed.  Update ea's closed state to reflect this.                       */
/*                                                                           */
/*  The closed state of ea is the number of ea's closed children whose       */
/*  value is 0, so this update adds 1 to the closed state if child_e's       */
/*  value (which is up to date because it is closed first) is 0.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndChildHasClosed(KHE_DRS_EXPR_AND ea,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->value.i == 0 )
    ea->closed_state += 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndSetClosedValue(KHE_DRS_EXPR_AND ea,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ea, set its value suitably for its closed state.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndSetClosedValue(KHE_DRS_EXPR_AND ea,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ea->value.i = (ea->closed_state > 0 ? 0 : 1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndEvalSignature(KHE_DRS_EXPR_AND ea,                     */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression ea within sig.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndEvalSignature(KHE_DRS_EXPR_AND ea,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, /* int next_di, */
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, next_di;  KHE_DRS_EXPR child_e;  KHE_DRS_VALUE val;

  next_di = KheDrsSignerOpenDayIndex(dsg);
  if( KheDrsOpenChildrenIndexIsFirst(&ea->open_children_by_day, next_di) )
  {
    /* first day, so we have a 1 (true) value here */
    val.i = 1;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) ea, next_di - 1, prev_sig);
  }

  /* accumulate the values of the children of ea that finalized today */
  KheDrsOpenChildrenForEach(&ea->open_children_by_day, next_di, child_e, i)
    if( child_e->value.i == 0 )
      val.i = 0;

  if( KheDrsOpenChildrenIndexIsLast(&ea->open_children_by_day, next_di) )
  {
    /* last day; incorporate closed_state and set value */
    if( ea->closed_state > 0 )
      val.i = 0;
    ea->value = val;
  }
  else
  {
    /* not last day; store value in next_soln */
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) ea);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndDoDebug(KHE_DRS_EXPR_AND ea,                           */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ea which is specific to it.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndDoDebug(KHE_DRS_EXPR_AND ea,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "AND(cs %d, iv %d)", ea->closed_state, ea->value.i);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ea, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_SUM"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_INT_SUM KheDrsExprIntSumMake(KHE_DRS_RESOURCE dr,           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_INT_SUM object with these attributes.            */
/*                                                                           */
/*****************************************************************************/

/* *** currently (and probably permanently) unused
static KHE_DRS_EXPR_INT_SUM KheDrsExprIntSumMake(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_INT_SUM res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_INT_SUM_TAG, dr, drs);
  res->value.i = UNDEF_INT;
  res->closed_state = 0;
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumAddChild(KHE_DRS_EXPR_INT_SUM eis,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to eis.  Update eis's closed       */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSumAddChild(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eis->closed_state += child_e->value.i;
  eis->value_ub.i += child_e->value_ub.i;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumChildHasOpened(KHE_DRS_EXPR_INT_SUM eis,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eis for solving, eis's child_index'th child, child_e, */
/*  has been opened.  Update eis's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of eis is the sum of the values of eis's s closed       */
/*  children, so this update subtracts child_e's value (which preserves      */
/*  its value before opening) from the closed state.                         */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSumChildHasOpened(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eis->closed_state -= child_e->value.i;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumChildHasClosed(KHE_DRS_EXPR_INT_SUM eis,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing eis, eis's child_index'th child, child_e, has been    */
/*  closed.  Update eis's closed state to reflect this.                      */
/*                                                                           */
/*  The closed state of eis is the sum of the values of eis's s closed       */
/*  children, so this update adds child_e's value (which preserves           */
/*  its value before opening) to the closed state.                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSumChildHasClosed(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eis->closed_state += child_e->value.i;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumSetClosedValue(KHE_DRS_EXPR_INT_SUM eis,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eis, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSumSetClosedValue(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eis->value.i = eis->closed_state;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumEvalSignature(KHE_DRS_EXPR_INT_SUM eis,             */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression eis within sig.                                      */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSumEvalSignature(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2;  KHE_DRS_EXPR child_e;  KHE_DRS_VALUE val;

  if( KheDrsOpenChildrenIndexIsFirst(&eis->open_children_by_day, next_di) )
  {
    ** first day, so (ignoring closed_state) we have a 0 value here **
    val.i = 0;
  }
  else
  {
    ** not first day, so retrieve a previous value **
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) eis, next_di - 1, prev_sig);
  }

  ** accumulate the open values of the children of eis that finalized today **
  KheDrsOpenChildrenForEach(&eis->open_children_by_day, next_di, child_e, i)
    val.i += child_e->value.i;

  if( KheDrsOpenChildrenIndexIsLast(&eis->open_children_by_day, next_di) )
  {
    ** last day; add closed_state to the value **
    eis->value.i = val.i + eis->closed_state;
  }
  else
  {
    ** this not eis's last day; store value in next_soln **
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) eis);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumDoDebug(KHE_DRS_EXPR_INT_SUM eis,                   */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eis which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSumDoDebug(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "INT_SUM(%d)", eis->closed_state);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eis, soln, false, drs, fp);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_FLOAT_SUM"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_FLOAT_SUM KheDrsExprFloatSumMake(KHE_DRS_RESOURCE dr,       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_FLOAT_SUM object with these attributes.          */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by EXPR_SUM
static KHE_DRS_EXPR_FLOAT_SUM KheDrsExprFloatSumMake(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_FLOAT_SUM res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_FLOAT_SUM_TAG, dr, drs);
  res->value.f = UNDEF_FLOAT;
  res->closed_state = 0.0;
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumAddChild(KHE_DRS_EXPR_FLOAT_SUM efs,              */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to efs.  Update efs's closed       */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatSumAddChild(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  efs->closed_state += child_e->value.f;
  efs->value_ub.f += child_e->value_ub.f;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumChildHasOpened(KHE_DRS_EXPR_FLOAT_SUM efs,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening efs for solving, efs's child_index'th child, child_e, */
/*  has been opened.  Update efs's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of efs is the sum of the values of efs's closed         */
/*  children, so this update subtracts child_e's value (which preserves      */
/*  its value before opening) from the closed state.                         */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatSumChildHasOpened(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  efs->closed_state -= child_e->value.f;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumChildHasClosed(KHE_DRS_EXPR_FLOAT_SUM efs,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening efs for solving, efs's child_index'th child, child_e, */
/*  has been closed.  Update efs's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of efs is the sum of the values of efs's closed         */
/*  children, so this update adds child_e's value (which preserves           */
/*  its value before opening) to the closed state.                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatSumChildHasClosed(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  efs->closed_state += child_e->value.f;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumSetClosedValue(KHE_DRS_EXPR_FLOAT_SUM efs,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing efs, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatSumSetClosedValue(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  efs->value.f = efs->closed_state;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheFloatToStoredInt(float val)                                       */
/*  float KheFloatFromStoredInt(int val)                                     */
/*                                                                           */
/*  KHE_DRS_EXPR_FLOAT_SUM should really store float values in the           */
/*  signature, but at present signatures only hold integers, so we           */
/*  use an approximate method implemented by these two little functions.     */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used, we're storing floats directly now
static int KheFloatToStoredInt(float val)
{
  return (int) (val * 100.0);
}

static float KheFloatFromStoredInt(int val)
{
  return (float) val / 100.0;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumEvalSignature(KHE_DRS_EXPR_FLOAT_SUM efs,         */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression efs in sig.                                          */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatSumEvalSignature(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2;  KHE_DRS_EXPR child_e;  KHE_DRS_VALUE val;

  if( KheDrsOpenChildrenIndexIsFirst(&efs->open_children_by_day, next_di) )
  {
    ** first day, so (ignoring closed_state) we have a 0.0 value here **
    val.f = 0.0;
  }
  else
  {
    ** not first day, so retrieve a previous value (stored as int) **
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) efs, next_di-1, prev_sig);
    ** ***
    val = KheFloatFromStoredInt(
      KheDrsExprDaySigVal((KHE_DRS_EXPR) efs, next_di-1, prev_sig));
    *** **
  }

  ** accumulate the open values of the children of efs that finalized today **
  KheDrsOpenChildrenForEach(&efs->open_children_by_day, next_di, child_e, i)
    val.f += child_e->value.f;

  if( KheDrsOpenChildrenIndexIsLast(&efs->open_children_by_day, next_di) )
  {
    ** last day; add closed_state to the value **
    efs->value.f = val.f + efs->closed_state;
  }
  else
  {
    ** this not efs's last day; store value (stored as int) in next_soln **
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) efs);
    ** ***
    KheDrsSignatureAddState(next_sig, KheFloatToStoredInt(val), dsg,
      (KHE_DRS_EXPR) efs);
    *** **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumDoDebug(KHE_DRS_EXPR_FLOAT_SUM efs,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of efs which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatSumDoDebug(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "FLOAT_SUM(%.2f)", efs->closed_state);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) efs, soln, true, drs, fp);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_DEV"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntDevDev(KHE_DRS_EXPR_INT_DEV eid, int val)               */
/*                                                                           */
/*  Deviation function to use with these expressions.                        */
/*                                                                           */
/*****************************************************************************/

/* ***
static int KheDrsExprIntDevDev(KHE_DRS_EXPR_INT_DEV eid, int val)
{
  if( val == 0 && eid->allow_zero )
    return 0;
  else if( val > eid->max_limit )
    return val - eid->max_limit;
  else if( val < eid->min_limit )
    return eid->min_limit - val;
  else
    return 0;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_INT_DEV KheDrsExprIntDevMake(int min_limit, int max_limit,  */
/*    bool allow_zero, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs) */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_INT_DEV object with these attributes.            */
/*                                                                           */
/*  There is no dom_test because its type is always KHE_DRS_DOM_UNUSED.      */
/*                                                                           */
/*****************************************************************************/

/* *** currently (and probably permanently) unused
static KHE_DRS_EXPR_INT_DEV KheDrsExprIntDevMake(int min_limit, int max_limit,
  bool allow_zero, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_INT_DEV res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_INT_DEV_TAG, dr, drs);
  res->value.i = UNDEF_INT;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->allow_zero = allow_zero;
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevAddChild(KHE_DRS_EXPR_INT_DEV eid,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to eid.  Update eid's closed       */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntDevAddChild(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ** nothing to do here **
  ** eid->value_ub should be initialized here, but to what? hard to say **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevChildHasOpened(KHE_DRS_EXPR_INT_DEV eid,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eid for solving, eid's child_index'th child, child_e, */
/*  has been opened.  Update eid's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of eid is the value of eid's single closed child.       */
/*  This is not stored explicitly, so this update does nothing.              */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntDevChildHasOpened(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ** nothing to do here **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevChildHasClosed(KHE_DRS_EXPR_INT_DEV eid,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eid for solving, eid's child_index'th child, child_e, */
/*  has been closed.  Update eid's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of eid is the value of eid's single closed child.       */
/*  This is not stored explicitly, so this update does nothing.              */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntDevChildHasClosed(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ** nothing to do here **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevSetClosedValue(KHE_DRS_EXPR_INT_DEV eid,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eid, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntDevSetClosedValue(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR child_e;
  child_e = HaArrayFirst(eid->children);
  eid->value.i = KheDrsExprIntDevDev(eid, child_e->value.i);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevEvalSignature(KHE_DRS_EXPR_INT_DEV eid,             */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression eid within sig.                                      */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntDevEvalSignature(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&eid->open_children_by_day, next_di) )
  {
    ** first day **
    val = 0;
  }
  else
  {
    ** not first day, so retrieve a previous value **
    HnAbort("KheDrsExprIntDevEvalSignature internal error (non-first day)");
    val = 0;  ** keep compiler happy **
  }

  ** accumulate the open values of the children of eid that finalized today **
  KheDrsOpenChildrenForEach(&eid->open_children_by_day, next_di, child_e, i)
    val += KheDrsExprIntDevDev(eid, child_e->value.i);

  if( KheDrsOpenChildrenIndexIsLast(&eid->open_children_by_day, next_di) )
  {
    ** last day; set value **
    eid->value.i = val;
  }
  else
  {
    ** not last day; store val in next_soln **
    HnAbort("KheDrsExprIntDevEvalSignature internal error (non-last day)");
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevDoDebug(KHE_DRS_EXPR_INT_DEV eid,                   */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eid which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntDevDoDebug(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "INT_DEV(%d, %d, %s)", eid->min_limit, eid->max_limit,
    bool_show(eid->allow_zero));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eid, soln, false, drs, fp);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_FLOAT_DEV"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprFloatDevDev(KHE_DRS_EXPR_FLOAT_DEV efd, float val)         */
/*                                                                           */
/*  Deviation function to use with these expressions.                        */
/*                                                                           */
/*****************************************************************************/

/* ***
static int KheDrsExprFloatDevDev(KHE_DRS_EXPR_FLOAT_DEV efd, float val)
{
  if( efd->allow_zero && val < 0.001 )
    return 0;
  else if( val < efd->min_limit - 0.001 )
    return (int) ceil(efd->min_limit - 0.001 - val);
  else if( val > efd->max_limit + 0.001 )
    return (int) ceil(val - efd->max_limit - 0.001);
  else
    return 0;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_FLOAT_DEV KheDrsExprFloatDevMake(int min_limit,             */
/*    int max_limit, bool allow_zero, KHE_DRS_RESOURCE dr,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_FLOAT_DEV object with these attributes.          */
/*                                                                           */
/*  There is no dom_test because its type is always KHE_DRS_DOM_UNUSED.      */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by FLOAT_SUM
static KHE_DRS_EXPR_FLOAT_DEV KheDrsExprFloatDevMake(int min_limit,
  int max_limit, bool allow_zero, KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_FLOAT_DEV res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_FLOAT_DEV_TAG, dr, drs);
  res->value.i = UNDEF_INT;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->allow_zero = allow_zero;
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevAddChild(KHE_DRS_EXPR_FLOAT_DEV efd,              */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to efd.  Update efd's closed       */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatDevAddChild(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ** nothing to do here **
  ** efd->value_ub should be initialized here, but to what? hard to say **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheLimitWorkloadMonitorDev(KHE_DRS_EXPR_FLOAT_DEV efd,               */
/*    float workload)                                                        */
/*                                                                           */
/*  Return the deviation of workload wrt efd.                                */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsExprFloatDevDev above
static int KheLimitWorkloadMonitorDev(KHE_DRS_EXPR_FLOAT_DEV efd,
  float workload)
{
  if( efd->allow_zero && workload < 0.001 )
    return 0;
  else if( workload < efd->min_limit - 0.001 )
    return (int) ceil(efd->min_limit - 0.001 - workload);
  else if( workload > efd->max_limit + 0.001 )
    return (int) ceil(workload - efd->max_limit - 0.001);
  else
    return 0;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevChildHasOpened(KHE_DRS_EXPR_FLOAT_DEV efd,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening efd for solving, efd's child_index'th child, child_e, */
/*  has been opened.  Update efd's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of efd is the value of efd's single closed child,       */
/*  so this update needs to do nothing.                                      */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatDevChildHasOpened(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ** nothing to do here **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevChildHasClosed(KHE_DRS_EXPR_FLOAT_DEV efd,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing efd, efd's child_index'th child, child_e,             */
/*  has been closed.  Update efd's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of efd is the value of efd's single closed child,       */
/*  so this update needs to do nothing.                                      */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatDevChildHasClosed(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ** nothing to do here **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevSetClosedValue(KHE_DRS_EXPR_FLOAT_DEV efd,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing efd, set its value suitably for its closed state.     */
/*  NB despite the floats everywhere, the value is an integer.               */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatDevSetClosedValue(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR child_e;
  child_e = HaArrayFirst(efd->children);
  efd->value.i = KheDrsExprFloatDevDev(efd, child_e->value.f);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevEvalSignature(KHE_DRS_EXPR_FLOAT_DEV efd,         */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression efd within sig.                                      */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatDevEvalSignature(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&efd->open_children_by_day, next_di) )
  {
    ** first day **
    val = 0;
  }
  else
  {
    ** not first day, so retrieve a previous value **
    HnAbort("KheDrsExprFloatDevEvalSignature internal error (non-first day)");
    val = 0;  ** keep compiler happy **
  }

  ** accumulate the open values of the children of efd that finalized today **
  KheDrsOpenChildrenForEach(&efd->open_children_by_day, next_di, child_e, i)
    val += KheDrsExprFloatDevDev(efd, child_e->value.f);

  if( KheDrsOpenChildrenIndexIsLast(&efd->open_children_by_day, next_di) )
  {
    ** last day; set value **
    efd->value.i = val;
  }
  else
  {
    ** not last day; store value in next_soln **
    HnAbort("KheDrsExprFloatDevEvalSignature internal error (non-last day)");
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevDoDebug(KHE_DRS_EXPR_FLOAT_DEV efd,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of efd which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatDevDoDebug(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "FLOAT_DEV(%d, %d, %s)", efd->min_limit, efd->max_limit,
    bool_show(efd->allow_zero));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) efd, soln, false, drs, fp);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_COST"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprCostUnweightedCost(KHE_DRS_EXPR_COST ec, int dev)          */
/*                                                                           */
/*  Return f(dev) for ec, only without multiplying by the weight.            */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprCostUnweightedCost(KHE_DRS_EXPR_COST ec, int dev)
{
  switch( ec->cost_fn )
  {
    case KHE_STEP_COST_FUNCTION:

      return dev > 0 ? 1 : 0;

    case KHE_LINEAR_COST_FUNCTION:

      return dev;

    case KHE_QUADRATIC_COST_FUNCTION:

      return dev * dev;

    default:

      HnAbort("KheDrsExprCostUnweightedCost internal error");
      return 0;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprCostCost(KHE_DRS_EXPR_COST ec, int dev)               */
/*                                                                           */
/*  Return f(dev) for ec.                                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsExprCostCost(KHE_DRS_EXPR_COST ec, int dev)
{
  return ec->combined_weight * KheDrsExprCostUnweightedCost(ec, dev);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCostSetConstraint(KHE_DRS_EXPR_COST ec,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Ensure that there is a constraint corresponding to ec, and that          */
/*  its min_history and max_history values include ec->history.              */
/*                                                                           */
/*  Implementation note.  We can be sure that c != NULL here, because        */
/*  the only monitors for which c == NULL are redundancy prefer resources    */
/*  monitors, and those satisfy a condition which ensures that they are      */
/*  not converted into cost expressions; instead, they add to asst_cost.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprCostSetConstraint(KHE_DRS_EXPR_COST ec,
  int history, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_CONSTRAINT c;  int index;  KHE_DRS_CONSTRAINT dc;
  c = KheMonitorConstraint(ec->monitor->monitor);
  HnAssert(c != NULL, "KheDrsExprCostSetConstraint internal error");
  index = KheConstraintIndex(c);
  HaArrayFill(drs->all_constraints, index + 1, NULL);
  dc = HaArray(drs->all_constraints, index);
  if( dc == NULL )
  {
    /* new, so build and add a new object */
    dc = KheDrsConstraintMake(c, history, (KHE_DRS_EXPR) ec, drs);
    HaArrayPut(drs->all_constraints, index, dc);
  }
  else
  {
    /* already built, so just update the fields */
    if( history < dc->min_history )
      dc->min_history = history;
    if( history > dc->max_history )
      dc->max_history = history;
    if( HaArrayCount(ec->children) > dc->max_child_count )
      dc->max_child_count = HaArrayCount(ec->children);
    dc->needs_corr_table = dc->needs_corr_table ||
      KheDrsExprNeedsCorrTable((KHE_DRS_EXPR) ec, drs->days_frame);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CONSTRAINT KheDrsExprCostConstraint(KHE_DRS_EXPR_COST ec,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return the DRS constraint object corresponding to ec.  It must exist.    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CONSTRAINT KheDrsExprCostConstraint(KHE_DRS_EXPR_COST ec,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_CONSTRAINT c;  int index;  KHE_DRS_CONSTRAINT res;
  c = KheMonitorConstraint(ec->monitor->monitor);
  index = KheConstraintIndex(c);
  HnAssert(index < HaArrayCount(drs->all_constraints),
    "KheDrsExprCostConstraint internal error 1");
  res = HaArray(drs->all_constraints, index);
  HnAssert(res != NULL, "KheDrsExprCostConstraint internal error 2");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_COUNTER - dominance testing"                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprCounterDoDomTest(KHE_DRS_EXPR_COUNTER ec,     */
/*    int unassigned_child_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Return a dom test for ec, assuming there are unassigned_child_count      */
/*  unassigned children.                                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprCounterDoDomTest(KHE_DRS_EXPR_COUNTER ec,
  int unassigned_child_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b;  KHE_DRS_DOM_TEST_TYPE dt_type;  KHE_DRS_CONSTRAINT dc;
  KHE_DRS_DIM2_TABLE main_dom_table2;

  /* calculate a as documented */
  a = ec->max_limit - unassigned_child_count;
  /* ***
  if( ec->max_limit < INT_MAX )
    a = ec->max_limit - unassigned_child_count;
  else
    a = INT_MAX;
  *** */

  /* calculate b as documented */
  b = ec->min_limit;
  /* ***
  if( ec->min_limit > 0 )
    b = ec->min_limit;
  else
    b = 0;
  *** */

  /* make and return the dom test */
  dc = KheDrsExprCostConstraint((KHE_DRS_EXPR_COST) ec, drs);
  if( DEBUG93 && !KheDrsDim3TableGetCheck(dc->counter_main_dom_table3,
      unassigned_child_count) )
    fprintf(stderr, "  KheDrsExprCounterDoDomTest(%s, %d) failing\n",
      KheDrsMonitorId(ec->monitor), unassigned_child_count);
  main_dom_table2 = KheDrsDim3TableGet(dc->counter_main_dom_table3,
    unassigned_child_count);
  dt_type = KheDomTestTypeActual(drs->solve_dom_test_type,
    ec->cost_fn != KHE_QUADRATIC_COST_FUNCTION, false);
  return KheDrsDomTestMakeInt(dt_type, (KHE_DRS_EXPR) ec, ec->allow_zero,
    /* ec->min_limit, ec->max_limit, */ a, b, ec->combined_weight,
    main_dom_table2, NULL, ec->monitor, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprCounterDomTest(KHE_DRS_EXPR_COUNTER ec,       */
/*    int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return a dom test suited to ec on day open_day_index.                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprCounterDomTest(KHE_DRS_EXPR_COUNTER ec,
  int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int unassigned_child_count;  KHE_DRS_DOM_TEST res;
  unassigned_child_count =
    KheDrsOpenChildrenAtOrAfter(&ec->open_children_by_day, open_day_index+1);
  if( DEBUG91(ec->monitor) )
    fprintf(stderr, "[ KheDrsExprCounterDomTest(counter %s, %d, drs), ucc %d\n",
      KheDrsMonitorId(ec->monitor), open_day_index, unassigned_child_count);
  res = KheDrsExprCounterDoDomTest(ec, unassigned_child_count, drs);
  if( DEBUG91(ec->monitor) )
  {
    fprintf(stderr, "] KheDrsExprCounterDomTest returning ");
    KheDrsDomTestDebug(res, 2, 0, stderr);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprCounterNeedsEval(KHE_DRS_EXPR_COUNTER ec,                 */
/*    int di, int later_open_and_assigned, KHE_DYNAMIC_RESOURCE_SOLVER drs,  */
/*    KHE_DRS_DOM_TEST *dom_test)                                            */
/*                                                                           */
/*  Helper function for deciding whether ec needs to be evaluated            */
/*  (i.e. needs to notify some signer) based on open day index di plus       */
/*  later_open_and_assigned, the number of open children after di with       */
/*  an assigned value.                                                       */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsExprCounterNeedsEval(KHE_DRS_EXPR_COUNTER ec,
  int di, int later_open_and_assigned, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_DOM_TEST *dom_test)
{
  int unass_after;
  if( later_open_and_assigned > 0 )
  {
    ** ec needs to be evaluated by the signer **
    unass_after = KheDrsOpenChildrenAtOrAfter(&ec->open_children_by_day, di)
      - later_open_and_assigned;
    HnAssert(unass_after >= 0, "KheDrsExprCounterNeedsEval internal error");
    if( unass_after > 0 )
      *dom_test = KheDrsExprCounterDoDomTest(ec, unass_after, drs);
    else
      *dom_test = NULL;
    return true;
  }
  else
  {
    ** ec does not need to be evaluated by the signer **
    *dom_test = NULL;
    return false;
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprCounterNeedsShiftEval(KHE_DRS_EXPR_COUNTER ec,            */
/*    KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs,                     */
/*    KHE_DRS_DOM_TEST *dom_test)                                            */
/*                                                                           */
/*  Like KheDrsExprCounterNeedsEval except for shifts.                       */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsExprCounterNeedsShiftEval(KHE_DRS_EXPR_COUNTER ec,
  KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_DOM_TEST *dom_test)
{
  int di, si, later_open_and_assigned;  bool res;
  si = ds->open_shift_index;
  di = ds->encl_day->open_day_index;
  later_open_and_assigned =
    KheDrsOpenChildrenWithIndex(&ec->open_children_by_shift, si);
  res = KheDrsExprCounterNeedsEval(ec, di, later_open_and_assigned,
    drs, dom_test);
  if( DEBUG89(ec) )
  {
    fprintf(stderr, "  KheDrsExprCounterNeedsShiftEval(%s, %s, %s) = %s\n",
      KheDrsMonitorId(ec->monitor), KheDrsShiftId(ds),
      *dom_test != NULL ? "non-NULL" : "NULL", bool_show(res));
    fprintf(stderr, "    si %d, di %d, later_open_and_assigned %d\n",
      si, di, later_open_and_assigned);
    KheDrsShiftDebug(ds, drs, 2, 4, stderr);
  }
  return res;
  ** ***
  if( open > 0 )
  {
    ** ec needs to be evaluated by ds's signer **
    unass_after =
      KheDrsOpenChildrenAtOrAfter(&ec->open_children_by_day, di) - open;
    if( unass_after > 0 )
      *dom_test = KheDrsExprIntSum CostDomTest(ec, unass_after, drs);
    else
      *dom_test = NULL;
    return true;
  }
  else
  {
    ** ec does not need to be evaluated by dsp's signer **
    *dom_test = NULL;
    return false;
  }
  *** **

  ** ***
  int si, val;
  si = ds->open_shift_index;
  if( KheDrsOpenChildrenWithIndex(&ec->open_children_by_shift, si) > 0 )
  {
    if( KheDrsExprCounterExprRequiresShiftDomTest(ec, si, drs, &val) )
      *dom_test = KheDrsExprIntSumC ostDomTest(ec, val, drs);
    else
      *dom_test = NULL;
    return true;
  }
  else
  {
    *dom_test = NULL;
    return false;
  }
  *** **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprCounterNeedsShiftPairEval(                                */
/*    KHE_DRS_EXPR_COUNTER ec, KHE_DRS_SHIFT_PAIR dsp,                       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_DOM_TEST *dom_test)           */
/*                                                                           */
/*  Like KheDrsExprCounterNeedsDayEval except for shift pairs.               */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsExprCounterNeedsShiftPairEval(
  KHE_DRS_EXPR_COUNTER ec, KHE_DRS_SHIFT_PAIR dsp,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_DOM_TEST *dom_test)
{
  int di, si1, si2, later_open_and_assigned;
  si1 = dsp->shift[0]->open_shift_index;
  si2 = dsp->shift[1]->open_shift_index;
  di = dsp->shift[0]->encl_day->open_day_index;
  later_open_and_assigned =
    KheDrsOpenChildrenWithIndex(&ec->open_children_by_shift, si1) +
    KheDrsOpenChildrenWithIndex(&ec->open_children_by_shift, si2);
  return KheDrsExprCounterNeedsEval(ec, di, later_open_and_assigned,
    drs, dom_test);
  ** ***
  if( open > 0 )
  {
    ** ec needs to be evaluated by dsp's signer **
    unass_after =
      KheDrsOpenChildrenAtOrAfter(&ec->open_children_by_day, di) - open;
    if( unass_after > 0 )
      *dom_test = KheDrsExprIntSumCo stDomTest(ec, unass_after, drs);
    else
      *dom_test = NULL;
    return true;
  }
  else
  {
    ** ec does not need to be evaluated by dsp's signer **
    *dom_test = NULL;
    return false;
  }
  *** **

  ** ***
  int si, val;
  si = ds->open_shift_index;
  if( KheDrsOpenChildrenWithIndex(&ec->open_children_by_shift, si) > 0 )
  {
    if( KheDrsExprCounterExprRequiresShiftDomTest(ec, si, drs, &val) )
      *dom_test = KheDrsExprIntSumC ostDomTest(ec, val, drs);
    else
      *dom_test = NULL;
    return true;
  }
  else
  {
    *dom_test = NULL;
    return false;
  }
  *** **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_EVAL_TYPE KheDrsExprEvalTypeShift(KHE_DRS_EXPR_COUNTER ec,  */
/*    int di, int scount, KHE_DYNAMIC_RESOURCE_SOLVER drs,                   */
/*    KHE_DRS_DOM_TEST *dom_test)                                            */
/*                                                                           */
/*  Carry out that part of KheDrsExprEvalType (just below) that is common    */
/*  to shift signers and shift pair signers.                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_EVAL_TYPE KheDrsExprEvalTypeShift(KHE_DRS_EXPR_COUNTER ec,
  int di, int scount, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_DOM_TEST *dom_test)
{
  int unass_after;
  if( scount == 0 )
    return KHE_DRS_EXPR_EVAL_NO;
  else
  {
    unass_after = KheDrsOpenChildrenAtOrAfter(&ec->open_children_by_day, di)
      - scount;
    HnAssert(unass_after >= 0, "KheDrsExprEvalType internal error 2");
    if( unass_after > 0 )
    {
      if( dom_test != NULL )
	*dom_test = KheDrsExprCounterDoDomTest(ec, unass_after, drs);
      return KHE_DRS_EXPR_EVAL_NOT_LAST;
    }
    else
      return KHE_DRS_EXPR_EVAL_LAST;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_EVAL_TYPE KheDrsExprEvalType(KHE_DRS_EXPR e,                */
/*    KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,                   */
/*    KHE_DRS_DOM_TEST *dom_test)                                            */
/*                                                                           */
/*  Return the type of evaluation required of e by dsg.  Also, if            */
/*  dom_test != NULL and the return value is KHE_DRS_EXPR_EVAL_NOT_LAST,     */
/*  set *dom_test to a suitable dominance test.                              */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_EVAL_TYPE KheDrsExprEvalType(KHE_DRS_EXPR e,
  KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_DOM_TEST *dom_test)
{
  int di, si, si1, si2, scount;  KHE_DRS_EXPR_COUNTER ec;
  di = dsg->encl_day->open_day_index;
  if( dom_test != NULL )
    *dom_test = NULL;
  switch( dsg->type )
  {
    case KHE_DRS_SIGNER_DAY:
    case KHE_DRS_SIGNER_RESOURCE_ON_DAY:

      if( !KheDrsOpenChildrenIndexInRange(&e->open_children_by_day, di) )
	return KHE_DRS_EXPR_EVAL_NO;
      else if( KheDrsOpenChildrenIndexIsLast(&e->open_children_by_day, di) )
	return KHE_DRS_EXPR_EVAL_LAST;
      else
      {
	if( dom_test != NULL )
	  *dom_test = KheDrsExprDomTest(e, di, drs);
	return KHE_DRS_EXPR_EVAL_NOT_LAST;
      }

    case KHE_DRS_SIGNER_SHIFT:

      HnAssert(e->tag == KHE_DRS_EXPR_COUNTER_TAG,
	"KheDrsExprEvalType internal error 1");
      ec = (KHE_DRS_EXPR_COUNTER) e;
      si = dsg->u.shift->open_shift_index;
      scount = KheDrsOpenChildrenWithIndex(&ec->open_children_by_shift, si);
      return KheDrsExprEvalTypeShift(ec, di, scount, drs, dom_test);

    case KHE_DRS_SIGNER_SHIFT_PAIR:

      HnAssert(e->tag == KHE_DRS_EXPR_COUNTER_TAG,
	"KheDrsExprEvalType internal error 3");
      ec = (KHE_DRS_EXPR_COUNTER) e;
      si1 = dsg->u.shift_pair->shift[0]->open_shift_index;
      si2 = dsg->u.shift_pair->shift[1]->open_shift_index;
      scount = KheDrsOpenChildrenWithIndex(&ec->open_children_by_shift, si1) +
	KheDrsOpenChildrenWithIndex(&ec->open_children_by_shift, si2);
      return KheDrsExprEvalTypeShift(ec, di, scount, drs, dom_test);

    default:

      HnAbort("KheDrsExprEvalType internal error (%d)\n", dsg->type);
      return KHE_DRS_EXPR_EVAL_NO;    /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCounterNotifyCoverSigners(KHE_DRS_EXPR_COUNTER ec,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Notify the relevant cover signers that ec, which is derived from         */
/*  a cover (event resource) constraint, is open.                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprCounterNotifyCoverSigners(KHE_DRS_EXPR_COUNTER ec,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int di, si, i, j, sig_index;  KHE_DRS_SHIFT ds;
  KHE_DRS_DAY day;  KHE_DRS_SHIFT_PAIR dsp;  KHE_DRS_SIGNER dsg;

  if( DEBUG89(ec) )
    fprintf(stderr, "[ KheDrsExprCounterNotifyCoverSigners(%s, drs)\n",
      KheDrsMonitorId(ec->monitor));

  /* notify day signers */
  HaArrayClear(ec->sig_indexes);
  KheDrsOpenChildrenForEachIndex(&ec->open_children_by_day, di)
  {
    day = HaArray(drs->open_days, di);
    dsg = KheDrsDaySigner(day);
    if( KheDrsSignerAddExpr(dsg, (KHE_DRS_EXPR) ec, drs, &sig_index) )
      HaArrayAddLast(ec->sig_indexes, sig_index);
  }

  /* notify shift signers */
  KheDrsOpenChildrenForEachIndex(&ec->open_children_by_shift, si)
  {
    ds = HaArray(drs->open_shifts, si);
    dsg = KheDrsShiftSigner(ds);
    KheDrsSignerAddExpr(dsg, (KHE_DRS_EXPR) ec, drs, &sig_index);
  }

  /* notify shift pair signers */
  KheDrsOpenChildrenForEachIndex(&ec->open_children_by_day, di)
  {
    day = HaArray(drs->open_days, di);
    HaArrayForEach(day->shifts, ds, i)
      HaArrayForEach(ds->shift_pairs, dsp, j)
      {
	dsg = KheDrsShiftPairSigner(dsp);
	KheDrsSignerAddExpr(dsg, (KHE_DRS_EXPR) ec, drs, &sig_index);
      }
  }

  if( DEBUG89(ec) )
    fprintf(stderr, "] KheDrsExprCounterNotifyCoverSigners returning\n");
}


/* *** old version
static void KheDrsExprCounterNotifyCoverSigners(
  KHE_DRS_EXPR_COUNTER ec, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int di, si, i, j;  KHE_DRS_SHIFT ds;  KHE_DRS_DOM_TEST dom_test;
  KHE_DRS_DAY day;  KHE_DRS_SHIFT_PAIR dsp;

  if( DEBUG89(ec) )
    fprintf(stderr, "[ KheDrsExprCounterNotifyCoverSigners(%s, drs)\n",
      KheDrsMonitorId(ec->monitor));

  ** notify day signers **
  HaArrayClear(ec->sig_indexes);
  KheDrsOpenChildrenForEachIndex(&ec->open_children_by_day, di)
  {
    day = HaArray(drs->open_days, di);
    if( KheDrsExprNeedsDayEval((KHE_DRS_EXPR) ec, day, drs, &dom_test) )
    {
      KheDrsDayAddOpenExpr(day, (KHE_DRS_EXPR) ec);
      if( dom_test != NULL )
	HaArrayAddLast(ec->sig_indexes,
	  KheDrsDayAddDomTest(day, dom_test, drs));
    }
  }

  ** notify shift signers **
  ** *** done on the fly now (or is it?)
  KheDrsOpenChildrenBuildShiftChildIndexes(&ec->open_children_by_shift);
  *** **
  KheDrsOpenChildrenForEachIndex(&ec->open_children_by_shift, si)
  {
    ds = HaArray(drs->open_shifts, si);
    if( KheDrsExprCounterNeedsShiftEval(ec, ds, drs, &dom_test) )
    {
      KheDrsShiftAddOpenExpr(ds, (KHE_DRS_EXPR) ec);
      if( dom_test != NULL )
	KheDrsShiftAddDomTest(ds, dom_test, drs);
    }
  }

  ** notify shift pair signers **
  KheDrsOpenChildrenForEachIndex(&ec->open_children_by_day, di)
  {
    day = HaArray(drs->open_days, di);
    HaArrayForEach(day->shifts, ds, i)
      HaArrayForEach(ds->shift_pairs, dsp, j)
	if( KheDrsExprCounterNeedsShiftPairEval(ec, dsp, drs, &dom_test) )
	{
	  KheDrsShiftPairAddOpenExpr(dsp, (KHE_DRS_EXPR) ec);
	  if( dom_test != NULL )
	    KheDrsShiftPairAddDomTest(dsp, dom_test, drs);
	}
  }

  if( DEBUG89(ec) )
    fprintf(stderr, "] KheDrsExprCounterNotifyCoverSigners returning\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprCounterChildDomTest(                          */
/*    KHE_DRS_EXPR_COUNTER ec, KHE_DYNAMIC_RESOURCE_SOLVER drs)              */
/*                                                                           */
/*  Return the dom test for a child of ec on its open_day_index'th           */
/*  open day.                                                                */
/*                                                                           */
/*  Note.  If ec is derived from a limit workload constraint, this           */
/*  code will not attempt to access any tables.                              */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_DOM_TEST KheDrsExprCounterChildDomTest(
  KHE_DRS_EXPR_COUNTER ec, int open_day_index, KHE_DRS_EXPR child_e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b, unassigned_child_count;  KHE_DRS_DIM4_TABLE corr_dom_table4;
  KHE_DRS_CONSTRAINT dc;

  ** calculate a **
  a = (ec->max_limit < INT_MAX ? -1 : INT_MAX);

  ** calculate b **
  b = (ec->min_limit > 0 ? INT_MAX : 0);

  ** make and return the dom test **
  if( child_e->tag == KHE_DRS_EXPR_OR_TAG )
  {
    unassigned_child_count = KheDrsOpenChildrenAtOrAfter(
      &ec->open_children_by_day, open_day_index + 1);
    dc = KheDrsExprCostConstraint((KHE_DRS_EXPR_COST) ec, drs);
    if( DEBUG57 )
    {
      fprintf(stderr, "[ calling KheDrsDim5TableGet(%p, %d), open_day_index "
	"%d, ec:\n", (void *) dc->counter_corr_dom_table5,
	unassigned_child_count, open_day_index);
      KheDrsExprDebug((KHE_DRS_EXPR) ec, drs, 2, 2, stderr);
      KheDrsOpenChildrenDebug(&ec->open_children_by_day, 2, 2, stderr);
      fprintf(stderr, "]\n");
    }
    corr_dom_table4 = KheDrsDim5TableGet(dc->counter_corr_dom_table5,
      unassigned_child_count);
    ** ***
    corr_dom_table4 = KheDrsDim5TableGet(ec->corr_dom_table5,
      unassigned_child_count);
    *** **
  }
  else
    corr_dom_table4 = NULL;
  return KheDrsDomTestMakeInt(KHE_DRS_DOM_TEST_SEPARATE, child_e, false, 0,
    INT_MAX, a, b, 0, NULL, corr_dom_table4, NULL, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprCounterExprRequiresShiftDomTest(                          */
/*    KHE_DRS_EXPR_COUNTER ec, int si,                                       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int *val)                             */
/*                                                                           */
/*  Find the number of unassigned children of ec there will be when the      */
/*  days before ds (the shift with open shift index si) begins are all       */
/*  assigned, and ds itself is assigned.  Return this number in *val, and    */
/*  return true if *val > 0.  This will mean that shift assignments for ds   */
/*  need a dom test, because assigning ds does not completely assign ec.     */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheDrsExprCounterExprRequiresShiftDomTest(
  KHE_DRS_EXPR_COUNTER ec, int si,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int *val)
{
  int di, assigned;  KHE_DRS_SHIFT ds;
  ds = HaArray(drs->open_shifts, si);
  di = ds->encl_day->open_day_index;
  assigned = KheDrsOpenChildrenBefore(&ec->open_children_by_day, di) +
    KheDrsOpenChildrenWithIndex(&ec->open_children_by_shift, si);
  *val = KheDrsOpenChildrenCount(&ec->open_children_by_day) - assigned;
  return *val > 0;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprCounterOldDomTest(                            */
/*    KHE_DRS_EXPR_COUNTER ec, int unassigned_child_count,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make and return a dominance test for ec, assuming that there are         */
/*  unassigned_child_count unassigned children (not including history).      */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_DOM_TEST KheDrsExprCounterOldDomTest(
  KHE_DRS_EXPR_COUNTER ec, int unassigned_child_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b;  KHE_DRS_DOM_TEST_TYPE dt_type;  KHE_DRS_CONSTRAINT dc;
  KHE_DRS_DIM2_TABLE main_dom_table2;

  ** calculate a **
  if( ec->max_limit < INT_MAX )
    a = ec->max_limit - unassigned_child_count;
  else
    a = INT_MAX;

  ** calculate b **
  if( ec->min_limit > 0 )
    b = ec->min_limit;
  else
    b = 0;

  ** make and return the dom test **
  dc = KheDrsExprCostConstraint((KHE_DRS_EXPR_COST) ec, drs);
  main_dom_table2 = KheDrsDim3TableGet(dc->counter_main_dom_table3,
    unassigned_child_count);
  ** ***
  main_dom_table2 = KheDrsDim3TableGet(ec->main_dom_table3,
    unassigned_child_count);
  *** **
  dt_type = KheDomTestTypeActual(drs->solve_dom_test_type,
    ec->cost_fn != KHE_QUADRATIC_COST_FUNCTION, false);
  return KheDrsDomTestMakeInt(dt_type, (KHE_DRS_EXPR) ec, ec->allow_zero,
    ec->min_limit, ec->max_limit, a, b, ec->combined_weight,
    main_dom_table2, NULL, ec->monitor->monitor, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprCounterDayDomTest(                            */
/*    KHE_DRS_EXPR_COUNTER ec, int open_day_index,                           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return the dom test for ec on the open day with index open_day_index.    */
/*                                                                           */
/*  When two solutions are compared using this dom test, the open children   */
/*  up to and including those on the day with index open_day_index will be   */
/*  be assigned, leaving                                                     */
/*                                                                           */
/*    KheDrsOpenChildrenAtOrAfter(ec, open_day_index + 1)                    */
/*                                                                           */
/*  unassigned children.                                                     */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_DOM_TEST KheDrsExprCounterOldDayDomTest(
  KHE_DRS_EXPR_COUNTER ec, int open_day_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int unassigned_child_count;
  unassigned_child_count = KheDrsOpenChildrenAtOrAfter(
    &ec->open_children_by_day, open_day_index + 1);
  return KheDrsExprCounterOldDomTest(ec, unassigned_child_count, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_COUNTER - general"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCounterCheck(KHE_DRS_EXPR_COUNTER ec,                     */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  A simple consistency check for ec, needed to track down a nasty bug.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprCounterCheck(KHE_DRS_EXPR_COUNTER ec,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( DEBUG79 && ec->closed_state > HaArrayCount(ec->children) )
  {
    fprintf(stderr, "%d# KheDrsExprCounterCheck closed_state (%d) > children"
      " (%d)\n", KheSolnDiversifier(drs->soln),
      ec->closed_state, HaArrayCount(ec->children));
    KheDrsExprDebug((KHE_DRS_EXPR) ec, drs, 2, 2, stderr);
    HnAbort("KheDrsExprCounterCheck internal error");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCounterCheckClosed(KHE_DRS_EXPR_COUNTER ec,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  A more elaborate consistency check which assumes that ec and its         */
/*  descendants are all closed, and checks that ec->closed_state is          */
/*  correct.                                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprCounterCheckClosed(KHE_DRS_EXPR_COUNTER ec,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int closed_state, i;  KHE_DRS_EXPR child_e;
  if( DEBUG79 )
  {
    closed_state = 0;
    HaArrayForEach(ec->children, child_e, i)
      if( !child_e->open )
	closed_state += child_e->value.i;
    if( closed_state != ec->closed_state )
    {
      fprintf(stderr, "%d# KheDrsExprCounterCheckClosed: actual closed "
	"state (%d) != stored value (%d)\n", KheSolnDiversifier(drs->soln),
	closed_state, ec->closed_state);
      KheDrsExprDebug((KHE_DRS_EXPR) ec, drs, 2, 2, stderr);
      HnAbort("KheDrsExprCounterCheckClosed internal error");
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprCounterDelta(KHE_DRS_EXPR_COUNTER ec,                      */
/*    int lower_det, int upper_det)                                          */
/*                                                                           */
/*  Return the deviation of ec for the given lower and upper determinants.   */
/*  This is function delta(l, u) from the documentation.                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprCounterDelta(KHE_DRS_EXPR_COUNTER ec,
  int lower_det, int upper_det)
{
  if( ec->allow_zero && lower_det == 0 )
    return 0;
  else if( lower_det > ec->max_limit )
    return lower_det - ec->max_limit;
  else if( upper_det < ec->min_limit )
    return ec->min_limit - upper_det;
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprCounterDev(KHE_DRS_EXPR_COUNTER ec,                        */
/*    int lower_det, int upper_det_minus_lower_det)                          */
/*                                                                           */
/*  Return the deviation of ec, assuming that ec has the given lower         */
/*  and upper determinants, with the upper determinant given by the amount   */
/*  by which it exceeds the lower determinant.  This may be more convenient. */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprCounterDev(KHE_DRS_EXPR_COUNTER ec,
  int lower_det, int upper_det_minus_lower_det)
{
  return KheDrsExprCounterDelta(ec, lower_det,
    lower_det + upper_det_minus_lower_det);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ADJUST_TYPE KheDrsAdjustType(KHE_COST_FUNCTION cost_fn,          */
/*    int max_limit)                                                         */
/*                                                                           */
/*  Return the adjust type, determined by the cost function and max limit.   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_ADJUST_TYPE KheDrsAdjustType(KHE_COST_FUNCTION cost_fn,
  int max_limit)
{
  if( max_limit == INT_MAX )
    return KHE_DRS_ADJUST_NO_MAX;
  else if( cost_fn == KHE_LINEAR_COST_FUNCTION )
    return KHE_DRS_ADJUST_LINEAR;
  else if( cost_fn == KHE_STEP_COST_FUNCTION )
    return KHE_DRS_ADJUST_STEP;
  else
    return KHE_DRS_ADJUST_ORDINARY;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsAdjustTypeShow(KHE_DRS_ADJUST_TYPE dct)                      */
/*                                                                           */
/*  Return a string representation of dct.                                   */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsAdjustTypeShow(KHE_DRS_ADJUST_TYPE dct)
{
  switch( dct )
  {
    case KHE_DRS_ADJUST_NO_MAX:	  return "NO_MAX";
    case KHE_DRS_ADJUST_LINEAR:	  return "LINEAR";
    case KHE_DRS_ADJUST_STEP:	  return "STEP";
    case KHE_DRS_ADJUST_ORDINARY: return "ORDINARY";
    default:			  return "??";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_COUNTER KheDrsExprCounterMakeBegin(                         */
/*    KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight,                   */
/*    KHE_DRS_MONITOR dm, int min_limit, int max_limit,                      */
/*    bool allow_zero, int history_before, int history_after,                */
/*    int history, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)     */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_COUNTER object with these attributes.            */
/*                                                                           */
/*  Implementation note.  The value 0 passed for value_ub is an initial      */
/*  value only.  It will be updated later by by AddChild.                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_COUNTER KheDrsExprCounterMakeBegin(
  KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight,
  KHE_DRS_MONITOR dm, int min_limit, int max_limit,
  bool allow_zero, int history_before, int history_after,
  int history, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_COUNTER res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_COUNTER_TAG,dr,drs);
  res->cost_fn = cost_fn;
  res->combined_weight = combined_weight;
  res->monitor = dm;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->history_before = history_before;
  res->history_after = history_after;
  res->history = history;
  res->adjust_type = KheDrsAdjustType(cost_fn, max_limit);
  res->closed_state = 0;
  res->allow_zero = allow_zero;
  KheDrsOpenChildrenInit(&res->open_children_by_shift,
    KHE_DRS_OPEN_CHILDREN_INDEX_SHIFT, false, drs);
  KheDrsAddWeight(drs, combined_weight);
  KheDrsExprCounterCheck(res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_COUNTER KheDrsExprCounterZeroMaxMakeBegin(                  */
/*    KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight,                   */
/*    KHE_DRS_MONITOR dm, KHE_DRS_RESOURCE dr,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make an COUNTER object for the case where there is a maximum limit       */
/*  of 0 (so the dominance test is <=) and no history.  This is the usual    */
/*  (but not the only) replacement for the now omitted EXPR_COST.            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_COUNTER KheDrsExprCounterZeroMaxMakeBegin(
  KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight,
  KHE_DRS_MONITOR dm, KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsExprCounterMakeBegin(cost_fn, combined_weight, dm,
    0, 0, false, 0, 0, 0, dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCounterAddChild(KHE_DRS_EXPR_COUNTER ec,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to ec.  Update ec's state          */
/*  appropriately.  This is similar to closing a child, but not identical.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprCounterAddChild(KHE_DRS_EXPR_COUNTER ec,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsExprCounterCheck(ec, drs);
  if( RERUN_PRINT(drs, ec->monitor) )
    fprintf(stderr, "%d# AddChild: closed_state (init %d) += %d\n",
      KheSolnDiversifier(drs->soln), ec->closed_state,
      child_e->value.i);
  ec->closed_state += child_e->value.i;
  KheDrsExprCounterCheck(ec, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCounterAdjustLimits(KHE_DRS_EXPR_COUNTER ec,              */
/*    int ub)                                                                */
/*                                                                           */
/*  Adjust the limits of ec.  Here ub is an upper bound on the sum of        */
/*  the values of the children, including history.                           */
/*                                                                           */
/*****************************************************************************/

/* *** I've decided that this is not worth doing
static void KheDrsExprCounterAdjustLimits(KHE_DRS_EXPR_COUNTER ec,
  int ub)
{
  ** make sure limits are consistent **
  HnAssert(ec->min_limit <= ec->max_limit,
    "KheDrsExprCounterAdjustLimits: before adjustment, inconsistent limits"
    " (min %d, max %d) for monitor %s", ec->min_limit, ec->max_limit,
    KheDrsMonitorId(ec->monitor));

  ** adjust ec->min_limit if possible **
  if( ec->min_limit < ec->history && ec->history <= ec->max_limit )
    ec->min_limit = ec->history;

  ** adjust ec->max_limit if possible **
  if( ec->max_limit > ub && ub >= ec->min_limit )
    ec->max_limit = ub;

  ** make sure limits are consistent **
  HnAssert(ec->min_limit <= ec->max_limit,
    "KheDrsExprCounterAdjustLimits: after adjustment, inconsistent limits"
    " (min %d, max %d) for monitor %s", ec->min_limit, ec->max_limit,
    KheDrsMonitorId(ec->monitor));
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprCounterValueUpperBound(KHE_DRS_EXPR_COUNTER ec, int ub)    */
/*                                                                           */
/*  Return a suitable value upper bound for es.  Here ub is an upper         */
/*  bound on the deviation of the sum of the values of the children,         */
/*  including history.  The value is stored but not actually used.           */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprCounterValueUpperBound(KHE_DRS_EXPR_COUNTER ec, int ub)
{
  int res;
  res = ec->allow_zero ? max(ec->min_limit - 1, 0) : ec->min_limit;
  if( res < ub - ec->max_limit )
    res = ub - ec->max_limit;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCounterMakeEnd(KHE_DRS_EXPR_COUNTER ec,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  End the initialization of ec.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprCounterMakeEnd(KHE_DRS_EXPR_COUNTER ec,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int ub;

  KheDrsExprInitEnd((KHE_DRS_EXPR) ec, drs);

  /* get the total value upper bound */
  ub = HaArrayCount(ec->children) + ec->history + ec->history_after;

  /* adjust limits */
  /* KheDrsExprCounterAdjustLimits(ec, ub); */

  /* set the value upper bound (not actually used, but anyway) */
  ec->value_ub.i = KheDrsExprCounterValueUpperBound(ec, ub);

  /* set constraint */
  KheDrsExprCostSetConstraint((KHE_DRS_EXPR_COST) ec, ec->history, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCounterChildHasOpened(KHE_DRS_EXPR_COUNTER ec,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening ec for solving, ec's child_index'th child,            */
/*  child_e, has been opened.  Update ec to reflect this, and also           */
/*  drs->solve_start_cost.                                                   */
/*                                                                           */
/*  The closed state of ec is the sum of the values of ec's closed           */
/*  children, so this update subtracts child_e's value (which preserves      */
/*  its value before opening) from the closed state.                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprCounterChildHasOpened(KHE_DRS_EXPR_COUNTER ec,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int old_dev, new_dev, ub;  KHE_COST old_cost, new_cost;
  if( RERUN_PRINT(drs, ec->monitor) )
  {
    fprintf(stderr, "%d# [ KheDrsExprCounterChildHasOpened(%s, child_e, "
      "%d):\n", KheSolnDiversifier(drs->soln),
      KheDrsMonitorId(ec->monitor), child_index);
    KheDrsExprDebug((KHE_DRS_EXPR) ec, drs, 2, 2, stderr);
  }
  KheDrsExprCounterCheck(ec, drs);

  /* find the deviation before the change */
  /* ***
  KheDrsOpenChildrenAddChildInDayOrder(&ec->open_children_by_day, child_e);
  *** */
  ub = KheDrsOpenChildrenCount(&ec->open_children_by_day);
  old_dev = KheDrsExprCounterDev(ec,
    ec->history + ec->closed_state, ub + ec->history_after);
  if( RERUN_PRINT(drs, ec->monitor) )
  {
    fprintf(stderr, "%d#   old_dev = %d = Dev(ec, %d + %d, %d - 1 + %d)\n",
    KheSolnDiversifier(drs->soln), old_dev, ec->history,
      ec->closed_state, ub, ec->history_after);
    fprintf(stderr, "%d#   closed_state (init %d) -= %d\n",
      KheSolnDiversifier(drs->soln), ec->closed_state, child_e->value.i);
  }

  /* update ec to reflect the new open child */
  KheDrsOpenChildrenAddChild(&ec->open_children_by_day, child_e);
  ec->closed_state -= child_e->value.i;

  /* find the deviation after the change */
  ub = KheDrsOpenChildrenCount(&ec->open_children_by_day);  /* one more */
  new_dev = KheDrsExprCounterDev(ec,
    ec->history + ec->closed_state, ub + ec->history_after);
  if( RERUN_PRINT(drs, ec->monitor) )
    fprintf(stderr, "%d#   new_dev = %d = Dev(ec, %d + %d, %d + %d)\n",
    KheSolnDiversifier(drs->soln), new_dev, ec->history,
    ec->closed_state, ub, ec->history_after);

  /* report the change in cost, if any, to drs->solve_start_cost */
  if( old_dev != new_dev || DEBUG78 )
  {
    old_cost = f(ec, old_dev);
    new_cost = f(ec, new_dev);
    drs->solve_start_cost += (new_cost - old_cost);
    if( RERUN_DEBUG(drs) && (new_cost - old_cost) != 0 )
      fprintf(stderr, "  KheDrsExprCounterChildHasOpened: solve_start_cost "
	"+= (%.5f - %.5f) = %.5f from %s\n", KheCostShow(new_cost),
	KheCostShow(old_cost), KheCostShow(drs->solve_start_cost), 
	KheDrsMonitorId(ec->monitor));
    KheDrsMonitorUpdateRerunCost(ec->monitor, (KHE_DRS_EXPR) ec, drs,
      NULL, KHE_DRS_OPEN, "open", child_index, "+-", new_cost, old_cost);
  }

  /* update open_children_by_shift, if required */
  if( ec->resource == NULL )
    KheDrsOpenChildrenAddChild(&ec->open_children_by_shift, child_e);

  /* all done */
  KheDrsExprCounterCheck(ec, drs);
  if( RERUN_PRINT(drs, ec->monitor) )
    fprintf(stderr, "%d# ] KheDrsExprCounterChildHasOpened\n",
      KheSolnDiversifier(drs->soln));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCounterChildHasClosed(KHE_DRS_EXPR_COUNTER ec,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing ec, ec's child_index'th child, child_e, has           */
/*  been closed.  Update ec's closed state to reflect this, and also         */
/*  drs->solve_start_cost.                                                   */
/*                                                                           */
/*  Although KheDrsExprChildHasOpened adds the newly opened child to         */
/*  e->open_children, KheDrsExprChildHasClosed does not remove it.  This     */
/*  is because the removal cannot be done efficiently while preserving       */
/*  the order of the open children.  However that does not matter here,      */
/*  and we need the correct number of open children, so we remove any one.   */
/*                                                                           */
/*  The closed state of ec is the sum of the values of ec's closed           */
/*  children, so this update adds child_e's value to the closed state.       */
/*                                                                           */
/*  Obsolete:                                                                */
/*  This is called after child_e is removed from ec->open_children.          */
/*  So when KheDrsExprCounterUpdateStartCost retrieves the number of         */
/*  open children, it gets the correct (that is, the changed) value.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprCounterChildHasClosed(KHE_DRS_EXPR_COUNTER ec,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int old_dev, new_dev, ub;  KHE_COST old_cost, new_cost;
  if( RERUN_PRINT(drs, ec->monitor) )
  {
    fprintf(stderr, "%d# [ KheDrsExprCounterChildHasClosed(%s, child_e, "
      "%d):\n", KheSolnDiversifier(drs->soln),
      KheDrsMonitorId(ec->monitor), child_index);
    KheDrsExprDebug((KHE_DRS_EXPR) ec, drs, 2, 2, stderr);
  }
  KheDrsExprCounterCheck(ec, drs);

  /* find the deviation before the change */
  ub = KheDrsOpenChildrenCount(&ec->open_children_by_day);
  old_dev = KheDrsExprCounterDev(ec,
    ec->history + ec->closed_state, ub + ec->history_after);
  if( RERUN_PRINT(drs, ec->monitor) )
  {
    fprintf(stderr, "%d#   old_dev = %d = Dev(ec, %d + %d, %d + %d)\n",
    KheSolnDiversifier(drs->soln), old_dev, ec->history,
      ec->closed_state, ub, ec->history_after);
    fprintf(stderr, "%d#   closed_state (init %d) += %d\n",
      KheSolnDiversifier(drs->soln), ec->closed_state, child_e->value.i);
  }

  /* update ec to reflect one less open child */
  KheDrsOpenChildrenDeleteChild(&ec->open_children_by_day, child_e);
  ec->closed_state += child_e->value.i;

  /* find the deviation after the change */
  ub = KheDrsOpenChildrenCount(&ec->open_children_by_day);  /* one less */
  new_dev = KheDrsExprCounterDev(ec,
    ec->history + ec->closed_state, ub + ec->history_after);
  if( RERUN_PRINT(drs, ec->monitor) )
    fprintf(stderr, "%d#   new_dev = %d = Dev(ec, %d + %d, %d + %d)\n",
    KheSolnDiversifier(drs->soln), new_dev, ec->history,
    ec->closed_state, ub, ec->history_after);

  /* report the change in cost, if any, to drs->solve_start_cost */
  if( old_dev != new_dev || DEBUG78 )
  {
    new_cost = f(ec, new_dev);
    old_cost = f(ec, old_dev);
    drs->solve_start_cost += (new_cost - old_cost);
    if( RERUN_DEBUG(drs) && (new_cost - old_cost) != 0 )
      fprintf(stderr, "  KheDrsResourceOnDayOpen: drs->solve_start_cost "
	"+= (%.5f - %.5f) = %.5f\n", KheCostShow(new_cost),
	KheCostShow(old_cost), KheCostShow(drs->solve_start_cost));
    KheDrsMonitorUpdateRerunCost(ec->monitor, (KHE_DRS_EXPR) ec, drs,
      NULL, KHE_DRS_CLOSE, "close", child_index, "+-", new_cost, old_cost);
  }

  /* update open_children_by_shift, if required */
  if( ec->resource == NULL )
    KheDrsOpenChildrenDeleteChild(&ec->open_children_by_shift, child_e);
  KheDrsExprCounterCheck(ec, drs);
  if( RERUN_PRINT(drs, ec->monitor) )
    fprintf(stderr, "%d# ] KheDrsExprCounterChildHasClosed\n",
      KheSolnDiversifier(drs->soln));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCounterSetClosedValue(KHE_DRS_EXPR_COUNTER ec,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ec, set its value suitably for its closed state.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprCounterSetClosedValue(KHE_DRS_EXPR_COUNTER ec,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here, since ec does not store a closed value */
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAdjustedSigVal(int val, KHE_DRS_ADJUST_TYPE adjust_type,       */
/*    int min_limit, int max_limit, int pc, int qc)                          */
/*                                                                           */
/*  Return the adjusted signature value.                                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAdjustedSigVal(int val, KHE_DRS_ADJUST_TYPE adjust_type,
  int min_limit, int max_limit, int history_after)
{
  int lbar;
  switch( adjust_type )
  {
    case KHE_DRS_ADJUST_ORDINARY:

      return val;

    case KHE_DRS_ADJUST_NO_MAX:

      lbar = max(0, min_limit - history_after);
      return min(lbar, val);

    case KHE_DRS_ADJUST_LINEAR:

      lbar = max_limit;
      return min(lbar, val);

    case KHE_DRS_ADJUST_STEP:

      lbar = max_limit + 1;
      return min(lbar, val);

    default:

      HnAbort("KheDrsAdjustedSigVal internal error (adjusttype)");
      return val;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprCounterInitialValue(KHE_DRS_EXPR_COUNTER ec)               */
/*                                                                           */
/*  Return the initial value of ec.  This is used just below, but it is      */
/*  also used for dominance testing of correlated expressions.               */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprCounterInitialValue(KHE_DRS_EXPR_COUNTER ec)
{
  return ec->history + ec->closed_state;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCounterUpdateLU(KHE_DRS_EXPR child_e,                     */
/*    int *ld2, int *ud2, int i, bool debug)                                 */
/*                                                                           */
/*  Update *ld2 and *ud2 to take account of the fact that child_e now has    */
/*  a value.  Parameters i and debug are for debugging only.                 */
/*                                                                           */
/*  Implementation note.  When child_e goes from unassigned to assigned,     */
/*  its value becomes available and is added to the lower determinant.       */
/*  It is also added to the upper determinant (which includes everything     */
/*  in the lower determinant), but its upper bound, which was previously     */
/*  part of the upper determinant, has to be taken away.                     */
/*                                                                           */
/*  To understand the old version below, if child_e goes from unassigned to  */
/*  active, there is one more active child, but active + unassigned stays    */
/*  the same.  If child_e goes from unassigned to inactive, the number of    */
/*  active children does not change, but active + unassigned decreases by 1. */
/*                                                                           */
/*  It is not hard to see that the two versions are the same.  Consider      */
/*  that child_e->value_ub.i == 1 always, and child_e->value.i == 0 or 1.    */
/*  If child_e->value.i == 0, the new version leaves *ld2 unchanged and      */
/*  subtracts 1 from *ud2.  If child_e->value.i == 1, the new version adds   */
/*  1 to *ld2 and leaves *ud2 unchanged.  The old version does the same.     */
/*                                                                           */
/*  We've chosen the current version because it generalizes to SUM_INT       */
/*  and SUM_FLOAT evaluations.  These have children whose values and         */
/*  value upper bounds are arbitrary; active and inactive don't apply.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprCounterUpdateLU(KHE_DRS_EXPR child_e,
  int *ld2, int *ud2, int i, bool debug)
{
  if( DEBUG68 && child_e == NULL )
    fprintf(stderr, "  KheDrsExprCounterUpdateLU failing\n");
  *ld2 += child_e->value.i;
  *ud2 += (child_e->value.i - child_e->value_ub.i);
  if( debug )
    fprintf(stderr, "%d:+%d-%d", i, child_e->value.i, child_e->value_ub.i);

  /* *** old version, see comment above
  if( child_e->value.i == 1 )
  {
    *ld2 += 1;  ** unassigned to active **
    if( debug )
      fprintf(stderr, "%d+", i);
  }
  else
  {
    *ud2 -= 1;  ** unassigned to inactive **
    if( debug )
      fprintf(stderr, "%d-", i);
  }
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCounterEvalSignature(KHE_DRS_EXPR_COUNTER ec,             */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Evaluate expression ec, assuming that next_di has just been assigned,    */
/*  and update the cost and states of sig with the result.                   */
/*                                                                           */
/*  Terminology used here        Terminology used in doc                     */
/*  --------------------------------------------------------                 */
/*  ld1                          l(m, S)                                     */
/*  ud1                          u(m, S)                                     */
/*  dev1                         delta(l(m, S), u(m, S)                      */
/*  cost1                        f(delta(l(m, S), u(m, S))                   */
/*  --------------------------------------------------------                 */
/*  ld2                          l(m, S')                                    */
/*  ud2                          u(m, S')                                    */
/*  dev2                         delta(l(m, S'), u(m, S')                    */
/*  cost2                        f(delta(l(m, S'), u(m, S'))                 */
/*  --------------------------------------------------------                 */
/*                                                                           */
/*  Here ld1 is the number of children that are assigned and have value 1    */
/*  in S, including history children, and ud1 is ld1 plus the number of      */
/*  children that are unassigned in S, including history after children.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprCounterEvalSignature(KHE_DRS_EXPR_COUNTER ec,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, /* int next_di, */
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)
{
  int ld1, ld2, ud1, ud2, i, i1, i2, dev1, dev2, si, count, next_di;
  KHE_DRS_EXPR child_e;  KHE_DRS_VALUE val;  /* bool must_store_state; */
  KHE_DRS_EXPR_EVAL_TYPE eval_type;

  /* get ld1, ud1, and dev1 (for all days before next_di) */
  next_di = KheDrsSignerOpenDayIndex(dsg);
  if( debug )
  {
    fprintf(stderr, "  [ eval %s, next_di %d, open_children_by_day:\n",
      KheDrsMonitorId(ec->monitor), next_di);
    KheDrsOpenChildrenDebug(&ec->open_children_by_day, 1, 4, stderr);
  }
  if( KheDrsOpenChildrenIndexIsFirstOrLess(&ec->open_children_by_day,next_di))
  {
    ld1 = KheDrsExprCounterInitialValue(ec);
    if( debug )
      fprintf(stderr, "    ld1 %d + %d", ec->history,
	ec->closed_state);
  }
  else
  {
    ld1 = KheDrsExprDaySigVal((KHE_DRS_EXPR) ec, next_di - 1, prev_sig).i;
    if( debug )
      fprintf(stderr, "    ld1 %d", ld1);
    if( DEBUG91(ec->monitor) && ld1 == 4 )
      fprintf(stderr, "  debug expr retrieving value 4 (next_di %d, %s)\n",
	next_di, KheDrsSignerTypeShow(dsg->type));
  }
  ud1 = ld1 + ec->history_after +
    KheDrsOpenChildrenAtOrAfter(&ec->open_children_by_day, next_di);
  dev1 = KheDrsExprCounterDelta(ec, ld1, ud1);
  if( debug )
    fprintf(stderr, " ud1 %d + %d + %d dev1 %d\n", ld1, ec->history_after,
      KheDrsOpenChildrenAtOrAfter(&ec->open_children_by_day, next_di), dev1);

  /* get ld2, ud2, and dev2 for one more day, shift, or shift pair */
  ld2 = ld1;
  ud2 = ud1;
  /* must_store_state = false; */  /* keep compiler happy */
  switch( dsg->type )
  {
    case KHE_DRS_SIGNER_DAY:
    case KHE_DRS_SIGNER_RESOURCE_ON_DAY:

      /* signer is for day or mtask solutions */
      if( debug )
	fprintf(stderr, " next_di %d ", next_di);
      KheDrsOpenChildrenForEach(&ec->open_children_by_day, next_di,child_e,i)
        KheDrsExprCounterUpdateLU(child_e, &ld2, &ud2, i, debug);
      /* ***
      must_store_state =
	!KheDrsOpenChildrenIndexIsLast(&ec->open_children_by_day, next_di);
      *** */
      break;

    case KHE_DRS_SIGNER_SHIFT:

      /* signer is for shift solutions */
      if( DEBUG68 )
	fprintf(stderr, "  evaluating dsg %p, ec %p\n", (void *) dsg,
	  (void *) ec);
      si = dsg->u.shift->open_shift_index;
      if( debug )
      {
	fprintf(stderr, "    open_children_by_shift:\n");
	KheDrsOpenChildrenDebug(&ec->open_children_by_shift, 1, 4, stderr);
	fprintf(stderr, "    si %d ", si);
      }
      KheDrsOpenChildrenForEach(&ec->open_children_by_shift, si, child_e, i)
        KheDrsExprCounterUpdateLU(child_e, &ld2, &ud2, i, debug);
      /* ***
      must_store_state =
	(KheDrsOpenChildrenWithIndex(&ec->open_children_by_day, next_di)
	> KheDrsOpenChildrenWithIndex(&ec->open_children_by_shift, si));
      *** */
      /* ***
      must_store_state =  ** wrong! just being last is not the whole deal **
	!KheDrsOpenChildrenIndexIsLast(&ec->open_children_by_shift, si);
      *** */
      break;

    case KHE_DRS_SIGNER_SHIFT_PAIR:

      /* signer is for shift pair solutions */
      if( debug )
      {
	fprintf(stderr, "    open_children_by_shift:\n");
	KheDrsOpenChildrenDebug(&ec->open_children_by_shift, 1, 4, stderr);
      }
      for( count = 0;  count < 2;  count++ )
      {
	si = dsg->u.shift_pair->shift[count]->open_shift_index;
	if( debug )
	  fprintf(stderr, "    si[%d] %d ", count, si);
	KheDrsOpenChildrenForEach(&ec->open_children_by_shift, si, child_e, i)
	  KheDrsExprCounterUpdateLU(child_e, &ld2, &ud2, i, debug);
	/* *** KheDrsSignerTypeShow(dsg->type)
	must_store_state =  ** wrong **
	  !KheDrsOpenChildrenIndexIsLast(&ec->open_children_by_shift, si);
	*** */
      }
      break;

    default:

      HnAbort("KheDrsExprCounterEvalSignature internal error 2");
      break;

  }
  dev2 = KheDrsExprCounterDelta(ec, ld2, ud2);
  if( debug )
    fprintf(stderr, " ld2 %d ud2 %d dev2 %d", ld2, ud2, dev2);

  /* if not the last evaluation, store ld2 (adjusted) in next_sig */
  eval_type = KheDrsExprEvalType((KHE_DRS_EXPR) ec, dsg, drs, NULL);
  if( eval_type == KHE_DRS_EXPR_EVAL_NOT_LAST )
  {
    val.i = KheDrsAdjustedSigVal(ld2, ec->adjust_type,
      ec->min_limit, ec->max_limit, ec->history_after);
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) ec);
    if( DEBUG91(ec->monitor) && val.i == 4 )
      fprintf(stderr, "  debug expr storing value 4 (ld1 %d, ud1 %d, ld2 %d, "
	"ud2 %d, next_di %d, %s)\n", ld1, ud1, ld2, ud2, next_di,
	KheDrsSignerTypeShow(dsg->type));
    if( debug )
      fprintf(stderr, " S%d", val.i);
  }

  /* report the extra cost, if any */
  if( dev2 != dev1 )
  {
    KheDrsSignatureAddCost(next_sig, f(ec, dev2) - f(ec, dev1));
    if( RERUN_DEBUG(drs) )
      fprintf(stderr, "  KheDrsExprCounterEvalSignature: sig->cost += "
	"(%.5f - %.5f) = %.5f from %s\n", KheCostShow(f(ec, dev2)),
	KheCostShow(f(ec, dev1)), KheCostShow(next_sig->cost),
	KheDrsMonitorId(ec->monitor));
    /* +++
    if( RERUN_DEBUG(drs) )
      fprintf(stderr, "  KheDrsExprCounterEvalSignature:  ");
    *** */
    KheDrsMonitorUpdateRerunCost(ec->monitor, (KHE_DRS_EXPR) ec, drs,
      dsg, KHE_DRS_SEARCH, "search", -1, "+-", f(ec, dev2), f(ec, dev1));
    if( debug )
      fprintf(stderr, " C%.5f", KheCostShow(f(ec, dev2) - f(ec, dev1)));
  }
  if( debug )
    fprintf(stderr, "\n  ]\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCounterDoDebug(KHE_DRS_EXPR_COUNTER ec,                   */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ec which is specific to it.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprCounterDoDebug(KHE_DRS_EXPR_COUNTER ec,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "COUNTER(%s, %s, closed_state %d)", KheDrsMonitorId(ec->monitor),
    KheDrsAdjustTypeShow(ec->adjust_type), ec->closed_state);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ec, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_SUM_INT - domninance testing"                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSumIntUpperBound(KHE_DRS_EXPR_SUM_INT esi, int child_index)*/
/*                                                                           */
/*  Return an upper bound on the values of the children at or after          */
/*  child_index.                                                             */
/*                                                                           */
/*****************************************************************************/

/* ***
static int KheDrsExprSumIntUpperBound(KHE_DRS_EXPR_SUM_INT esi, int child_index)
{
  return HaArray(esi->value_upper_bounds, child_index);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprSumIntOldDomTest(KHE_DRS_EXPR_SUM_INT esi,    */
/*    int unassigned_child_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Make and return a dominance test for esi, assuming that there are        */
/*  unassigned_child_count unassigned children (not including history).      */
/*                                                                           */
/*****************************************************************************/

/* *** never correct and now unwanted
static KHE_DRS_DOM_TEST KheDrsExprSumIntDoDomTest(KHE_DRS_EXPR_SUM_INT esi,
  int unassigned_child_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b;  KHE_DRS_DOM_TEST_TYPE dt_type;  KHE_DRS_CONSTRAINT dc;
  KHE_DRS_DIM2_TABLE main_dom_table2;

  ** calculate a **
  if( esi->max_limit < INT_MAX )
    a = esi->max_limit - unassigned_child_count;
  else
    a = INT_MAX;

  ** calculate b **
  if( esi->min_limit > 0 )
    b = esi->min_limit;
  else
    b = 0;

  ** make and return the dom test **
  dc = KheDrsExprCostConstraint((KHE_DRS_EXPR_COST) esi, drs);
  main_dom_table2 = KheDrsDim3TableGet(dc->counter_main_dom_table3,
    unassigned_child_count);
  dt_type = KheDomTestTypeActual(drs->solve_dom_test_type,
    false, ** esi->cost_fn != KHE_QUADRATIC_COST_FUNCTION, ** false);
  return KheDrsDomTestMakeInt(dt_type, (KHE_DRS_EXPR) esi, esi->allow_zero,
    ** esi->min_limit, esi->max_limit, ** a, b, esi->combined_weight,
    main_dom_table2, NULL, esi->monitor->monitor, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprSumIntDomTest(KHE_DRS_EXPR_SUM_INT esi,       */
/*    int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return a dom test suited to esi on day open_day_index, taking the        */
/*  context of esi into account.                                             */
/*                                                                           */
/*  Implementation note.  The documentation divides sum expressions into     */
/*  four cases.  However only cases (1) and (4) apply here.                  */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprSumIntDomTest(KHE_DRS_EXPR_SUM_INT esi,
  int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b, ub;
  if( esi->combined_weight > 0 )
  {
    /* case (1), sum monitor */
    a = -1;
    b = 0;
    return KheDrsDomTestMakeInt(KHE_DRS_DOM_TEST_SEPARATE_INT,
      (KHE_DRS_EXPR) esi, esi->allow_zero, a, b, esi->combined_weight,
      NULL, NULL, esi->monitor, drs);
  }
  else
  {
    /* case (4), child of a sum monitor */
    ub = KheDrsOpenChildrenUpperBoundInt(&esi->open_children_by_day,
      open_day_index + 1);
    a = esi->max_limit - ub;
    b = esi->min_limit;
    return KheDrsDomTestMakeInt(KHE_DRS_DOM_TEST_SEPARATE_INT,
      (KHE_DRS_EXPR) esi, esi->allow_zero, a, b, 0, NULL, NULL, NULL, drs);
  }
  /* ***
  int unassigned_child_count;
  unassigned_child_count = KheDrsOpenChildrenAtOrAfter(
    &esi->open_children_by_day, open_day_index + 1);
  return KheDrsExprSumIntDoDomTest(esi, unassigned_child_count, drs);
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprSumIntChildDomTest(KHE_DRS_EXPR_SUM_INT esi,  */
/*    int open_day_index, KHE_DRS_EXPR child_e,                              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return the dom test for a child of esi on its open_day_index'th          */
/*  open day.                                                                */
/*                                                                           */
/*  Note.  If esi is derived from a limit workload constraint, this          */
/*  code will not attempt to access any tables.                              */
/*                                                                           */
/*****************************************************************************/

/* *** needs to be replaced by KheDrsExprDomTest
static KHE_DRS_DOM_TEST KheDrsExprSumIntChildDomTest(KHE_DRS_EXPR_SUM_INT esi,
  int open_day_index, KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b, unassigned_child_count;  KHE_DRS_DIM4_TABLE corr_dom_table4;
  KHE_DRS_CONSTRAINT dc;

  ** calculate a **
  a = (esi->max_limit < INT_MAX ? -1 : INT_MAX);

  ** calculate b **
  b = (esi->min_limit > 0 ? INT_MAX : 0);

  ** make and return the dom test **
  if( child_e->tag == KHE_DRS_EXPR_OR_TAG )
  {
    unassigned_child_count = KheDrsOpenChildrenAtOrAfter(
      &esi->open_children_by_day, open_day_index + 1);
    dc = KheDrsExprCostConstraint((KHE_DRS_EXPR_COST) esi, drs);
    if( DEBUG57 )
    {
      fprintf(stderr, "[ calling KheDrsDim5TableGet(%p, %d), open_day_index "
	"%d, esi:\n", (void *) dc->counter_corr_dom_table5,
	unassigned_child_count, open_day_index);
      KheDrsExprDebug((KHE_DRS_EXPR) esi, drs, 2, 2, stderr);
      KheDrsOpenChildrenDebug(&esi->open_children_by_day, 2, 2, stderr);
      fprintf(stderr, "]\n");
    }
    corr_dom_table4 = KheDrsDim5TableGet(dc->counter_corr_dom_table5,
      unassigned_child_count);
  }
  else
    corr_dom_table4 = NULL;
  return KheDrsDomTestMakeInt(KHE_DRS_DOM_TEST_SEPARATE, child_e, false, 0,
    INT_MAX, a, b, 0, NULL, corr_dom_table4, NULL, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_SUM_INT - general"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumIntCheck(KHE_DRS_EXPR_SUM_INT esi,                     */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  A simple consistency check for esi, needed to track down a nasty bug.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumIntCheck(KHE_DRS_EXPR_SUM_INT esi,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* *** not sensible for EXPR_SUM as it stands
  if( DEBUG79 && esi->tag == KHE_DRS_EXPR_SUM_INT &&
      esi->closed_state.i > HaArrayCount(esi->children) )
  {
    fprintf(stderr, "%d# KheDrsExprSumIntCheck closed_state (%d) > children"
      " (%d)\n", KheSolnDiversifier(drs->soln),
      esi->closed_state.i, HaArrayCount(esi->children));
    KheDrsExprDebug((KHE_DRS_EXPR) esi, drs, 2, 2, stderr);
    HnAbort("KheDrsExprSumIntCheck internal error");
  }
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumIntCheckClosed(KHE_DRS_EXPR_SUM_INT esi,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  A more elaborate consistency check which assumes that esi and its        */
/*  descendants are all closed, and checks that esi->closed_state is         */
/*  correct.                                                                 */
/*                                                                           */
/*****************************************************************************/

/* currently unused
static void KheDrsExprSumIntCheckClosed(KHE_DRS_EXPR_SUM_INT esi,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int closed_state, i;  KHE_DRS_EXPR child_e;
  if( DEBUG79 )
  {
    closed_state = 0;
    HaArrayForEach(esi->children, child_e, i)
      if( !child_e->open )
	closed_state += child_e->value.i;
    if( closed_state != esi->closed_state.i )
    {
      fprintf(stderr, "%d# KheDrsExprSumIntCheckClosed: actual closed "
	"state (%d) != stored value (%d)\n", KheSolnDiversifier(drs->soln),
	closed_state, esi->closed_state.i);
      KheDrsExprDebug((KHE_DRS_EXPR) esi, drs, 2, 2, stderr);
      HnAbort("KheDrsExprSumIntCheckClosed internal error");
    }
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSumIntDelta(KHE_DRS_EXPR_SUM_INT esi, int lower_det,       */
/*    int upper_det)                                                         */
/*                                                                           */
/*  Return the deviation of esi for the given lower and upper determinants.  */
/*  This is function delta(l, u) from the documentation.                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprSumIntDelta(KHE_DRS_EXPR_SUM_INT esi, int lower_det,
  int upper_det)
{
  if( esi->allow_zero && lower_det == 0 )
    return 0;
  else if( lower_det > esi->max_limit )
    return lower_det - esi->max_limit;
  else if( upper_det < esi->min_limit )
    return esi->min_limit - upper_det;
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSumIntDev(KHE_DRS_EXPR_SUM_INT esi, int lower_det,         */
/*    int upper_det_minus_lower_det)                                         */
/*                                                                           */
/*  Return the deviation of esi, assuming that esi has the given lower and   */
/*  upper determinants, with the upper determinant given by the amount by    */
/*  which it exceeds the lower determinant.  This may be more convenient.    */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprSumIntDev(KHE_DRS_EXPR_SUM_INT esi, int lower_det,
  int upper_det_minus_lower_det)
{
  return KheDrsExprSumIntDelta(esi, lower_det,
    lower_det + upper_det_minus_lower_det);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ADJUST_TYPE KheDrsAdjustType(KHE_COST_FUNCTION cost_fn,          */
/*    int max_limit)                                                         */
/*                                                                           */
/*  Return the adjust type, determined by the cost function and max limit.   */
/*                                                                           */
/*****************************************************************************/

/* *** can reuse earlier version
static KHE_DRS_ADJUST_TYPE KheDrsAdjustType(KHE_COST_FUNCTION cost_fn,
  int max_limit)
{
  if( max_limit == INT_MAX )
    return KHE_DRS_ADJUST_NO_MAX;
  else if( cost_fn == KHE_LINEAR_COST_FUNCTION )
    return KHE_DRS_ADJUST_LINEAR;
  else if( cost_fn == KHE_STEP_COST_FUNCTION )
    return KHE_DRS_ADJUST_STEP;
  else
    return KHE_DRS_ADJUST_ORDINARY;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsAdjustTypeShow(KHE_DRS_ADJUST_TYPE dct)                      */
/*                                                                           */
/*  Return a string representation of dct.                                   */
/*                                                                           */
/*****************************************************************************/

/* *** can reuse earlier version
static char *KheDrsAdjustTypeShow(KHE_DRS_ADJUST_TYPE dct)
{
  switch( dct )
  {
    case KHE_DRS_ADJUST_NO_MAX:	  return "NO_MAX";
    case KHE_DRS_ADJUST_LINEAR:	  return "LINEAR";
    case KHE_DRS_ADJUST_STEP:	  return "STEP";
    case KHE_DRS_ADJUST_ORDINARY: return "ORDINARY";
    default:			  return "??";
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_SUM_INT KheDrsExprSumIntMakeBegin(KHE_DRS_MONITOR dm,       */
/*    KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight, int min_limit,    */
/*    int max_limit, bool allow_zero, int history_before, int history_after, */
/*    int history, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)     */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_SUM_INT object with these attributes.            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_SUM_INT KheDrsExprSumIntMakeBegin(KHE_DRS_MONITOR dm,
  KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight, int min_limit,
  int max_limit, bool allow_zero, int history_before, int history_after,
  int history, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_SUM_INT res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_SUM_INT_TAG, dr, drs);
  res->cost_fn = cost_fn;
  res->combined_weight = combined_weight;
  res->monitor = dm;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->history_before = history_before;
  res->history_after = history_after;
  res->history = history;
  res->adjust_type = KheDrsAdjustType(cost_fn, max_limit);
  res->closed_state = 0;
  res->allow_zero = allow_zero;
  KheDrsAddWeight(drs, combined_weight);
  KheDrsExprSumIntCheck(res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_SUM_INT KheDrsExprSumIntMakeRootBegin(KHE_DRS_MONITOR dm,   */
/*    KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight,                   */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Make a root sum int expression.                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_SUM_INT KheDrsExprSumIntMakeRootBegin(KHE_DRS_MONITOR dm,
  KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight, int min_limit,
  int max_limit, bool allow_zero, KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsExprSumIntMakeBegin(dm, cost_fn, combined_weight,
    min_limit, max_limit, allow_zero, 0, 0, 0, dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_SUM_INT KheDrsExprSumIntMakeNonRootBegin(int min_limit,     */ /*    int max_limit, bool allow_zero, KHE_DRS_RESOURCE dr,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a non-root sum int expression.                                      */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_SUM_INT KheDrsExprSumIntMakeNonRootBegin(int min_limit,
  int max_limit, bool allow_zero, KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsExprSumIntMakeBegin(NULL, KHE_LINEAR_COST_FUNCTION, 0,
    min_limit, max_limit, allow_zero, 0, 0, 0, dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumIntAddChild(KHE_DRS_EXPR_SUM_INT esi,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to esi.  Update esi's state        */
/*  appropriately.  This is similar to closing a child, but not identical.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumIntAddChild(KHE_DRS_EXPR_SUM_INT esi,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsExprSumIntCheck(esi, drs);
  if( RERUN_PRINT(drs, esi->monitor) )
    fprintf(stderr, "%d# AddChild: closed_state (init %d) += %d\n",
      KheSolnDiversifier(drs->soln), esi->closed_state, child_e->value.i);
  esi->closed_state += child_e->value.i;
  KheDrsExprSumIntCheck(esi, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumIntAdjustLimits(KHE_DRS_EXPR_SUM_INT esi, int ub)      */
/*                                                                           */
/*  Adjust the limits of esi.  Here ub is an upper bound on the sum of       */
/*  the values of the children, including history.                           */
/*                                                                           */
/*****************************************************************************/

/* *** I've decided that this is not worth doing
static void KheDrsExprSumIntAdjustLimits(KHE_DRS_EXPR_SUM_INT esi, int ub)
{
  ** make sure limits are consistent **
  HnAssert(esi->min_limit <= esi->max_limit,
    "KheDrsExprSumIntAdjustLimits: before adjustment, inconsistent limits"
    " (min %d, max %d) for monitor %s", esi->min_limit, esi->max_limit,
    esi->monitor != NULL ? KheDrsMonitorId(esi->monitor) : "-");

  ** adjust esi->min_limit if possible **
  if( esi->min_limit < esi->history && esi->history <= esi->max_limit )
    esi->min_limit = esi->history;

  ** adjust esi->max_limit if possible **
  if( esi->max_limit > ub && ub >= esi->min_limit )
    esi->max_limit = ub;

  ** make sure limits are consistent **
  HnAssert(esi->min_limit <= esi->max_limit,
    "KheDrsExprSumIntAdjustLimits: after adjustment, inconsistent limits"
    " (min %d, max %d) for monitor %s", esi->min_limit, esi->max_limit,
    esi->monitor != NULL ? KheDrsMonitorId(esi->monitor) : "-");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSumIntValueUpperBound(KHE_DRS_EXPR_SUM_INT esi, int ub)    */
/*                                                                           */
/*  Return a suitable value upper bound for esi.  Here ub is an upper bound  */
/*  on the sum of the values of the children, including history.             */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprSumIntValueUpperBound(KHE_DRS_EXPR_SUM_INT esi, int ub)
{
  int res;
  res = esi->allow_zero ? max(esi->min_limit - 1, 0) : esi->min_limit;
  if( res < ub - esi->max_limit )
    res = ub - esi->max_limit;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumIntMakeEnd(KHE_DRS_EXPR_SUM_INT esi,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  End the initialization of esi.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumIntMakeEnd(KHE_DRS_EXPR_SUM_INT esi,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int ub, i;  KHE_DRS_EXPR child_e;

  KheDrsExprInitEnd((KHE_DRS_EXPR) esi, drs);

  /* get the total value upper bound, including history */
  ub = esi->history + esi->history_after;
  HaArrayForEach(esi->children, child_e, i)
    ub += child_e->value_ub.i;

  /* adjust limits */
  /* KheDrsExprSumIntAdjustLimits(esi, ub); */

  /* set the value upper bound */
  esi->value_ub.i = KheDrsExprSumIntValueUpperBound(esi, ub);

  /* set constraint - NO! */
  /* ***
  if( esi->combined_weight > 0 )
    KheDrsExprCostSetConstraint((KHE_DRS_EXPR_COST) esi, 0, drs);
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprSumIntReportsCost(KHE_DRS_EXPR_SUM_INT esi)               */
/*                                                                           */
/*  Return true if esi reports a cost.                                       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprSumIntReportsCost(KHE_DRS_EXPR_SUM_INT esi)
{
  return esi->combined_weight > 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumIntChildHasOpened(KHE_DRS_EXPR_SUM_INT esi,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening esi for solving, esi's child_index'th child,          */
/*  child_e, has been opened.  Update esi to reflect this, and also          */
/*  drs->solve_start_cost if required.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumIntChildHasOpened(KHE_DRS_EXPR_SUM_INT esi,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int old_dev, new_dev, ub;  KHE_COST old_cost, new_cost;
  if( RERUN_PRINT(drs, esi->monitor) )
  {
    fprintf(stderr, "%d# [ KheDrsExprSumIntChildHasOpened(%s, child_e, "
      "%d):\n", KheSolnDiversifier(drs->soln),
      KheDrsMonitorId(esi->monitor), child_index);
    KheDrsExprDebug((KHE_DRS_EXPR) esi, drs, 2, 2, stderr);
  }
  KheDrsExprSumIntCheck(esi, drs);

  /* find the deviation before the change */
  ub = KheDrsOpenChildrenUpperBoundIntAll(&esi->open_children_by_day);
  old_dev = KheDrsExprSumIntDev(esi, esi->history + esi->closed_state,
    ub + esi->history_after);
  if( RERUN_PRINT(drs, esi->monitor) )
  {
    fprintf(stderr, "%d#   old_dev = %d = Dev(esi, %d + %d, %d + %d)\n",
    KheSolnDiversifier(drs->soln), old_dev, esi->history, esi->closed_state,
      ub, esi->history_after);
    fprintf(stderr, "%d#   closed_state (init %d) -= %d\n",
      KheSolnDiversifier(drs->soln), esi->closed_state, child_e->value.i);
  }

  /* update esi to reflect the new open child */
  KheDrsOpenChildrenAddChild(&esi->open_children_by_day, child_e);
  esi->closed_state -= child_e->value.i;

  /* find the deviation after the change */
  ub = KheDrsOpenChildrenUpperBoundIntAll(&esi->open_children_by_day);
  new_dev = KheDrsExprSumIntDev(esi, esi->history + esi->closed_state,
    ub + esi->history_after);
  if( RERUN_PRINT(drs, esi->monitor) )
    fprintf(stderr, "%d#   new_dev = %d = Dev(esi, %d + %d, %d + %d)\n",
    KheSolnDiversifier(drs->soln), new_dev, esi->history, esi->closed_state,
    ub, esi->history_after);

  /* report the change in cost, if any, to drs->solve_start_cost */
  if( KheDrsExprSumIntReportsCost(esi) && (old_dev != new_dev || DEBUG78) )
  {
    old_cost = f(esi, old_dev);
    new_cost = f(esi, new_dev);
    drs->solve_start_cost += (new_cost - old_cost);
    if( RERUN_DEBUG(drs) && (new_cost - old_cost) != 0 )
      fprintf(stderr, "  KheDrsExprSumIntChildHasOpened: drs->solve_start_cost "
	"+= (%.5f - %.5f) = %.5f\n", KheCostShow(new_cost),
	KheCostShow(old_cost), KheCostShow(drs->solve_start_cost));
    KheDrsMonitorUpdateRerunCost(esi->monitor, (KHE_DRS_EXPR) esi, drs,
      NULL, KHE_DRS_OPEN, "open", child_index, "+-", new_cost, old_cost);
  }

  /* all done */
  KheDrsExprSumIntCheck(esi, drs);
  if( RERUN_PRINT(drs, esi->monitor) )
    fprintf(stderr, "%d# ] KheDrsExprSumIntChildHasOpened\n",
      KheSolnDiversifier(drs->soln));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumIntChildHasClosed(KHE_DRS_EXPR_SUM_INT esi,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing esi, esi's child_index'th child, child_e, has         */
/*  been closed.  Update esi's closed state to reflect this, and also        */
/*  drs->solve_start_cost if required.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumIntChildHasClosed(KHE_DRS_EXPR_SUM_INT esi,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int old_dev, new_dev, ub;  KHE_COST old_cost, new_cost;
  if( RERUN_PRINT(drs, esi->monitor) )
  {
    fprintf(stderr, "%d# [ KheDrsExprSumIntChildHasClosed(%s, child_e, "
      "%d):\n", KheSolnDiversifier(drs->soln),
      KheDrsMonitorId(esi->monitor), child_index);
    KheDrsExprDebug((KHE_DRS_EXPR) esi, drs, 2, 2, stderr);
  }
  KheDrsExprSumIntCheck(esi, drs);

  /* find the deviation before the change */
  ub = KheDrsOpenChildrenUpperBoundIntAll(&esi->open_children_by_day);
  old_dev = KheDrsExprSumIntDev(esi, esi->history + esi->closed_state,
    ub + esi->history_after);
  if( RERUN_PRINT(drs, esi->monitor) )
  {
    fprintf(stderr, "%d#   old_dev = %d = Dev(esi, %d + %d, %d + %d)\n",
    KheSolnDiversifier(drs->soln), old_dev, esi->history,
      esi->closed_state, ub, esi->history_after);
    fprintf(stderr, "%d#   closed_state (init %d) += %d\n",
      KheSolnDiversifier(drs->soln), esi->closed_state, child_e->value.i);
  }

  /* update esi to reflect one less open child */
  KheDrsOpenChildrenDeleteChild(&esi->open_children_by_day, child_e);
  esi->closed_state += child_e->value.i;

  /* find the deviation after the change */
  ub = KheDrsOpenChildrenUpperBoundIntAll(&esi->open_children_by_day);
  new_dev = KheDrsExprSumIntDev(esi, esi->history + esi->closed_state,
    ub + esi->history_after);
  if( RERUN_PRINT(drs, esi->monitor) )
    fprintf(stderr, "%d#   new_dev = %d = Dev(esi, %d + %d, %d + %d)\n",
    KheSolnDiversifier(drs->soln), new_dev, esi->history,
    esi->closed_state, ub, esi->history_after);

  /* report the change in cost, if any, to drs->solve_start_cost */
  if( KheDrsExprSumIntReportsCost(esi) && (old_dev != new_dev || DEBUG78) )
  {
    new_cost = f(esi, new_dev);
    old_cost = f(esi, old_dev);
    drs->solve_start_cost += new_cost - old_cost;
    if( RERUN_DEBUG(drs) && (new_cost - old_cost) != 0 )
      fprintf(stderr, "  KheDrsExprSumIntChildHasClosed: drs->solve_start_cost "
	"+= (%.5f - %.5f) = %.5f\n", KheCostShow(new_cost),
	KheCostShow(old_cost), KheCostShow(drs->solve_start_cost));
    KheDrsMonitorUpdateRerunCost(esi->monitor, (KHE_DRS_EXPR) esi, drs,
      NULL, KHE_DRS_CLOSE, "close", child_index, "+-", new_cost, old_cost);
  }
  /* ***
  if( esi->resource == NULL )
    KheDrsOpenChildrenDeleteLast(&esi->open_children_by_shift);
  *** */
  KheDrsExprSumIntCheck(esi, drs);
  if( RERUN_PRINT(drs, esi->monitor) )
    fprintf(stderr, "%d# ] KheDrsExprSumIntChildHasClosed\n",
      KheSolnDiversifier(drs->soln));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumIntSetClosedValue(KHE_DRS_EXPR_SUM_INT esi,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing esi, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumIntSetClosedValue(KHE_DRS_EXPR_SUM_INT esi,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  esi->value.i = KheDrsExprSumIntDev(esi, esi->history + esi->closed_state,
    esi->history_after);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAdjustedSigVal(int val, KHE_DRS_ADJUST_TYPE adjust_type,       */
/*    int min_limit, int max_limit, int pc, int qc)                          */
/*                                                                           */
/*  Return the adjusted signature value.                                     */
/*                                                                           */
/*****************************************************************************/

/* ***
static int KheDrsAdjustedSigVal(int val, KHE_DRS_ADJUST_TYPE adjust_type,
  int min_limit, int max_limit, int history_after)
{
  int lbar;
  switch( adjust_type )
  {
    case KHE_DRS_ADJUST_ORDINARY:

      return val;

    case KHE_DRS_ADJUST_NO_MAX:

      lbar = max(0, min_limit - history_after);
      return min(lbar, val);

    case KHE_DRS_ADJUST_LINEAR:

      lbar = max_limit;
      return min(lbar, val);

    case KHE_DRS_ADJUST_STEP:

      lbar = max_limit + 1;
      return min(lbar, val);

    default:

      HnAbort("KheDrsAdjustedSigVal internal error (adjusttype)");
      return val;  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSumIntInitialValue(KHE_DRS_EXPR_SUM_INT esi)               */
/*                                                                           */
/*  Return the initial value of esi.  This is used just below, but it is     */
/*  also used for dominance testing of correlated expressions.               */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprSumIntInitialValue(KHE_DRS_EXPR_SUM_INT esi)
{
  return esi->history + esi->closed_state;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumIntUpdateLU(KHE_DRS_EXPR child_e, int *ld2, int *ud2,  */
/*    int i, bool debug)                                                     */
/*                                                                           */
/*  Update *ld2 and *ud2 to take account of the fact that child_e now has    */
/*  a value.  Parameters i and debug are for debugging only.                 */
/*                                                                           */
/*  Implementation note.  See the comment to KheDrsExprCounterUpdateLU.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumIntUpdateLU(KHE_DRS_EXPR child_e, int *ld2, int *ud2,
  int i, bool debug)
{
  if( DEBUG68 && child_e == NULL )
    fprintf(stderr, "  KheDrsExprCounterUpdateLU failing\n");
  *ld2 += child_e->value.i;
  *ud2 += (child_e->value.i - child_e->value_ub.i);
  if( debug )
    fprintf(stderr, "%d:+%d-%d", i, child_e->value.i, child_e->value_ub.i);

  /* *** old version, but it's wrong here
  if( child_e->value.i == 1 )
  {
    *ld2 += 1;  ** unassigned to active **
    if( debug )
      fprintf(stderr, "%d+", i);
  }
  else
  {
    *ud2 -= 1;  ** unassigned to inactive **
    if( debug )
      fprintf(stderr, "%d-", i);
  }
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumIntEvalSignature(KHE_DRS_EXPR_SUM_INT esi,             */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Evaluate expression esi, assuming that next_di has just been assigned,   */
/*  and update the cost and states of sig with the result.                   */
/*                                                                           */
/*  Terminology used here        Terminology used in doc                     */
/*  --------------------------------------------------------                 */
/*  ld1                          l(m, S)                                     */
/*  ud1                          u(m, S)                                     */
/*  dev1                         delta(l(m, S), u(m, S)                      */
/*  cost1                        f(delta(l(m, S), u(m, S))                   */
/*  --------------------------------------------------------                 */
/*  ld2                          l(m, S')                                    */
/*  ud2                          u(m, S')                                    */
/*  dev2                         delta(l(m, S'), u(m, S')                    */
/*  cost2                        f(delta(l(m, S'), u(m, S'))                 */
/*  --------------------------------------------------------                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumIntEvalSignature(KHE_DRS_EXPR_SUM_INT esi,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, /* int next_di, */
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)
{
  int ld1, ld2, ud1, ud2, i, i1, i2, dev1, dev2, ub, next_di;
  KHE_DRS_EXPR child_e;  KHE_DRS_VALUE val;

  /* get ld1, ud1, and dev1 (for all days before next_di) */
  next_di = KheDrsSignerOpenDayIndex(dsg);
  if( debug )
  {
    fprintf(stderr, "  [ eval %s, next_di %d, open_children_by_day:\n",
      KheDrsMonitorId(esi->monitor), next_di);
    KheDrsOpenChildrenDebug(&esi->open_children_by_day, 1, 4, stderr);
  }
  if( KheDrsOpenChildrenIndexIsFirstOrLess(&esi->open_children_by_day,next_di))
  {
    ld1 = KheDrsExprSumIntInitialValue(esi);
    if( debug )
      fprintf(stderr, "    ld1 %d + %d", esi->history, esi->closed_state);
  }
  else
  {
    ld1 = KheDrsExprDaySigVal((KHE_DRS_EXPR) esi, next_di - 1, prev_sig).i;
    if( debug )
      fprintf(stderr, "    ld1 %d", ld1);
  }
  ub = KheDrsOpenChildrenUpperBoundInt(&esi->open_children_by_day, next_di);
  ud1 = ld1 + esi->history_after + ub;
  dev1 = KheDrsExprSumIntDelta(esi, ld1, ud1);
  if( debug )
    fprintf(stderr, " ud1 %d + %d + %d dev1 %d\n", ld1, esi->history_after,
      ub, dev1);

  /* get ld2, ud2, and dev2 (for one more day) */
  ld2 = ld1;
  ud2 = ud1;
  if( debug )
    fprintf(stderr, " next_di %d ", next_di);
  KheDrsOpenChildrenForEach(&esi->open_children_by_day, next_di, child_e, i)
    KheDrsExprSumIntUpdateLU(child_e, &ld2, &ud2, i, debug);
  dev2 = KheDrsExprSumIntDelta(esi, ld2, ud2);
  if( debug )
    fprintf(stderr, " ld2 %d ud2 %d dev2 %d", ld2, ud2, dev2);

  if( KheDrsOpenChildrenIndexIsLast(&esi->open_children_by_day, next_di) )
  {
    /* last day; set value */
    esi->value.i = dev2;
  }
  else
  {
    /* not last day; store ld2 in next_soln */
    /* *** no signature adjustment unless I think hard about it first
    val.i = KheDrsAdjustedSigVal(ld2, esi->adjust_type,
      esi->min_limit, esi->max_limit, esi->history_after);
    *** */
    val.i = ld2;
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) esi);
    if( debug )
      fprintf(stderr, " S%d", val.i);
  }

  /* if reporting a cost, do so */
  if( KheDrsExprSumIntReportsCost(esi) && dev2 != dev1 )
  {
    KheDrsSignatureAddCost(next_sig, f(esi, dev2) - f(esi, dev1));
    if( RERUN_DEBUG(drs) )
      fprintf(stderr, "  KheDrsExprSumIntEvalSignature: sig->cost += "
	"(%.5f - %.5f) = %.5f from %s\n", KheCostShow(f(esi, dev2)),
	KheCostShow(f(esi, dev1)), KheCostShow(next_sig->cost),
	KheDrsMonitorId(esi->monitor));
    KheDrsMonitorUpdateRerunCost(esi->monitor, (KHE_DRS_EXPR) esi, drs,
      dsg, KHE_DRS_SEARCH, "search", -1, "+-", f(esi, dev2), f(esi, dev1));
    if( debug )
      fprintf(stderr, " C%.5f", KheCostShow(f(esi, dev2) - f(esi, dev1)));
  }
  if( debug )
    fprintf(stderr, "\n  ]\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumIntDoDebug(KHE_DRS_EXPR_SUM_INT esi,                   */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of esi which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumIntDoDebug(KHE_DRS_EXPR_SUM_INT esi,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "COUNTER(%s, %s, %d)", KheDrsMonitorId(esi->monitor),
    KheDrsAdjustTypeShow(esi->adjust_type), esi->closed_state);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) esi, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_SUM_FLOAT - dom tests and notifying signers"     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprSumFloatChildDomTest(                         */
/*    KHE_DRS_EXPR_SUM_FLOAT esf, int open_day_index,                        */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return the dom test for a child of esf on its open_day_index'th          */
/*  open day.                                                                */
/*                                                                           */
/*  Note.  If esf is derived from a limit workload constraint, this          */
/*  code will not attempt to access any tables.                              */
/*                                                                           */
/*****************************************************************************/

/* *** needs to be replaced by KheDrsExprDomTest
static KHE_DRS_DOM_TEST KheDrsExprSumFloatChildDomTest(
  KHE_DRS_EXPR_SUM_FLOAT esf, int open_day_index,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b, unassigned_child_count;  KHE_DRS_DIM4_TABLE corr_dom_table4;
  KHE_DRS_CONSTRAINT dc;

  ** calculate a **
  a = (esf->max_limit < INT_MAX ? -1 : INT_MAX);

  ** calculate b **
  b = (esf->min_limit > 0 ? INT_MAX : 0);

  ** make and return the dom test **
  if( child_e->tag == KHE_DRS_EXPR_OR_TAG )
  {
    unassigned_child_count = KheDrsOpenChildrenAtOrAfter(
      &esf->open_children_by_day, open_day_index + 1);
    dc = KheDrsExprCostConstraint((KHE_DRS_EXPR_COST) esf, drs);
    if( DEBUG57 )
    {
      fprintf(stderr, "[ calling KheDrsDim5TableGet(%p, %d), open_day_index "
	"%d, esf:\n", (void *) dc->counter_corr_dom_table5,
	unassigned_child_count, open_day_index);
      KheDrsExprDebug((KHE_DRS_EXPR) esf, drs, 2, 2, stderr);
      KheDrsOpenChildrenDebug(&esf->open_children_by_day, 2, 2, stderr);
      fprintf(stderr, "]\n");
    }
    corr_dom_table4 = KheDrsDim5TableGet(dc->counter_corr_dom_table5,
      unassigned_child_count);
    ** ***
    corr_dom_table4 = KheDrsDim5TableGet(esf->corr_dom_table5,
      unassigned_child_count);
    *** **
  }
  else
    corr_dom_table4 = NULL;
  return KheDrsDomTestMakeInt(KHE_DRS_DOM_TEST_SEPARATE, child_e, false, 0,
    INT_MAX, a, b, 0, NULL, corr_dom_table4, NULL, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprCounterExprRequiresShiftDomTest(                          */
/*    KHE_DRS_EXPR_SUM_FLOAT esf, int si,                                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int *val)                             */
/*                                                                           */
/*  Find the number of unassigned children of esf there will be when the     */
/*  days before ds (the shift with open shift index si) begins are all       */
/*  assigned, and ds itself is assigned.  Return this number in *val, and    */
/*  return true if *val > 0.  This will mean that shift assignments for ds   */
/*  need a dom test, because assigning ds does not completely assign esf.    */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheDrsExprCounterExprRequiresShiftDomTest(
  KHE_DRS_EXPR_SUM_FLOAT esf, int si,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int *val)
{
  int di, assigned;  KHE_DRS_SHIFT ds;
  ds = HaArray(drs->open_shifts, si);
  di = ds->encl_day->open_day_index;
  assigned = KheDrsOpenChildrenBefore(&esf->open_children_by_day, di) +
    KheDrsOpenChildrenWithIndex(&esf->open_children_by_shift, si);
  *val = KheDrsOpenCh ildrenCount(&esf->open_children_by_day) - assigned;
  return *val > 0;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE KheDomTestTypeActual(KHE_DRS_DOM_TEST_TYPE dt,     */
/*    bool tradeoff_allowed)                                                 */
/*                                                                           */
/*  Return the dom test type to actually use, given than the requested       */
/*  type is dt and tradoff_allowed says whether tradeoff dominance is        */
/*  allowed.                                                                 */
/*                                                                           */
/*****************************************************************************/

/* *** redefinition
static KHE_DRS_DOM_TEST_TYPE KheDomTestTypeActual(KHE_DRS_DOM_TEST_TYPE dt,
  bool tradeoff_allowed)
{combined_weight
  if( dt == KHE_DRS_DOM_TEST_TRADEOFF && !tradeoff_allowed )
    return KHE_DRS_DOM_TEST_SEPARATE;
  else
    return dt;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprSumFloatDoDomTest(KHE_DRS_EXPR_SUM_FLOAT esf, */
/*    int unassigned_child_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Make and return a dominance test for esf, assuming that there are        */
/*  unassigned_child_count unassigned children (not including history).      */
/*                                                                           */
/*****************************************************************************/

/* *** never correct, and now unwanted
static KHE_DRS_DOM_TEST KheDrsExprSumFloatDoDomTest(KHE_DRS_EXPR_SUM_FLOAT esf,
  int unassigned_child_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b;  KHE_DRS_DOM_TEST_TYPE dt_type;  KHE_DRS_CONSTRAINT dc;
  KHE_DRS_DIM2_TABLE main_dom_table2;

  ** calculate a **
  if( esf->max_limit < FLT_MAX )
    a = (int) (esf->max_limit - unassigned_child_count);
  else
    a = INT_MAX;

  ** calculate b **
  if( esf->min_limit > 0.0 )
    b = (int) esf->min_limit;
  else
    b = 0;

  ** make and return the dom test **
  dc = KheDrsExprCostConstraint((KHE_DRS_EXPR_COST) esf, drs);
  main_dom_table2 = KheDrsDim3TableGet(dc->counter_main_dom_table3,
    unassigned_child_count);
  dt_type = KheDomTestTypeActual(drs->solve_dom_test_type,
    false, ** esf->cost_fn != KHE_QUADRATIC_COST_FUNCTION, ** true);
  return KheDrsDomTestMakeInt(dt_type, (KHE_DRS_EXPR) esf, esf->allow_zero,
    ** (int) esf->min_limit, (int) esf->max_limit, ** a, b,
    esf->combined_weight, main_dom_table2, NULL, esf->monitor->monitor, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprSumFloatDomTest(KHE_DRS_EXPR_SUM_FLOAT esf,   */
/*    int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return the dom test for esf on the open day with index open_day_index.   */
/*                                                                           */
/*  When two solutions are compared using this dom test, the open children   */
/*  up to and including those on the day with index open_day_index will be   */
/*  be assigned, leaving                                                     */
/*                                                                           */
/*    KheDrsOpenChildrenAtOrAfter(esf, open_day_index + 1)                   */
/*                                                                           */
/*  unassigned children.                                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprSumFloatDomTest(KHE_DRS_EXPR_SUM_FLOAT esf,
  int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  float a, b, ub;
  if( esf->combined_weight > 0 )
  {
    /* case (3), sum monitor */
    ub = KheDrsOpenChildrenUpperBoundFloat(&esf->open_children_by_day,
      open_day_index + 1);
    a = esf->max_limit - ub;
    b = esf->min_limit;
    return KheDrsDomTestMakeFloat(KHE_DRS_DOM_TEST_SEPARATE_FLOAT,
      (KHE_DRS_EXPR) esf, esf->allow_zero, a, b, esf->combined_weight,
      NULL, NULL, esf->monitor, drs);
  }
  else
  {
    /* case (4), child of a sum monitor */
    ub = KheDrsOpenChildrenUpperBoundFloat(&esf->open_children_by_day,
      open_day_index + 1);
    a = esf->max_limit - ub;
    b = esf->min_limit;
    return KheDrsDomTestMakeFloat(KHE_DRS_DOM_TEST_SEPARATE_FLOAT,
      (KHE_DRS_EXPR) esf, esf->allow_zero, a, b, 0, NULL, NULL, NULL, drs);
  }
  /* ***
  int unassigned_child_count;
  unassigned_child_count = KheDrsOpenChildrenAtOrAfter(
    &esf->open_children_by_day, open_day_index + 1);
  return KheDrsExprSumFloatDoDomTest(esf, unassigned_child_count, drs);
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_SUM_FLOAT - general"                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  float KheDrsExprSumFloatChildrenAtOrAfterUpperBound(                     */
/*    KHE_DRS_EXPR_SUM_FLOAT esf, int day_index)                             */
/*                                                                           */
/*  Return an upper bound on the values of the children at or after day      */
/*  day_index.                                                               */
/*                                                                           */
/*****************************************************************************/

/* ***
static float KheDrsExprSumFloatChildrenAtOrAfterUpperBound(
  KHE_DRS_EXPR_SUM_FLOAT esf, int day_index)
{
  return HaArrayLast(esf->value_upper_bounds);
  ** *** precomputing all this now
  if( day_index == 0 )
    return HaArrayLast(esf->value_upper_bounds).f -
      HaArray(esf->value_upper_bounds, day_index - 1).f;
  else
    return HaArrayLast(esf->value_upper_bounds).f -
      HaArray(esf->value_upper_bounds, day_index - 1).f;
  *** **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumFloatCheck(KHE_DRS_EXPR_SUM_FLOAT esf,                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  A simple consistency check for esf, needed to track down a nasty bug.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumFloatCheck(KHE_DRS_EXPR_SUM_FLOAT esf,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* *** not sensible for EXPR_SUM as it stands
  if( DEBUG79 && esf->tag == KHE_DRS_EXPR_SUM_FLOAT &&
      esf->closed_state.i > HaArrayCount(esf->children) )
  {
    fprintf(stderr, "%d# KheDrsExprSumFloatCheck closed_state (%d) > children"
      " (%d)\n", KheSolnDiversifier(drs->soln),
      esf->closed_state.i, HaArrayCount(esf->children));
    KheDrsExprDebug((KHE_DRS_EXPR) esf, drs, 2, 2, stderr);
    HnAbort("KheDrsExprSumFloatCheck internal error");
  }
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumFloatCheckClosed(KHE_DRS_EXPR_SUM_FLOAT esf,           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  A more elaborate consistency check which assumes that esf and its        */
/*  descendants are all closed, and checks that esf->closed_state is         */
/*  correct.                                                                 */
/*                                                                           */
/*****************************************************************************/

/* currently unused
static void KheDrsExprSumFloatCheckClosed(KHE_DRS_EXPR_SUM_FLOAT esf,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int closed_state, i;  KHE_DRS_EXPR child_e;
  if( DEBUG79 )
  {
    closed_state = 0;
    HaArrayForEach(esf->children, child_e, i)
      if( !child_e->open )
	closed_state += child_e->value.i;
    if( closed_state != esf->closed_state.i )
    {
      fprintf(stderr, "%d# KheDrsExprSumFloatCheckClosed: actual closed "
	"state (%d) != stored value (%d)\n", KheSolnDiversifier(drs->soln),
	closed_state, esf->closed_state.i);
      KheDrsExprDebug((KHE_DRS_EXPR) es, drs, 2, 2, stderr);
      HnAbort("KheDrsExprSumFloatCheckClosed internal error");
    }
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSumFloatDelta(KHE_DRS_EXPR_SUM_FLOAT esf,                  */
/*    float lower_det, float upper_det)                                      */
/*                                                                           */
/*  Return the deviation of esf for the given lower and upper determinants.  */
/*  This is function delta(l, u) from the documentation.                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprSumFloatDelta(KHE_DRS_EXPR_SUM_FLOAT esf,
  float lower_det, float upper_det)
{
  if( esf->allow_zero && lower_det == 0.0 )
    return 0;
  else if( lower_det > esf->max_limit + 0.001 )
    return (int) ceil(lower_det - esf->max_limit - 0.001);
  else if( upper_det < esf->min_limit - 0.001 )
    return (int) ceil(esf->min_limit - 0.001 - upper_det);
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSumFloatDev(KHE_DRS_EXPR_SUM_FLOAT esf,                    */
/*    float lower_det, float upper_det_minus_lower_det)                      */
/*                                                                           */
/*  Return the deviation of esf, passing lower_det and upper_det slightly    */
/*  differently.  This is similar to delta but more convenient at times.     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprSumFloatDev(KHE_DRS_EXPR_SUM_FLOAT esf,
  float lower_det, float upper_det_minus_lower_det)
{
  return KheDrsExprSumFloatDelta(esf, lower_det,
    lower_det + upper_det_minus_lower_det);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ADJUST_TYPE KheDrsAdjustType(KHE_COST_FUNCTION cost_fn,          */
/*    int max_limit)                                                         */
/*                                                                           */
/*  Return the adjust type, determined by the cost function and max limit.   */
/*                                                                           */
/*****************************************************************************/

/* *** can reuse earlier version
static KHE_DRS_ADJUST_TYPE KheDrsAdjustType(KHE_COST_FUNCTION cost_fn,
  int max_limit)
{
  if( max_limit == INT_MAX )
    return KHE_DRS_ADJUST_NO_MAX;
  else if( cost_fn == KHE_LINEAR_COST_FUNCTION )
    return KHE_DRS_ADJUST_LINEAR;
  else if( cost_fn == KHE_STEP_COST_FUNCTION )
    return KHE_DRS_ADJUST_STEP;
  else
    return KHE_DRS_ADJUST_ORDINARY;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsAdjustTypeShow(KHE_DRS_ADJUST_TYPE dct)                      */
/*                                                                           */
/*  Return a string representation of dct.                                   */
/*                                                                           */
/*****************************************************************************/

/* *** can reuse earlier version
static char *KheDrsAdjustTypeShow(KHE_DRS_ADJUST_TYPE dct)
{
  switch( dct )
  {
    case KHE_DRS_ADJUST_NO_MAX:	  return "NO_MAX";
    case KHE_DRS_ADJUST_LINEAR:	  return "LINEAR";
    case KHE_DRS_ADJUST_STEP:	  return "STEP";
    case KHE_DRS_ADJUST_ORDINARY: return "ORDINARY";
    default:			  return "??";
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_SUM_FLOAT KheDrsExprSumFloatMakeBegin(KHE_DRS_MONITOR dm,   */
/*    KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight, float min_limit,  */
/*    float max_limit, bool allow_zero, float history_before,                */
/*    float history_after, float history, KHE_DRS_RESOURCE dr,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_SUM_FLOAT object with these attributes.          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_SUM_FLOAT KheDrsExprSumFloatMakeBegin(KHE_DRS_MONITOR dm,
  KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight, float min_limit,
  float max_limit, bool allow_zero, float history_before,
  float history_after, float history, KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_SUM_FLOAT res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_SUM_FLOAT_TAG, dr, drs);
  res->cost_fn = cost_fn;
  res->combined_weight = combined_weight;
  res->monitor = dm;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->history_before = history_before;
  res->history_after = history_after;
  res->history = history;
  res->adjust_type = KheDrsAdjustType(cost_fn, max_limit);
  res->closed_state = 0.0;
  res->allow_zero = allow_zero;
  KheDrsAddWeight(drs, combined_weight);
  KheDrsExprSumFloatCheck(res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_SUM_FLOAT KheDrsExprSumFloatMakeRootBegin(                  */
/*    KHE_DRS_MONITOR dm, KHE_COST_FUNCTION cost_fn,                         */
/*    KHE_COST combined_weight, float min_limit, float max_limit,            */
/*    bool allow_zero, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs) */
/*                                                                           */
/*  Make a root sum float expression.                                        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_SUM_FLOAT KheDrsExprSumFloatMakeRootBegin(
  KHE_DRS_MONITOR dm, KHE_COST_FUNCTION cost_fn,
  KHE_COST combined_weight, float min_limit, float max_limit,
  bool allow_zero, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsExprSumFloatMakeBegin(dm, cost_fn, combined_weight, min_limit,
    max_limit, allow_zero, 0.0, 0.0, 0.0, dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_SUM_FLOAT KheDrsExprSumFloatMakeNonRootBegin(               */
/*    float min_limit, float max_limit, bool allow_zero,                     */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Make a non-root sum float expression.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_SUM_FLOAT KheDrsExprSumFloatMakeNonRootBegin(
  float min_limit, float max_limit, bool allow_zero,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsExprSumFloatMakeBegin(NULL, KHE_LINEAR_COST_FUNCTION, 0,
    min_limit, max_limit, allow_zero, 0.0, 0.0, 0.0, dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumFloatAddChild(KHE_DRS_EXPR_SUM_FLOAT esf,              */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to esf.  Update esf's state        */
/*  appropriately.  This is similar to closing a child, but not identical.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumFloatAddChild(KHE_DRS_EXPR_SUM_FLOAT esf,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsExprSumFloatCheck(esf, drs);
  if( RERUN_PRINT(drs, esf->monitor) )
    fprintf(stderr, "%d# AddChild: closed_state (init %.1f) += %.1f\n",
      KheSolnDiversifier(drs->soln), esf->closed_state, child_e->value.f);
  esf->closed_state += child_e->value.f;
  KheDrsExprSumFloatCheck(esf, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumFloatAdjustLimits(KHE_DRS_EXPR_SUM_FLOAT esf,          */
/*    float ub)                                                              */
/*                                                                           */
/*  Adjust the limits of esf.  Here ub is an upper bound on the sum of       */
/*  the values of the children, including history.                           */
/*                                                                           */
/*****************************************************************************/

/* *** I've decided that this is not worth doing
static void KheDrsExprSumFloatAdjustLimits(KHE_DRS_EXPR_SUM_FLOAT esf,
  float ub)
{
  ** make sure limits are consistent **
  HnAssert(esf->min_limit <= esf->max_limit,
    "KheDrsExprSumFloatAdjustLimits: before adjustment, inconsistent limits"
    " (min %.1f, max %.1f) for monitor %s", esf->min_limit, esf->max_limit,
    esf->monitor != NULL ? KheDrsMonitorId(esf->monitor) : "-");

  ** adjust esf->min_limit if possible **
  if( esf->min_limit < esf->history && esf->history <= esf->max_limit )
    esf->min_limit = esf->history;

  ** adjust esf->max_limit if possible **
  if( esf->max_limit > ub && ub >= esf->min_limit )
    esf->max_limit = ub;

  ** make sure limits are consistent **
  HnAssert(esf->min_limit <= esf->max_limit,
    "KheDrsExprSumFloatAdjustLimits: after adjustment, inconsistent limits"
    " (min %.1f, max %.1f) for monitor %s", esf->min_limit, esf->max_limit,
    esf->monitor != NULL ? KheDrsMonitorId(esf->monitor) : "-");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  float KheDrsExprSumFloatValueUpperBound(KHE_DRS_EXPR_SUM_FLOAT esf,      */
/*    float ub)                                                              */
/*                                                                           */
/*  Return a suitable value upper bound for esf.  Here ub is an upper bound  */
/*  on the sum of the values of the children, including history.             */
/*                                                                           */
/*****************************************************************************/

static float KheDrsExprSumFloatValueUpperBound(KHE_DRS_EXPR_SUM_FLOAT esf,
  float ub)
{
  float res;
  /* *** this exposes the weakness of allowing allow_zero at all
  res = esf->allow_zero ? max(esf->min_limit - 1.0, 0.0) : esf->min_limit;
  *** */
  res = esf->min_limit;
  if( res < ub - esf->max_limit )
    res = ub - esf->max_limit;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumFloatMakeEnd(KHE_DRS_EXPR_SUM_FLOAT esf,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  End the initialization of esf.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumFloatMakeEnd(KHE_DRS_EXPR_SUM_FLOAT esf,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  float ub;  int i;  KHE_DRS_EXPR child_e;

  KheDrsExprInitEnd((KHE_DRS_EXPR) esf, drs);

  /* get the total value upper bound */
  ub = esf->history + esf->history_after;
  HaArrayForEach(esf->children, child_e, i)
    ub += child_e->value_ub.f;

  /* adjust limits */
  /* KheDrsExprSumFloatAdjustLimits(esf, ub); */

  /* set the value upper bound */
  esf->value_ub.f = KheDrsExprSumFloatValueUpperBound(esf, ub);

  /* set constraint - NO! */
  /* ***
  if( esf->combined_weight > 0 )
    KheDrsExprCostSetConstraint((KHE_DRS_EXPR_COST) esf, 0, drs);
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprSumFloatReportsCost(KHE_DRS_EXPR_SUM_FLOAT esf)           */
/*                                                                           */
/*  Return true if esf reports a cost.                                       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprSumFloatReportsCost(KHE_DRS_EXPR_SUM_FLOAT esf)
{
  return esf->combined_weight > 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumFloatChildHasOpened(KHE_DRS_EXPR_SUM_FLOAT esf,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening esf for solving, esf's child_index'th child,          */
/*  child_e, has been opened.  Update esf to reflect this, and also          */
/*  drs->solve_start_cost if required.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumFloatChildHasOpened(KHE_DRS_EXPR_SUM_FLOAT esf,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int old_dev, new_dev;  float ub;  KHE_COST old_cost, new_cost;
  if( RERUN_PRINT(drs, esf->monitor) )
  {
    fprintf(stderr, "%d# [ KheDrsExprSumFloatChildHasOpened(%s, "
      "child_e(ub %.1f), %d)m child_e:\n", KheSolnDiversifier(drs->soln),
      KheDrsMonitorId(esf->monitor), child_e->value_ub.f, child_index);
    KheDrsExprDebug(child_e, drs, 2, 2, stderr);
    /* KheDrsExprDebug((KHE_DRS_EXPR) esf, drs, 2, 2, stderr); */
  }
  KheDrsExprSumFloatCheck(esf, drs);

  /* find the deviation before the change */
  ub = KheDrsOpenChildrenUpperBoundFloatAll(&esf->open_children_by_day);
  old_dev = KheDrsExprSumFloatDev(esf, esf->history + esf->closed_state,
    ub + esf->history_after);
  if( RERUN_PRINT(drs, esf->monitor) )
  {
    fprintf(stderr, "%d#   old_dev = %d = Dev(esf, %.1f + %.1f, %.1f + %.1f)\n",
    KheSolnDiversifier(drs->soln), old_dev, esf->history, esf->closed_state,
      ub, esf->history_after);
    fprintf(stderr, "%d#   closed_state (init %.1f) -= %.1f\n",
      KheSolnDiversifier(drs->soln), esf->closed_state, child_e->value.f);
  }

  /* update esf to reflect the new open child */
  KheDrsOpenChildrenAddChild(&esf->open_children_by_day, child_e);
  esf->closed_state -= child_e->value.f;

  /* find the deviation after the change */
  ub = KheDrsOpenChildrenUpperBoundFloatAll(&esf->open_children_by_day);
  new_dev = KheDrsExprSumFloatDev(esf, esf->history + esf->closed_state,
    ub + esf->history_after);
  if( RERUN_PRINT(drs, esf->monitor) )
    fprintf(stderr, "%d#   new_dev = %d = Dev(esf, %.1f + %.1f, %.1f + %.1f)\n",
    KheSolnDiversifier(drs->soln), new_dev, esf->history, esf->closed_state,
    ub, esf->history_after);

  /* report the change in cost, if any, to drs->solve_start_cost */
  if( KheDrsExprSumFloatReportsCost(esf) && old_dev != new_dev )
  {
    old_cost = f(esf, old_dev);
    new_cost = f(esf, new_dev);
    drs->solve_start_cost += (new_cost - old_cost);
    if( RERUN_DEBUG(drs) && (new_cost - old_cost) != 0 )
      fprintf(stderr,"  KheDrsExprSumFloatChildHasOpened: drs->solve_start_cost"
	" += (%.5f - %.5f) = %.5f\n", KheCostShow(new_cost),
	KheCostShow(old_cost), KheCostShow(drs->solve_start_cost));
    KheDrsMonitorUpdateRerunCost(esf->monitor, (KHE_DRS_EXPR) esf, drs,
      NULL, KHE_DRS_OPEN, "open", child_index, "+-", new_cost, old_cost);
  }

  /* all done */
  KheDrsExprSumFloatCheck(esf, drs);
  if( RERUN_PRINT(drs, esf->monitor) )
    fprintf(stderr, "%d# ] KheDrsExprSumFloatChildHasOpened\n",
      KheSolnDiversifier(drs->soln));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumFloatChildHasClosed(KHE_DRS_EXPR_SUM_FLOAT esf,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing esf, esf's child_index'th child, child_e, has         */
/*  been closed.  Update esf's closed state to reflect this, and also        */
/*  drs->solve_start_cost if required.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumFloatChildHasClosed(KHE_DRS_EXPR_SUM_FLOAT esf,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int old_dev, new_dev;  float ub;  KHE_COST old_cost, new_cost;
  if( RERUN_PRINT(drs, esf->monitor) )
  {
    fprintf(stderr, "%d# [ KheDrsExprSumFloatChildHasClosed(%s, child_e, "
      "%d):\n", KheSolnDiversifier(drs->soln),
      KheDrsMonitorId(esf->monitor), child_index);
    /* KheDrsExprDebug((KHE_DRS_EXPR) esf, drs, 2, 2, stderr); */
  }
  KheDrsExprSumFloatCheck(esf, drs);

  /* find the deviation before the change */
  ub = KheDrsOpenChildrenUpperBoundFloatAll(&esf->open_children_by_day);
  old_dev = KheDrsExprSumFloatDev(esf, esf->history + esf->closed_state,
    ub + esf->history_after);
  if( RERUN_PRINT(drs, esf->monitor) )
  {
    fprintf(stderr, "%d#   old_dev = %d = Dev(esf, %.1f + %.1f, %.1f + %.1f)\n",
    KheSolnDiversifier(drs->soln), old_dev, esf->history, esf->closed_state,
      ub, esf->history_after);
    fprintf(stderr, "%d#   closed_state (init %.1f) += %.1f\n",
      KheSolnDiversifier(drs->soln), esf->closed_state, child_e->value.f);
  }

  /* update esf to reflect one less open child */
  KheDrsOpenChildrenDeleteChild(&esf->open_children_by_day, child_e);
  esf->closed_state += child_e->value.f;

  /* find the deviation after the change */
  ub = KheDrsOpenChildrenUpperBoundFloatAll(&esf->open_children_by_day);
  new_dev = KheDrsExprSumFloatDev(esf, esf->history + esf->closed_state,
    ub + esf->history_after);
  if( RERUN_PRINT(drs, esf->monitor) )
    fprintf(stderr, "%d#   new_dev = %d = Dev(esf, %.1f + %.1f, %.1f + %.1f)\n",
    KheSolnDiversifier(drs->soln), new_dev, esf->history, esf->closed_state,
    ub, esf->history_after);

  /* report the change in cost, if any, to drs->solve_start_cost */
  if( KheDrsExprSumFloatReportsCost(esf) && old_dev != new_dev )
  {
    new_cost = f(esf, new_dev);
    old_cost = f(esf, old_dev);
    drs->solve_start_cost += (new_cost - old_cost);
    if( RERUN_DEBUG(drs) && (new_cost - old_cost) != 0 )
      fprintf(stderr,"  KheDrsExprSumFloatChildHasClosed: drs->solve_start_cost"
	" += (%.5f - %.5f) = %.5f\n", KheCostShow(new_cost),
	KheCostShow(old_cost), KheCostShow(drs->solve_start_cost));
    KheDrsMonitorUpdateRerunCost(esf->monitor, (KHE_DRS_EXPR) esf, drs,
      NULL, KHE_DRS_CLOSE, "close", child_index, "+-", new_cost, old_cost);
  }
  /* ***
  if( esf->resource == NULL )
    KheDrsOpenChildrenDeleteLast(&esf->open_children_by_shift);
  *** */
  KheDrsExprSumFloatCheck(esf, drs);
  if( RERUN_PRINT(drs, esf->monitor) )
    fprintf(stderr, "%d# ] KheDrsExprSumFloatChildHasClosed\n",
      KheSolnDiversifier(drs->soln));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumFloatSetClosedValue(KHE_DRS_EXPR_SUM_FLOAT esf,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing esf, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumFloatSetClosedValue(KHE_DRS_EXPR_SUM_FLOAT esf,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  esf->value.f = KheDrsExprSumFloatDev(esf, esf->history + esf->closed_state,
    esf->history_after);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAdjustedSigVal(int val, KHE_DRS_ADJUST_TYPE adjust_type,       */
/*    int min_limit, int max_limit, int pc, int qc)                          */
/*                                                                           */
/*  Return the adjusted signature value.                                     */
/*                                                                           */
/*****************************************************************************/

/* *** prev version will do for now
static int KheDrsAdjustedSigVal(int val, KHE_DRS_ADJUST_TYPE adjust_type,
  int min_limit, int max_limit, int history_after)
{
  int lbar;
  switch( adjust_type )
  {
    case KHE_DRS_ADJUST_ORDINARY:

      return val;

    case KHE_DRS_ADJUST_NO_MAX:

      lbar = max(0, min_limit - history_after);
      return min(lbar, val);

    case KHE_DRS_ADJUST_LINEAR:

      lbar = max_limit;
      return min(lbar, val);

    case KHE_DRS_ADJUST_STEP:

      lbar = max_limit + 1;
      return min(lbar, val);

    default:

      HnAbort("KheDrsAdjustedSigVal internal error (adjusttype)");
      return val;  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  float KheDrsExprSumFloatInitialValue(KHE_DRS_EXPR_SUM_FLOAT esf)         */
/*                                                                           */
/*  Return the initial value of esf.  This is used just below, but it is     */
/*  also used for dominance testing of correlated expressions.               */
/*                                                                           */
/*****************************************************************************/

static float KheDrsExprSumFloatInitialValue(KHE_DRS_EXPR_SUM_FLOAT esf)
{
  return esf->history + esf->closed_state;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumFloatUpdateLU(KHE_DRS_EXPR child_e, int *ld2,          */
/*    int *ud2, int i, bool debug)                                           */
/*                                                                           */
/*  Update *ld2 and *ud2 to take account of the fact that child_e now has    */
/*  a value.  Parameters i and debug are for debugging only.                 */
/*                                                                           */
/*  Implementation note.  See the comment to KheDrsExprCounterUpdateLU.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumFloatUpdateLU(KHE_DRS_EXPR child_e, float *ld2,
  float *ud2, int i, bool debug)
{
  if( DEBUG68 && child_e == NULL )
    fprintf(stderr, "  KheDrsExprSumFloatUpdateLU failing\n");
  *ld2 += child_e->value.f;
  *ud2 += (child_e->value.f - child_e->value_ub.f);
  if( debug )
    fprintf(stderr, "%d:+%.1f-%.1f", i, child_e->value.f, child_e->value_ub.f);

  /* *** old version, but it's wrong here
  if( child_e->value.f == 1 )
  {
    *ld2 += 1.0;  ** unassigned to active **
    if( debug )
      fprintf(stderr, "%d+", i);
  }
  else
  {
    *ud2 -= 1.0;  ** unassigned to inactive **
    if( debug )
      fprintf(stderr, "%d-", i);
  }
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumFloatEvalSignature(KHE_DRS_EXPR_SUM_FLOAT esf,         */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs,           */
/*    bool debug)                                                            */
/*                                                                           */
/*  Evaluate expression esf, assuming that next_di has just been assigned,   */
/*  and update the cost and states of sig with the result.                   */
/*                                                                           */
/*  Terminology used here        Terminology used in doc                     */
/*  --------------------------------------------------------                 */
/*  ld1                          l(m, S)                                     */
/*  ud1                          u(m, S)                                     */
/*  dev1                         delta(l(m, S), u(m, S)                      */
/*  cost1                        f(delta(l(m, S), u(m, S))                   */
/*  --------------------------------------------------------                 */
/*  ld2                          l(m, S')                                    */
/*  ud2                          u(m, S')                                    */
/*  dev2                         delta(l(m, S'), u(m, S')                    */
/*  cost2                        f(delta(l(m, S'), u(m, S'))                 */
/*  --------------------------------------------------------                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumFloatEvalSignature(KHE_DRS_EXPR_SUM_FLOAT esf,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, /* int next_di, */
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  bool debug)
{
  float ld1, ld2, ud1, ud2, ub;  KHE_DRS_EXPR child_e;  KHE_DRS_VALUE val;
  int i, i1, i2, dev1, dev2, next_di;

  /* get ld1, ud1, and dev1 (for all days before next_di) */
  next_di = KheDrsSignerOpenDayIndex(dsg);
  if( debug )
  {
    fprintf(stderr, "  [ eval %s, next_di %d, open_children_by_day:\n",
      KheDrsMonitorId(esf->monitor), next_di);
    KheDrsOpenChildrenDebug(&esf->open_children_by_day, 1, 4, stderr);
  }
  if( KheDrsOpenChildrenIndexIsFirstOrLess(&esf->open_children_by_day,next_di))
  {
    ld1 = KheDrsExprSumFloatInitialValue(esf);
    if( debug )
      fprintf(stderr, "    ld1 %.1f + %.1f", esf->history, esf->closed_state);
  }
  else
  {
    ld1 = KheDrsExprDaySigVal((KHE_DRS_EXPR) esf, next_di - 1, prev_sig).f;
    if( debug )
      fprintf(stderr, "    ld1 %.1f", ld1);
  }
  ub = KheDrsOpenChildrenUpperBoundFloat(&esf->open_children_by_day, next_di);
  ud1 = ld1 + esf->history_after + ub;
  dev1 = KheDrsExprSumFloatDelta(esf, ld1, ud1);
  if( debug )
    fprintf(stderr, " ud1 %.1f + %.1f + %.1f dev1 %d\n", ld1,
      esf->history_after, ub, dev1);

  /* get ld2, ud2, and dev2 for one more day, shift, or shift pair */
  ld2 = ld1;
  ud2 = ud1;
  if( debug )
    fprintf(stderr, " next_di %d ", next_di);
  KheDrsOpenChildrenForEach(&esf->open_children_by_day, next_di, child_e, i)
    KheDrsExprSumFloatUpdateLU(child_e, &ld2, &ud2, i, debug);
  dev2 = KheDrsExprSumFloatDelta(esf, ld2, ud2);
  if( debug )
    fprintf(stderr, " ld2 %.1f ud2 %.1f dev2 %d", ld2, ud2, dev2);

  if( KheDrsOpenChildrenIndexIsLast(&esf->open_children_by_day, next_di) )
  {
    /* last day; set value (NB saved value is an int) */
    esf->value.i = dev2;
  }
  else
  {
    /* not last day; store ld2 in next_soln (NB stored value is a float) */
    /* *** no signature adjustment unless I think about it first
    val.f = KheDrsAdjustedSigVal(ld2, esf->adjust_type,
      esf->min_limit, esf->max_limit, esf->history_after);
    *** */
    val.f = ld2;
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) esf);
    if( debug )
      fprintf(stderr, " S%.1f", val.f);
  }

  /* if reporting a cost, do so */
  if( KheDrsExprSumFloatReportsCost(esf) && dev2 != dev1 )
  {
    KheDrsSignatureAddCost(next_sig, f(esf, dev2) - f(esf, dev1));
    if( RERUN_DEBUG(drs) )
      fprintf(stderr, "  KheDrsExprSumFloatEvalSignature: sig->cost += "
	"(%.5f - %.5f) = %.5f from %s\n", KheCostShow(f(esf, dev2)),
	KheCostShow(f(esf, dev1)), KheCostShow(next_sig->cost),
	KheDrsMonitorId(esf->monitor));
    KheDrsMonitorUpdateRerunCost(esf->monitor, (KHE_DRS_EXPR) esf, drs,
      dsg, KHE_DRS_SEARCH, "search", -1, "+-", f(esf, dev2), f(esf, dev1));
    if( debug )
      fprintf(stderr, " C%.5f", KheCostShow(f(esf, dev2) - f(esf, dev1)));
  }
  if( debug )
    fprintf(stderr, "\n  ]\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSumFloatDoDebug(KHE_DRS_EXPR_SUM_FLOAT esf,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of esf which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSumFloatDoDebug(KHE_DRS_EXPR_SUM_FLOAT esf,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "SUM_FLOAT(%s, %s, %.1f)",
    KheDrsMonitorId(esf->monitor),
    KheDrsAdjustTypeShow(esf->adjust_type), esf->closed_state);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) esf, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_CLOSED_SEQ"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CLOSED_SEQ KheDrsClosedSeqMake(KHE_DYNAMIC_RESOURCE_SOLVER drs,  */
/*    int start_index, int stop_index)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_CLOSED_SEQ object with these attributes.              */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CLOSED_SEQ KheDrsClosedSeqMake(KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int start_index, int stop_index)
{
  KHE_DRS_CLOSED_SEQ res;
  if( HaArrayCount(drs->closed_seq_free_list) > 0 )
    res = HaArrayLastAndDelete(drs->closed_seq_free_list);
  else
    HaMake(res, drs->arena);
  res->start_index = start_index;
  res->stop_index = stop_index;
  res->active_at_left = 0;		/* placeholder (unless empty) */
  res->active_at_right = 0;		/* placeholder (unless empty) */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CLOSED_SEQ KheDrsClosedSeqMakeInit(                              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make an initial empty closed sequence.                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CLOSED_SEQ KheDrsClosedSeqMakeInit(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsClosedSeqMake(drs, 0, 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqDelete(KHE_DRS_CLOSED_SEQ dcs,                       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Delete dcs.                                                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqDelete(KHE_DRS_CLOSED_SEQ dcs,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->closed_seq_free_list, dcs);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsClosedSeqLength(KHE_DRS_CLOSED_SEQ dcs)                        */
/*                                                                           */
/*  Return the length of dcs.                                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsClosedSeqLength(KHE_DRS_CLOSED_SEQ dcs)
{
  return dcs->stop_index - dcs->start_index;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsClosedSeqAllActive(KHE_DRS_CLOSED_SEQ dcs)                    */
/*                                                                           */
/*  Return true if every child of dcs is active.                             */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsClosedSeqAllActive(KHE_DRS_CLOSED_SEQ dcs)
{
  return KheDrsClosedSeqLength(dcs) == dcs->active_at_left;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqExtendToRight(KHE_DRS_CLOSED_SEQ dcs, int val)       */
/*                                                                           */
/*  Extend dcs one place to the right, with a child with value val.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqExtendToRight(KHE_DRS_CLOSED_SEQ dcs, int val)
{
  if( val == 0 )
    dcs->active_at_right = 0;
  else
  {
    if( KheDrsClosedSeqAllActive(dcs) )
      dcs->active_at_left++;
    dcs->active_at_right++;
  }
  dcs->stop_index++;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqSetActiveLengths(KHE_DRS_CLOSED_SEQ dcs,             */
/*    KHE_DRS_EXPR_SEQUENCE es)                                              */
/*                                                                           */
/*  Assuming dcs->start_index and dcs->stop_index are up to date, bring      */
/*  dcs->active_at_left and dcs-> active_at_right up to date.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqSetActiveLengths(KHE_DRS_CLOSED_SEQ dcs,
  KHE_DRS_EXPR_SEQUENCE es)
{
  KHE_DRS_EXPR child_e;  int i;

  /* bring dcs->active_at_left up to date */
  dcs->active_at_left = 0;
  for( i = dcs->start_index;  i < dcs->stop_index;  i++ )
  {
    child_e = HaArray(es->children, i);
    if( child_e->value.i != 1 )
      break;
    dcs->active_at_left++;
  }

  /* bring dcs->active_at_right up to date */
  dcs->active_at_right = 0;
  for( i = dcs->stop_index - 1;  i >= dcs->start_index;  i-- )
  {
    child_e = HaArray(es->children, i);
    if( child_e->value.i != 1 )
      break;
    dcs->active_at_right++;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CLOSED_SEQ KheDrsClosedSeqSplit(KHE_DRS_CLOSED_SEQ dcs,          */
/*    int child_index)                                                       */
/*                                                                           */
/*  Split dcs at child_index.  The child at child_index will be in           */
/*  neither of the two fragments.                                            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CLOSED_SEQ KheDrsClosedSeqSplit(KHE_DRS_CLOSED_SEQ dcs,
  int child_index, KHE_DRS_EXPR_SEQUENCE es,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CLOSED_SEQ res;

  /* make the new fragment and reset the stop index of the old one */
  HnAssert(dcs->start_index <= child_index,
    "KheDrsClosedSeqSplit internal error 1");
  HnAssert(child_index < dcs->stop_index,
    "KheDrsClosedSeqSplit internal error 2 (child_index %d, dcs->stop_index %d",
    child_index, dcs->stop_index);
  res = KheDrsClosedSeqMake(drs, child_index + 1, dcs->stop_index);
  dcs->stop_index = child_index;

  /* update the active lengths of the two fragments, and return the new one */
  KheDrsClosedSeqSetActiveLengths(dcs, es);
  KheDrsClosedSeqSetActiveLengths(res, es);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqMerge(KHE_DRS_CLOSED_SEQ dcs1,                       */
/*    KHE_DRS_CLOSED_SEQ dcs2, KHE_DRS_EXPR_SEQUENCE es,                     */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Merge adjacent closed sequences dcs1 and dcs2.  The formerly open        */
/*  child separating them has closed, and its closed value has been set.     */
/*                                                                           */
/*  Place the merged closed sequence in dcs1, and delete dcs2.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqMerge(KHE_DRS_CLOSED_SEQ dcs1,
  KHE_DRS_CLOSED_SEQ dcs2, KHE_DRS_EXPR_SEQUENCE es,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* update dcs1 to hold the merged sequence */
  HnAssert(dcs1->stop_index + 1 == dcs2->start_index,
    "KheDrsClosedSeqMerge internal error 1");
  dcs1->stop_index = dcs2->stop_index;
  KheDrsClosedSeqSetActiveLengths(dcs1, es);

  /* delete dcs2 */
  KheDrsClosedSeqDelete(dcs2, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqDebug(KHE_DRS_CLOSED_SEQ dcs, int verbosity,         */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dcs onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqDebug(KHE_DRS_CLOSED_SEQ dcs, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "[CLOSED_SEQ %d-%d, active_at_left %d, active_at_right %d]",
    dcs->start_index, dcs->stop_index, dcs->active_at_left,
    dcs->active_at_right);
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_A_INTERVAL"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalMake(int start_index, int stop_index,  */
/*    bool unassigned_precedes)                                              */
/*                                                                           */
/*  Make and return a new a-interval with these attributes.                  */
/*                                                                           */
/*****************************************************************************/

KHE_DRS_A_INTERVAL KheDrsAIntervalMake(int start_index, int stop_index,
  bool unassigned_precedes)
{
  KHE_DRS_A_INTERVAL res;
  res.start_index = start_index;
  res.stop_index = stop_index;
  res.unassigned_precedes = unassigned_precedes;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalMakeLeft(                              */
/*    KHE_DRS_EXPR_SEQUENCE es, int index, int active_len)                   */
/*                                                                           */
/*  Make an a-interval of length active_len which ends just before the       */
/*  open child with whose index in es->open_children_by_day is index.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_A_INTERVAL KheDrsAIntervalMakeLeft(
  KHE_DRS_EXPR_SEQUENCE es, int index, int active_len)
{
  KHE_DRS_CLOSED_SEQ dcs;
  dcs = HaArray(es->closed_seqs, index);
  return KheDrsAIntervalMake(dcs->stop_index - active_len, dcs->stop_index,
    false);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalFindLeft(KHE_DRS_EXPR_SEQUENCE es,     */
/*    int index)                                                             */
/*                                                                           */
/*  Find the (possibly empty) a-interval immediately to the left of the      */
/*  child whose index in es->open_children_by_day is index.  Include         */
/*  history if necessary.                                                    */
/*                                                                           */
/*  Implementation note.  This interval consists entirely of the last        */
/*  active_at_right children of the preceding KHE_DRS_CLOSED_SEQ             */
/*  (possibly plus history), even if their number is zero; because they      */
/*  are preceded by another open child, or an inactive child, or nothing.    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_A_INTERVAL KheDrsAIntervalFindLeft(KHE_DRS_EXPR_SEQUENCE es,
  int index)
{
  KHE_DRS_CLOSED_SEQ dcs;
  dcs = HaArray(es->closed_seqs, index);
  if( !KheDrsClosedSeqAllActive(dcs) )
  {
    /* an inactive child precedes the active_at_right active children */
    return KheDrsAIntervalMake(dcs->stop_index - dcs->active_at_right,
      dcs->stop_index, false);
  }
  else if( index > 0 )
  {
    /* an unassigned child precedes the active_at_right active children */
    return KheDrsAIntervalMake(dcs->stop_index - dcs->active_at_right,
      dcs->stop_index, true);
  }
  else
  {
    /* nothing but history precedes the active_at_right active children */
    return KheDrsAIntervalMake(dcs->stop_index - dcs->active_at_right
      - es->history, dcs->stop_index, false);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalFindRight(                             */
/*    KHE_DRS_EXPR_SEQUENCE es, int index, bool child_unassigned)            */
/*                                                                           */
/*  Find the (possibly empty) a-interval immediately to the right of the     */
/*  child whose index in es->open_children_by_day is index.  That child      */
/*  is assumed to be unassigned when child_unassigned is true.               */
/*                                                                           */
/*  Implementation note.  This interval consists entirely of the first       */
/*  active_at_left children of the following KHE_DRS_CLOSED_SEQ,             */
/*  even if their number is zero; because they are followed by another       */
/*  open child, or an inactive child, or history_after, or nothing.          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_A_INTERVAL KheDrsAIntervalFindRight(
  KHE_DRS_EXPR_SEQUENCE es, int index, bool child_unassigned)
{
  KHE_DRS_CLOSED_SEQ dcs;
  dcs = HaArray(es->closed_seqs, index + 1);
  return KheDrsAIntervalMake(dcs->start_index,
    dcs->start_index + dcs->active_at_left, child_unassigned);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalMerge(KHE_DRS_A_INTERVAL ai_left,      */
/*    KHE_DRS_A_INTERVAL ai_right)                                           */
/*                                                                           */
/*  Merge ai_left with ai_right.  These intervals are separated by one       */
/*  child (this is checked).  That child is assumed to be active, but that   */
/*  is not checked, and indeed may not be true - we might be building this   */
/*  interval to reflect the old state of that child.                         */
/*                                                                           */
/*  It is safe to set the unassigned_precedes value of the result interval   */
/*  to ai_left.unassigned_precedes.  Even when ai_left is empty, its place   */
/*  is well defined, and ai_left.unassigned_precedes is correct for there.   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_A_INTERVAL KheDrsAIntervalMerge(KHE_DRS_A_INTERVAL ai_left,
  KHE_DRS_A_INTERVAL ai_right)
{
  HnAssert(ai_left.stop_index + 1 == ai_right.start_index,
    "KheDrsAIntervalMerge internal error");
  return KheDrsAIntervalMake(ai_left.start_index, ai_right.stop_index,
    ai_left.unassigned_precedes);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAIntervalDev(KHE_DRS_A_INTERVAL ai,                            */
/*    KHE_DRS_EXPR_SEQUENCE es)                                              */
/*                                                                           */
/*  Return the deviation of a-interval ai.  If ai.unassigned_precedes is     */
/*  true, there is an unassigned child immediately before ai.first_index,    */
/*  so return 0 if the cost function is KHE_STEP_COST_FUNCTION.              */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAIntervalDev(KHE_DRS_A_INTERVAL ai,
  KHE_DRS_EXPR_SEQUENCE es)
{
  int len;

  /* return 0 if an unassigned child immediately precedes this interval */
  /* and the cost function is step                                      */
  if( ai.unassigned_precedes && es->cost_fn == KHE_STEP_COST_FUNCTION )
    return 0;

  /* calculate the length and deviation */
  len = ai.stop_index - ai.start_index;
  return len > es->max_limit ? len - es->max_limit : 0;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsAIntervalCost(KHE_DRS_A_INTERVAL ai,                      */
/*    KHE_DRS_EXPR_SEQUENCE es)                                              */
/*                                                                           */
/*  Return the cost of ai.                                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsAIntervalCost(KHE_DRS_A_INTERVAL ai,
  KHE_DRS_EXPR_SEQUENCE es)
{
  int dev;
  dev = KheDrsAIntervalDev(ai, es);
  return dev == 0 ? 0 : f(es, dev);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_AU_INTERVAL"                                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalMake(int start_index,                */
/*    int stop_index, bool has_active_child)                                 */
/*                                                                           */
/*  Make and return an au-interval with these attributes.  However, the      */
/*  has_active_child attribute is ignored when the interval is empty:  an    */
/*  empty interval cannot contain an active child.                           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalMake(int start_index,
  int stop_index, bool has_active_child)
{
  KHE_DRS_AU_INTERVAL res;
  res.start_index = start_index;
  res.stop_index = stop_index;
  res.has_active_child = (start_index < stop_index && has_active_child);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAUIntervalLength(KHE_DRS_AU_INTERVAL aui)                      */
/*                                                                           */
/*  Return the length of aui.                                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAUIntervalLength(KHE_DRS_AU_INTERVAL aui)
{
  return aui.stop_index - aui.start_index;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAUIntervalExtendToLeft(KHE_DRS_AU_INTERVAL *aui,              */
/*    int extra_len, bool has_active_child)                                  */
/*                                                                           */
/*  Extend *aui extra_len places to the left.  If extra_len > 0,             */
/*  has_active_child says whether the extra elements are active or not.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAUIntervalExtendToLeft(KHE_DRS_AU_INTERVAL *aui,
  int extra_len, bool has_active_child)
{
  if( extra_len > 0 )
  {
    aui->start_index -= extra_len;
    if( has_active_child )
      aui->has_active_child = true;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAUIntervalExtendToRight(KHE_DRS_AU_INTERVAL *aui,             */
/*    int extra_len, bool has_active_child)                                  */
/*                                                                           */
/*  Extend *aui extra_len places to the right.  If extra_len > 0,            */
/*  has_active_child says whether the extra elements are active or not.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAUIntervalExtendToRight(KHE_DRS_AU_INTERVAL *aui,
  int extra_len, bool has_active_child)
{
  if( extra_len > 0 )
  {
    aui->stop_index += extra_len;
    if( has_active_child )
      aui->has_active_child = true;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalMakeLeft(                            */
/*    KHE_DRS_EXPR_SEQUENCE es, int index, int active_len)                   */
/*                                                                           */
/*  Make an au-interval of length active_len which ends just before          */
/*  the open child whose index in es->open_children_by_day is index.         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalMakeLeft(
  KHE_DRS_EXPR_SEQUENCE es, int index, int active_len)
{
  KHE_DRS_CLOSED_SEQ dcs;
  dcs = HaArray(es->closed_seqs, index);
  return KheDrsAUIntervalMake(dcs->stop_index - active_len, dcs->stop_index,
    true);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalFindLeft(                            */
/*    KHE_DRS_EXPR_SEQUENCE es, int index)                                   */
/*                                                                           */
/*  Find the au-interval ending just before the child whose index in         */
/*  es->open_children_by_day is index.                                       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalFindLeft(
  KHE_DRS_EXPR_SEQUENCE es, int index)
{
  KHE_DRS_CLOSED_SEQ dcs;  KHE_DRS_AU_INTERVAL res;  int i;

  /* initialize res to the interval of active children at the right of dcs */
  dcs = HaArray(es->closed_seqs, index);
  res = KheDrsAUIntervalMake(dcs->stop_index - dcs->active_at_right,
    dcs->stop_index, true);
  if( !KheDrsClosedSeqAllActive(dcs) )
    return res;

  /* now keep looking to the left of there */
  for( i = index - 1;  i >= 0;  i-- )
  {
    /* return early if es->min_limit reached */
    if( KheDrsAUIntervalLength(res) >= es->min_limit )
      return res;

    /* res includes the open unassigned child before the previous dcs */
    KheDrsAUIntervalExtendToLeft(&res, 1, false);

    /* res includes the active children at the right end of the next dcs */
    dcs = HaArray(es->closed_seqs, i);
    KheDrsAUIntervalExtendToLeft(&res, dcs->active_at_right, true);
    if( !KheDrsClosedSeqAllActive(dcs) )
      return res;
  }

  /* at the start, so res includes history */
  KheDrsAUIntervalExtendToLeft(&res, es->history, true);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalFindRight(KHE_DRS_EXPR_SEQUENCE es,  */
/*    int index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                            */
/*                                                                           */
/*  Find the au-interval ending just after the child whose index in          */
/*  es->open_children_by_day is index.                                       */
/*                                                                           */
/*  Implementation note.  Parameter drs is included only to support debug.   */
/*                                                                           */
/*****************************************************************************/
static char *DrsAUIntervalShow(KHE_DRS_AU_INTERVAL aui);

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalFindRight(KHE_DRS_EXPR_SEQUENCE es,
  int index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CLOSED_SEQ dcs;  KHE_DRS_AU_INTERVAL res;  int i;

  /* initialize res to the interval of active children at the left of dcs */
  if( DEBUG20 && RERUN_PRINT(drs, es->monitor) )
  {
    fprintf(stderr, "[ KheDrsAUIntervalFindRight(%s, %d, drs)\n",
      KheDrsMonitorId(es->monitor), index);
    for( i = index + 1;  i < HaArrayCount(es->closed_seqs);  i++ )
    {
      dcs = HaArray(es->closed_seqs, i);
      fprintf(stderr, "  dcs %d ", i);
      KheDrsClosedSeqDebug(dcs, 2, 0, stderr);
    }
  }
  dcs = HaArray(es->closed_seqs, index + 1);
  res = KheDrsAUIntervalMake(dcs->start_index,
    dcs->start_index + dcs->active_at_left, true);
  if( !KheDrsClosedSeqAllActive(dcs) )
  {
    if( DEBUG20 && RERUN_PRINT(drs, es->monitor) )
      fprintf(stderr, "] KheDrsAUIntervalFindRight(es, %d) 1= %s\n",
	index, DrsAUIntervalShow(res));
    return res;
  }

  /* now keep looking to the right of there */
  for( i = index + 2;  i < HaArrayCount(es->closed_seqs);  i++ )
  {
    /* return early if es->min_limit reached */
    if( KheDrsAUIntervalLength(res) >= es->min_limit )
    {
      if( DEBUG20 && RERUN_PRINT(drs, es->monitor) )
	fprintf(stderr, "] KheDrsAUIntervalFindRight(es, %d) 2= %s\n",
	  index, DrsAUIntervalShow(res));
      return res;
    }

    /* res includes the open unassigned child after the previous dcs */
    KheDrsAUIntervalExtendToRight(&res, 1, false);

    /* res includes the active children at the left end of the next dcs */
    dcs = HaArray(es->closed_seqs, i);
    KheDrsAUIntervalExtendToRight(&res, dcs->active_at_left, true);
    if( !KheDrsClosedSeqAllActive(dcs) )
    {
      if( DEBUG20 && RERUN_PRINT(drs, es->monitor) )
	fprintf(stderr, "] KheDrsAUIntervalFindRight(es, %d) 3= %s\n",
	  index, DrsAUIntervalShow(res));
      return res;
    }
  }

  /* at the end, so res includes history_after */
  KheDrsAUIntervalExtendToRight(&res, es->history_after, false);
  if( DEBUG20 && RERUN_PRINT(drs, es->monitor) )
    fprintf(stderr, "] KheDrsAUIntervalFindRight(es, %d) 4= %s\n",
      index, DrsAUIntervalShow(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAUIntervalFindRightLengths(KHE_DRS_EXPR_SEQUENCE es,          */
/*    int index, int *q, int *r)                                             */
/*                                                                           */
/*  Similar to KheDrsAUIntervalFindRight, only the interval includes the     */
/*  child whose index in es->open_children_by_day is index, and instead      */
/*  of returning the interval, we return *q, the number of unassigned        */
/*  children in it, and *r, the number of active children in it.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAUIntervalFindRightLengths(KHE_DRS_EXPR_SEQUENCE es,
  int index, int *q, int *r)
{
  KHE_DRS_CLOSED_SEQ dcs;  int i;

  /* handle each unassigned child and its *following* closed interval */
  *q = *r = 0;
  for( i = index + 1;  i < HaArrayCount(es->closed_seqs);  i++ )
  {
    dcs = HaArray(es->closed_seqs, i);

    /* include the open (unassigned) child just before dcs */
    *q += 1;

    /* include the active children at the left end of dcs */
    *r += dcs->active_at_left;

    /* if there is an inactive child in dcs, it's time to stop */
    if( !KheDrsClosedSeqAllActive(dcs) )
      return;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalMerge(KHE_DRS_AU_INTERVAL aui_left,  */
/*    KHE_DRS_AU_INTERVAL aui_right, bool sep_is_active)                     */
/*                                                                           */
/*  Return the merge of aui_left and aui_right.  They are separated by one   */
/*  child, and that child is active if sep_is_active.                        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalMerge(KHE_DRS_AU_INTERVAL aui_left,
  KHE_DRS_AU_INTERVAL aui_right, bool sep_is_active)
{
  HnAssert(aui_left.stop_index + 1 == aui_right.start_index,
    "KheDrsAUIntervalMerge internal error");
  return KheDrsAUIntervalMake(aui_left.start_index, aui_right.stop_index,
    aui_left.has_active_child || aui_right.has_active_child || sep_is_active);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAUIntervalDev(KHE_DRS_AU_INTERVAL aui,                         */
/*    KHE_DRS_EXPR_SEQUENCE es)                                              */
/*                                                                           */
/*  Return the deviation of au-interval aui.  If aui.has_active_child is     */
/*  false, the interval has no active child so return 0.  This should        */
/*  also handle the special case of len == 0, where the result is also 0.    */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAUIntervalDev(KHE_DRS_AU_INTERVAL aui,
  KHE_DRS_EXPR_SEQUENCE es)
{
  int len;

  /* return 0 if the interval has no active child (including len == 0) */
  if( !aui.has_active_child )
    return 0;

  /* calculate the length and deviation */
  len = aui.stop_index - aui.start_index;
  return len < es->min_limit ? es->min_limit - len : 0;
}


/*****************************************************************************/
/*                                                                           */
/*  char *DrsAUIntervalShow(KHE_DRS_AU_INTERVAL aui)                         */
/*                                                                           */
/*  Return a display of aui in static memory.                                */
/*                                                                           */
/*****************************************************************************/

static char *DrsAUIntervalShow(KHE_DRS_AU_INTERVAL aui)
{
  static char buff[20];
  snprintf(buff, 20, "[%d, %d%s]", aui.start_index, aui.stop_index,
    aui.has_active_child ? "a" : "");
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsAUIntervalCost(KHE_DRS_AU_INTERVAL aui,                   */
/*    KHE_DRS_EXPR_SEQUENCE es)                                              */
/*                                                                           */
/*  Return the cost of aui.                                                  */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsAUIntervalCost(KHE_DRS_AU_INTERVAL aui,
  KHE_DRS_EXPR_SEQUENCE es)
{
  int dev;
  dev = KheDrsAUIntervalDev(aui, es);
  return dev == 0 ? 0 : f(es, dev);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_SEQUENCE - dominance testing"                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprSequenceChildDomTest(                         */
/*    KHE_DRS_EXPR_SEQUENCE es, int open_day_index,                          */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return a dom test suited to a child of es.                               */
/*                                                                           */
/*****************************************************************************/

/* *** needs to be replaced by KheDrsExprDomTest
static KHE_DRS_DOM_TEST KheDrsExprSequenceChildDomTest(
  KHE_DRS_EXPR_SEQUENCE es, int open_day_index,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b;

  ** calculate a **
  a = (es->max_limit < INT_MAX ? -1 : INT_MAX);

  ** calculate b **
  b = (es->min_limit > 0 ? INT_MAX : 0);

  ** make and return the dom test **
  return KheDrsDomTestMakeInt(KHE_DRS_DOM_TEST_SEPARATE, child_e, false, 0,
    INT_MAX, a, b, ** false, false, ** 0, NULL, NULL, NULL, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSequenceOpenDayIndexToChildIndex(                          */
/*    KHE_DRS_EXPR_SEQUENCE es, int open_day_index)                          */
/*                                                                           */
/*  Convert open_day_index to the index of the first open child whose        */
/*  last open day is that day, and from there to the index in the            */
/*  full list of children of that open child.                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprSequenceOpenDayIndexToChildIndex(
  KHE_DRS_EXPR_SEQUENCE es, int open_day_index)
{
  int open_child_index;  KHE_DRS_CLOSED_SEQ dcs;
  open_child_index = KheDrsOpenChildrenBefore(&es->open_children_by_day,
    open_day_index);
  HnAssert(open_child_index <
    KheDrsOpenChildrenCount(&es->open_children_by_day),
    "KheDrsExprSequenceOpenDayIndexToChildIndex internal error");
  dcs = HaArray(es->closed_seqs, open_child_index);
  return dcs->stop_index;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprSequenceDomTest(KHE_DRS_EXPR_SEQUENCE es,     */
/*    int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return a dom test suited to es on the open day with this index.          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprSequenceDomTest(KHE_DRS_EXPR_SEQUENCE es,
  int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b, p, q, r, t, index;  KHE_DRS_CONSTRAINT dc;
  KHE_DRS_DOM_TEST_TYPE dom_test_type;  KHE_DRS_DIM2_TABLE d2;

  /* calculate a */
  if( es->max_limit < INT_MAX )
  {
    t = KheDrsOpenChildrenAtOrAfter(&es->open_children_by_day,open_day_index);
    a = es->max_limit - t;
  }
  else
    a = INT_MAX;

  /* calculate b */
  b = es->min_limit;
  /* ***
  if( es->min_limit > 0 )
    b = es->min_limit;
  else
    b = 0;
  *** */

  /* sort out tabulated dominance test */
  index = KheDrsExprSequenceOpenDayIndexToChildIndex(es, open_day_index+1);
  p = HaArrayCount(es->children) - index;
  KheDrsAUIntervalFindRightLengths(es, open_day_index + 1, &q, &r);

  /* make and return the dom test */
  dc = KheDrsExprCostConstraint((KHE_DRS_EXPR_COST) es, drs);
  d2 = KheDrsDim5TableGet3(dc->sequence_main_dom_table5, p, q, r);
  if( q < 0 || DEBUG44_MONITOR(es->monitor->monitor) )
  {
    fprintf(stderr, "[ KheDrsExprSequenceDomTest(%s, %s, drs)\n",
      KheMonitorId(es->monitor->monitor),
      KheDrsDayId(HaArray(drs->open_days, open_day_index)));
    fprintf(stderr, "  p = %d, q = %d, r = %d, dts:\n", p, q, r);
    KheDrsDim2TableDebug(d2, 2, 2, stderr);
    fprintf(stderr, "]\n");
  }
  HnAssert(q >= 0, "KheDrsExprSequenceDayDomTest internal error (q = %d)", q);
  dom_test_type = KheDomTestTypeActual(drs->solve_dom_test_type,
    es->cost_fn != KHE_QUADRATIC_COST_FUNCTION, false);
  return KheDrsDomTestMakeInt(dom_test_type, (KHE_DRS_EXPR) es,
    false, /* es->min_limit, es->max_limit, */ a, b,
    es->combined_weight, d2, NULL, es->monitor, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_SEQUENCE"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSequenceDelta(KHE_DRS_EXPR_SEQUENCE es,                    */
/*    int lower_det, int upper_det)                                          */
/*                                                                           */
/*  Return the deviation of es for the given lower and upper determinants.   */
/*  This is function delta(l, u) from the documentation.                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprSequenceDelta(KHE_DRS_EXPR_SEQUENCE es,
  int z, int lower_det, int upper_det)
{
  if( z == 0 )
    return 0;
  else if( lower_det > es->max_limit )
    return lower_det - es->max_limit;
  else if( upper_det < es->min_limit )
    return es->min_limit - upper_det;
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_SEQUENCE KheDrsExprSequenceMakeBegin(KHE_DRS_MONITOR dm,    */
/*    KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight, int min_limit,    */
/*    int max_limit, int history_before, int history_after, int history,     */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_SEQUENCE object with these attributes.           */
/*                                                                           */
/*  Implementation note.  The value 0 passed for value_ub is an initial      */
/*  value only.  It will be updated later by by AddChild.                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_SEQUENCE KheDrsExprSequenceMakeBegin(KHE_DRS_MONITOR dm,
  KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight, int min_limit,
  int max_limit, int history_before, int history_after, int history,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_SEQUENCE res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_SEQUENCE_TAG,dr,drs);
  res->cost_fn = cost_fn;
  res->combined_weight = combined_weight;
  res->monitor = dm;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->history_before = history_before;
  res->history_after = history_after;
  res->history = history;
  res->seq_type = KHE_DRS_SEQ_NONE;  /* may change later */
  res->seq_index = 0;
  res->adjust_type = KheDrsAdjustType(cost_fn, max_limit);
  /* res->first_open_child_index = 0; */  /* ready for when we open */
  HaArrayInit(res->closed_seqs, drs->arena);
  HaArrayAddLast(res->closed_seqs, KheDrsClosedSeqMakeInit(drs));
  KheDrsAddWeight(drs, combined_weight);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSequenceAddChild(KHE_DRS_EXPR_SEQUENCE es,                */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to es.  Update es's state          */
/*  appropriately.  That is, extend the sole closed sequence one place.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSequenceAddChild(KHE_DRS_EXPR_SEQUENCE es,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CLOSED_SEQ dcs;
  HnAssert(HaArrayCount(es->closed_seqs) == 1,
    "KheDrsExprSequenceAddChild internal error (%d closed seqs)",
    HaArrayCount(es->closed_seqs));
  dcs = HaArrayFirst(es->closed_seqs);
  KheDrsClosedSeqExtendToRight(dcs, child_e->value.i);
  /* es->value_ub.i += 2;  done by KheDrsExprSequenceMakeEnd */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSequenceSetSeqType(KHE_DRS_EXPR_SEQUENCE es,              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Set es->seq_type and es->seq_index if es matches days_frame.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSequenceSetSeqType(KHE_DRS_EXPR_SEQUENCE es,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  bool found_positive, found_negative, found_single;
  KHE_DRS_EXPR child_e, child_child_e;  int i, index, pos;
  KHE_DRS_EXPR_BUSY_DAY busy_day_e;  KHE_DRS_EXPR_FREE_DAY free_day_e;
  KHE_DRS_EXPR_BUSY_TIME busy_time_e;  KHE_TIME_GROUP frame_tg;

  /* fail immediately if the number of children is wrong */
  if( HaArrayCount(es->children) != KheFrameTimeGroupCount(drs->days_frame) )
    return;

  /* compare time groups and check all positive, negative, or positive single */
  found_positive = found_negative = found_single = false;
  index = -1;
  HaArrayForEach(es->children, child_e, i)
  {
    frame_tg = KheFrameTimeGroup(drs->days_frame, i);

    if( child_e->tag == KHE_DRS_EXPR_OR_TAG )
    {
      /* if not a single child, fail */
      if( HaArrayCount(child_e->children) != 1 )
	return;

      child_child_e = HaArrayFirst(child_e->children);
      if( child_child_e->tag == KHE_DRS_EXPR_BUSY_DAY_TAG )
      {
	/* this is a positive, so if not all positive, fail */
	if( found_negative || found_single )
	  return;
	found_positive = true;

	/* if time groups don't match, fail */
	busy_day_e = (KHE_DRS_EXPR_BUSY_DAY) child_child_e;
	if( !KheTimeGroupEqual(busy_day_e->time_group, frame_tg) )
	  return;
      }
      else if( child_child_e->tag == KHE_DRS_EXPR_BUSY_TIME_TAG )
      {
	/* this is a single positive, so if not all single, fail */
	if( found_negative || found_positive )
	  return;
	found_single = true;

	/* its time has to be an element of frame_tg */
	busy_time_e = (KHE_DRS_EXPR_BUSY_TIME) child_child_e;
	if( !KheTimeGroupContains(frame_tg, busy_time_e->time, &pos) )
	  return;

	/* its time has to have the same index as the other singles */
	if( index == -1 )
	  index = pos;
	else if( pos != index )
	  return;
      }
      else
	return;
    }
    else if( child_e->tag == KHE_DRS_EXPR_AND_TAG )
    {
      /* this is negative, so if not all negative, fail */
      if( found_positive || found_single )
	return;
      found_negative = true;

      /* if not a single child, fail */
      if( HaArrayCount(child_e->children) != 1 )
	return;

      /* if not a free day child, fail */
      child_child_e = HaArrayFirst(child_e->children);
      if( child_child_e->tag != KHE_DRS_EXPR_FREE_DAY_TAG )
	return;

      /* if time groups don't match, fail */
      free_day_e = (KHE_DRS_EXPR_FREE_DAY) child_child_e;
      if( !KheTimeGroupEqual(free_day_e->time_group, frame_tg) )
	return;
    }
    else
    {
      HnAbort("KheDrsExprSequenceSetSeqType internal error 1");
      return;  /* keep compiler happy */
    }
  }

  /* all good, so wrap up */
  if( found_positive )
    es->seq_type = KHE_DRS_SEQ_DAYS_POSITIVE, es->seq_index = -1;
  else if( found_negative )
    es->seq_type = KHE_DRS_SEQ_DAYS_NEGATIVE, es->seq_index = -1;
  else if( found_single )
    es->seq_type = KHE_DRS_SEQ_SINGLE_POSITIVE, es->seq_index = index;
  else
    return;

  /* if we get to this point, success */
  if( DEBUG54 )
  {
    fprintf(stderr, "[ %s matches frame:\n", KheDrsMonitorId(es->monitor));
    KheDrsExprDebug((KHE_DRS_EXPR) es, drs, 1, 2, stderr);
    fprintf(stderr, "]\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSequenceAdjustLimits(KHE_DRS_EXPR_SEQUENCE es,            */
/*    int ub)                                                                */
/*                                                                           */
/*  Adjust the limits of es.                                                 */
/*                                                                           */
/*****************************************************************************/

/* *** I've decided that this is not worth doing
static void KheDrsExprSequenceAdjustLimits(KHE_DRS_EXPR_SEQUENCE es,
  int ub)
{
  ** make sure limits are consistent **
  HnAssert(es->min_limit <= es->max_limit,
    "KheDrsExprSequenceAdjustLimits: before adjustment, inconsistent limits"
    " (min %d, max %d) for monitor %s", es->min_limit, es->max_limit,
    KheDrsMonitorId(es->monitor));

  ** adjust es->min_limit if possible **
  if( es->min_limit < 1 && 1 <= es->max_limit )
    es->min_limit = 1;

  ** adjust es->max_limit if possible **
  if( es->max_limit > ub && ub >= es->min_limit )
    es->max_limit = ub;

  ** make sure limits are consistent **
  HnAssert(es->min_limit <= es->max_limit,
    "KheDrsExprSequenceAdjustLimits: after adjustment, inconsistent limits"
    " (min %d, max %d) for monitor %s", es->min_limit, es->max_limit,
    KheDrsMonitorId(es->monitor));
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprSequenceValueUpperBound(KHE_DRS_EXPR_SEQUENCE es)          */
/*                                                                           */
/*  Return a suitable value upper bound for es.  Here ub is an upper         */
/*  bound on the sum of the values of the children, including history.       */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprSequenceValueUpperBound(KHE_DRS_EXPR_SEQUENCE es,
  int ub)
{
  int res;
  res = es->min_limit - 1;
  if( res < ub - es->max_limit )
    res = ub - es->max_limit;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSequenceMakeEnd(KHE_DRS_EXPR_SEQUENCE es,                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  End the initialization of es.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSequenceMakeEnd(KHE_DRS_EXPR_SEQUENCE es,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int ub;

  KheDrsExprInitEnd((KHE_DRS_EXPR) es, drs);

  /* get the total number of children, including history children */
  ub = HaArrayCount(es->children) + es->history + es->history_after;

  /* adjust limits */
  /* KheDrsExprSequenceAdjustLimits(es, ub); */

  /* set the value upper bound */
  es->value_ub.i = KheDrsExprSequenceValueUpperBound(es, ub);

  /* set constraint and sequence type */
  KheDrsExprCostSetConstraint((KHE_DRS_EXPR_COST) es, es->history, drs);
  KheDrsExprSequenceSetSeqType(es, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSequenceClosedSeqDebug(KHE_DRS_EXPR_SEQUENCE es,          */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug printf of es's closed sequences.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSequenceClosedSeqDebug(KHE_DRS_EXPR_SEQUENCE es,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_CLOSED_SEQ dcs;  int i;
  HaArrayForEach(es->closed_seqs, dcs, i)
    KheDrsClosedSeqDebug(dcs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSequenceChildHasOpened(KHE_DRS_EXPR_SEQUENCE es,          */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform internal expression es that its child_index'th child, child_e,    */
/*  has opened.  The child will have its correct closed_state and value      */
/*  fields when this is called.  It is added here to es->open_children.      */
/*                                                                           */
/*  Implementation note.  There are three indexes here whose meaning can     */
/*  easily be confused:                                                      */
/*                                                                           */
/*    child_index                                                            */
/*      This is the index of child_e in the sequence of all children of      */
/*      es.  It is only used here at the start, by KheDrsClosedSeqSplit.     */
/*      Closed sequences use this kind of index for their endpoints.         */
/*                                                                           */
/*    open_index                                                             */
/*      This is the open index of child_e - the index of its last open       */
/*      day, possibly adjusted.  It is not used by this function.  The       */
/*      days that things happen on do not matter here.                       */
/*                                                                           */
/*    index                                                                  */
/*      This is the index of child_e in es->open_children_by_day.  This      */
/*      is the index mainly used by this function.  In paricular, it is      */
/*      used to index into es->closed_seqs, since there is one closed        */
/*      sequence for each gap between open children, including before the    */
/*      first open child and after the last open child.                      */
/*                                                                           */
/*****************************************************************************/

#define DEBUG_RERUN(drs, es) (DEBUG14 && RERUN_PRINT((drs), (es)->monitor))

static void KheDrsExprSequenceChildHasOpened(KHE_DRS_EXPR_SEQUENCE es,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CLOSED_SEQ dcs, new_dcs;  KHE_DRS_EXPR e;  int i, index;
  KHE_DRS_AU_INTERVAL aui_left, aui_right, aui_merged, aui_before, aui_after;
  KHE_DRS_A_INTERVAL ai_left, ai_right, ai_right_before, ai_merged;
  KHE_COST delta_cost;

  /* add child_e to the end of the sequence of open children */
  if( DEBUG_RERUN(drs, es) )
    fprintf(stderr, "[ KheDrsExprSequenceChildHasOpened(%s, child_e, %d)\n",
      KheDrsMonitorId(es->monitor), child_index);
  KheDrsOpenChildrenAddChild(&es->open_children_by_day, child_e);

  /* split the last closed seq at child_index */
  dcs = HaArrayLast(es->closed_seqs);
  new_dcs = KheDrsClosedSeqSplit(dcs, child_index, es, drs);
  HaArrayAddLast(es->closed_seqs, new_dcs);
  if( DEBUG_RERUN(drs, es) )
  {
    KheDrsExprSequenceClosedSeqDebug(es, 2, 2, stderr);
    HaArrayForEach(es->children, e, i)
      KheDrsExprDebug(e, drs, 2, 4, stderr);
  }

  /* find the index of child_e in es->open_children (easy: at the end) */
  index = KheDrsOpenChildrenCount(&es->open_children_by_day) - 1;

  /* update the au-intervals and a-intervals to take account of the change */
  if( child_e->value.i == 0 )
  {
    /* the opened child moves from inactive to unassigned */
    /* the au-intervals on each side merge */
    aui_left = KheDrsAUIntervalFindLeft(es, index);
    aui_right = KheDrsAUIntervalFindRight(es, index, drs);
    aui_merged = KheDrsAUIntervalMerge(aui_left, aui_right, false);
    delta_cost = KheDrsAUIntervalCost(aui_merged, es)
      - KheDrsAUIntervalCost(aui_left, es)
      - KheDrsAUIntervalCost(aui_right, es);
    drs->solve_start_cost += delta_cost;
    if( RERUN_DEBUG(drs) && delta_cost != 0 )
      fprintf(stderr,"  KheDrsExprSequenceChildHasOpened: drs->solve_start_cost"
	" += (%.5f - %.5f - %.5f) = %.5f\n",
        KheCostShow(KheDrsAUIntervalCost(aui_merged, es)),
	KheCostShow(KheDrsAUIntervalCost(aui_left, es)),
	KheCostShow(KheDrsAUIntervalCost(aui_right, es)),
	KheCostShow(drs->solve_start_cost));
    KheDrsMonitorUpdateRerunCost(es->monitor, (KHE_DRS_EXPR) es, drs,
      NULL, KHE_DRS_OPEN, "open0a", child_index, "+--",
      KheDrsAUIntervalCost(aui_merged, es),
      KheDrsAUIntervalCost(aui_left, es),
      KheDrsAUIntervalCost(aui_right, es));

    /* the a-interval to the right changes its unassigned_precedes */
    if( es->cost_fn == KHE_STEP_COST_FUNCTION )
    {
      ai_right_before = KheDrsAIntervalFindRight(es, index, false);
      ai_right        = KheDrsAIntervalFindRight(es, index, true);
      delta_cost = KheDrsAIntervalCost(ai_right, es)
	- KheDrsAIntervalCost(ai_right_before, es);
      drs->solve_start_cost += delta_cost;
      if( RERUN_DEBUG(drs) && delta_cost != 0 )
	fprintf(stderr,
	  "  KheDrsExprSequenceChildHasOpened: drs->solve_start_cost"
	  " += (%.5f - %.5f) = %.5f\n",
	  KheCostShow(KheDrsAIntervalCost(ai_right, es)),
	  KheCostShow(KheDrsAIntervalCost(ai_right_before, es)),
	  KheCostShow(drs->solve_start_cost));
      KheDrsMonitorUpdateRerunCost(es->monitor, (KHE_DRS_EXPR) es,
	drs, NULL, KHE_DRS_OPEN, "open0b", child_index, "+-",
	KheDrsAIntervalCost(ai_right, es),
	KheDrsAIntervalCost(ai_right_before, es));
    }
  }
  else
  {
    /* the opened child moves from active to unassigned */
    /* the enclosing au-interval is unchanged, but its cost may change */
    aui_left = KheDrsAUIntervalFindLeft(es, index);
    aui_right = KheDrsAUIntervalFindRight(es, index, drs);
    aui_before = KheDrsAUIntervalMerge(aui_left, aui_right, true);
    aui_after = KheDrsAUIntervalMerge(aui_left, aui_right, false);
    delta_cost = KheDrsAUIntervalCost(aui_after, es)
      - KheDrsAUIntervalCost(aui_before, es);
    drs->solve_start_cost += delta_cost;
    if( RERUN_DEBUG(drs) && delta_cost != 0 )
      fprintf(stderr,
	"  KheDrsExprSequenceChildHasOpened: drs->solve_start_cost"
	" += (%.5f - %.5f) = %.5f\n",
	KheCostShow(KheDrsAUIntervalCost(aui_after, es)),
	KheCostShow(KheDrsAUIntervalCost(aui_before, es)),
	KheCostShow(drs->solve_start_cost));
    KheDrsMonitorUpdateRerunCost(es->monitor, (KHE_DRS_EXPR) es,
      drs, NULL, KHE_DRS_OPEN, "open1a", child_index, "+-",
      KheDrsAUIntervalCost(aui_after, es),
      KheDrsAUIntervalCost(aui_before, es));

    /* the enclosing a-interval splits */
    /* NB unassigned_precedes is correct when ai_left or ai_right is empty */
    ai_left = KheDrsAIntervalFindLeft(es, index);
    ai_right = KheDrsAIntervalFindRight(es, index, true);
    ai_merged = KheDrsAIntervalMerge(ai_left, ai_right);
    delta_cost = KheDrsAIntervalCost(ai_left, es)
      + KheDrsAIntervalCost(ai_right, es)
      - KheDrsAIntervalCost(ai_merged, es);
    drs->solve_start_cost += delta_cost;
    if( RERUN_DEBUG(drs) && delta_cost != 0 )
      fprintf(stderr,
	"  KheDrsExprSequenceChildHasOpened: drs->solve_start_cost"
	" += (%.5f + %.5f - %.5f) = %.5f\n",
	KheCostShow(KheDrsAIntervalCost(ai_left, es)),
	KheCostShow(KheDrsAIntervalCost(ai_right, es)),
	KheCostShow(KheDrsAIntervalCost(ai_merged, es)),
	KheCostShow(drs->solve_start_cost));
    KheDrsMonitorUpdateRerunCost(es->monitor, (KHE_DRS_EXPR) es,
      drs, NULL, KHE_DRS_OPEN, "open1b", child_index, "++-",
      KheDrsAIntervalCost(ai_left, es),
      KheDrsAIntervalCost(ai_right, es),
      KheDrsAIntervalCost(ai_merged, es));
  }
  if( DEBUG_RERUN(drs, es) )
    fprintf(stderr, "] KheDrsExprSequenceChildHasOpened\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSequenceChildHasClosed(KHE_DRS_EXPR_SEQUENCE es,          */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform internal expression es that its child_index'th child, child_e,    */
/*  has closed.  The child will have its correct closed_state and value      */
/*  fields when this is called; but it will not have been removed from       */
/*  e->open_children.                                                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSequenceChildHasClosed(KHE_DRS_EXPR_SEQUENCE es,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int index;  KHE_DRS_CLOSED_SEQ dcs1, dcs2;  KHE_COST delta_cost;
  KHE_DRS_AU_INTERVAL aui_left, aui_right, aui_merged, aui_before, aui_after;
  KHE_DRS_A_INTERVAL ai_left, ai_right, ai_right_before, ai_merged;

  /* find the index of child_e in es->open_children (easy: at the start) */
  index = 0;
  /* index = es->first_open_child_index; */
  HnAssert(HaArray(es->open_children_by_day.open_children, index).child_e
    == child_e, "KheDrsExprSequenceChildHasClosed internal error 1");
  /* es->first_open_child_index++; */

  /* update the au-intervals and a-intervals to take account of the change */
  if( child_e->value.i == 0 )
  {
    /* the closed child moves from unassigned to inactive */
    /* the enclosing au-interval splits */
    aui_left = KheDrsAUIntervalFindLeft(es, 0 /* not index */);
    aui_right = KheDrsAUIntervalFindRight(es, index, drs);
    aui_merged = KheDrsAUIntervalMerge(aui_left, aui_right, false);
    delta_cost = KheDrsAUIntervalCost(aui_left, es)
      + KheDrsAUIntervalCost(aui_right, es)
      - KheDrsAUIntervalCost(aui_merged, es);
    drs->solve_start_cost += delta_cost;
    if( RERUN_DEBUG(drs) && delta_cost != 0 )
      fprintf(stderr,
	"  KheDrsExprSequenceChildHasClosed: drs->solve_start_cost"
	" += (%.5f + %.5f - %.5f) = %.5f\n",
	KheCostShow(KheDrsAUIntervalCost(aui_left, es)),
	KheCostShow(KheDrsAUIntervalCost(aui_right, es)),
	KheCostShow(KheDrsAUIntervalCost(aui_merged, es)),
	KheCostShow(drs->solve_start_cost));
    KheDrsMonitorUpdateRerunCost(es->monitor, (KHE_DRS_EXPR) es,
      drs, NULL, KHE_DRS_CLOSE, "close0a", child_index, "++-",
      KheDrsAUIntervalCost(aui_left, es),
      KheDrsAUIntervalCost(aui_right, es),
      KheDrsAUIntervalCost(aui_merged, es));

    /* the a-interval to the right changes its unassigned_precedes */
    ai_right_before = KheDrsAIntervalFindRight(es, index, true);
    ai_right        = KheDrsAIntervalFindRight(es, index, false);
    delta_cost = KheDrsAIntervalCost(ai_right, es)
      - KheDrsAIntervalCost(ai_right_before, es);
    drs->solve_start_cost += delta_cost;
    if( RERUN_DEBUG(drs) && delta_cost != 0 )
      fprintf(stderr,
	"  KheDrsExprSequenceChildHasClosed: drs->solve_start_cost"
	" += (%.5f - %.5f) = %.5f\n",
	KheCostShow(KheDrsAIntervalCost(ai_right, es)),
	KheCostShow(KheDrsAIntervalCost(ai_right_before, es)),
	KheCostShow(drs->solve_start_cost));
    KheDrsMonitorUpdateRerunCost(es->monitor, (KHE_DRS_EXPR) es,
      drs, NULL, KHE_DRS_CLOSE, "close0b", child_index, "+-",
      KheDrsAIntervalCost(ai_right, es),
      KheDrsAIntervalCost(ai_right_before, es));
  }
  else
  {
    /* the closed child moves from unassigned to active */
    /* the enclosing au-interval is unchanged, but its cost may change */
    aui_left = KheDrsAUIntervalFindLeft(es, 0 /* not index */);
    aui_right = KheDrsAUIntervalFindRight(es, index, drs);
    aui_before = KheDrsAUIntervalMerge(aui_left, aui_right, false);
    aui_after = KheDrsAUIntervalMerge(aui_left, aui_right, true);
    delta_cost = KheDrsAUIntervalCost(aui_after, es)
      - KheDrsAUIntervalCost(aui_before, es);
    drs->solve_start_cost += delta_cost;
    if( RERUN_DEBUG(drs) && delta_cost != 0 )
      fprintf(stderr,
	"  KheDrsExprSequenceChildHasClosed: drs->solve_start_cost"
	" += (%.5f - %.5f) = %.5f\n",
	KheCostShow(KheDrsAUIntervalCost(aui_after, es)),
	KheCostShow(KheDrsAUIntervalCost(aui_before, es)),
	KheCostShow(drs->solve_start_cost));
    KheDrsMonitorUpdateRerunCost(es->monitor, (KHE_DRS_EXPR) es,
      drs, NULL, KHE_DRS_CLOSE, "close1a", child_index, "+-",
      KheDrsAUIntervalCost(aui_after, es),
      KheDrsAUIntervalCost(aui_before, es));

    /* the a-intervals on each side merge */
    ai_left = KheDrsAIntervalFindLeft(es, 0 /* not index */);
    ai_right = KheDrsAIntervalFindRight(es, index, true);
    ai_merged = KheDrsAIntervalMerge(ai_left, ai_right);
    delta_cost = KheDrsAIntervalCost(ai_merged, es)
      - KheDrsAIntervalCost(ai_left, es)
      - KheDrsAIntervalCost(ai_right, es);
    drs->solve_start_cost += delta_cost;
    if( RERUN_DEBUG(drs) && delta_cost != 0 )
      fprintf(stderr,
	"  KheDrsExprSequenceChildHasClosed: drs->solve_start_cost"
	" += (%.5f - %.5f = %.5f) = %.5f\n",
        KheCostShow(KheDrsAIntervalCost(ai_merged, es)),
        KheCostShow(KheDrsAIntervalCost(ai_left, es)),
        KheCostShow(KheDrsAIntervalCost(ai_right, es)),
	KheCostShow(drs->solve_start_cost));
    KheDrsMonitorUpdateRerunCost(es->monitor, (KHE_DRS_EXPR) es,
      drs, NULL, KHE_DRS_CLOSE, "close1b", child_index, "+--",
      KheDrsAIntervalCost(ai_merged, es), KheDrsAIntervalCost(ai_left, es),
      KheDrsAIntervalCost(ai_right, es));
  }

  /* remove the open child and merge the two relevant closed seqs */
  KheDrsOpenChildrenDeleteChild(&es->open_children_by_day, child_e);
  dcs1 = HaArray(es->closed_seqs, 0);
  dcs2 = HaArray(es->closed_seqs, index + 1);
  KheDrsClosedSeqMerge(dcs1, dcs2, es, drs);
  HaArrayDeleteAndShift(es->closed_seqs, index + 1);
  /* HaArrayPut(es->closed_seqs, open_index + 1, NULL); */  /* defensive */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSequenceSetClosedValue(KHE_DRS_EXPR_SEQUENCE es,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing es, set its value suitably for its closed state.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSequenceSetClosedValue(KHE_DRS_EXPR_SEQUENCE es,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* clear out es->first_open_child_index and es->closed_seqs */
  /* es->first_open_child_index = 0; */
  /* *** cleared out during closing of children now
  while( HaArrayCount(es->closed_seqs) > 0 &&
      HaArrayLast(es->closed_seqs) == NULL )
    HaArrayDeleteLast(es->closed_seqs);
  *** */
  HnAssert(HaArrayCount(es->closed_seqs) == 1,
    "KheDrsExprSequenceSetClosedValue internal error 1");

  /* nothing else to do here, since es has a cost but no value */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDebugActiveLen(KHE_DRS_EXPR_SEQUENCE es, char *label,         */
/*    KHE_DRS_CLOSED_SEQ dcs, int active_len,KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Debug these values if wanted.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDebugActiveLen(KHE_DRS_EXPR_SEQUENCE es, char *label,
  int open_index, KHE_DRS_EXPR child_e, KHE_DRS_SIGNATURE prev_sig,
  KHE_DRS_CLOSED_SEQ dcs, int active_len, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( RERUN_PRINT(drs, es->monitor) )
  {
    fprintf(stderr, "  [ active_len %s: ", label);
    KheDrsClosedSeqDebug(dcs, 1, -1, stderr);
    fprintf(stderr, " --> %d\n", active_len);
    fprintf(stderr, "    child at open_index %d:", open_index);
    KheDrsExprDebug(child_e, drs, 1, 2, stderr);
    fprintf(stderr, "  ]\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSequenceEvalSignature(KHE_DRS_EXPR_SEQUENCE es,           */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Evaluate expression es in sig.                                           */
/*                                                                           */
/*  Implementation note.  Much of this code is the same as the code for      */
/*  closing each child_e, except that the a-interval and au-interval at      */
/*  left both have length active_len, and the cost is updated in             */
/*  next_soln->cost rather than in drs->solve_start_cost.                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSequenceEvalSignature(KHE_DRS_EXPR_SEQUENCE es,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, /* int next_di, */
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int index, i1, i2, active_len, next_di;  KHE_DRS_EXPR child_e;
  KHE_DRS_AU_INTERVAL aui_left, aui_right, aui_merged;
  KHE_DRS_AU_INTERVAL aui_before, aui_after;
  KHE_DRS_A_INTERVAL ai_right_before, ai_right_after;
  KHE_DRS_A_INTERVAL ai_left, ai_right, ai_merged;
  KHE_DRS_CLOSED_SEQ dcs;  KHE_DRS_VALUE val;  KHE_COST delta_cost;

  /* initialize active_len */
  next_di = KheDrsSignerOpenDayIndex(dsg);
  if( KheDrsOpenChildrenIndexIsFirst(&es->open_children_by_day, next_di) )
  {
    /* first day */
    dcs = HaArrayFirst(es->closed_seqs);
    active_len = dcs->active_at_right;
    if( KheDrsClosedSeqAllActive(dcs) )
      active_len += es->history;
  }
  else
  {
    /* not first day, so retrieve previous active_len */
    active_len = KheDrsExprDaySigVal((KHE_DRS_EXPR) es, next_di-1,prev_sig).i;
  }

  /* handle each child_e whose last open day is next_di */
  KheDrsOpenChildrenForEach(&es->open_children_by_day, next_di, child_e,index)
  {
    if( child_e->value.i == 0 )
    {
      /* child_e moves from unassigned to inactive: update cost */
      /* the enclosing au-interval splits */
      aui_left = KheDrsAUIntervalMakeLeft(es, index, active_len);
      aui_right = KheDrsAUIntervalFindRight(es, index, drs);
      aui_merged = KheDrsAUIntervalMerge(aui_left, aui_right, false);
      delta_cost = KheDrsAUIntervalCost(aui_left, es) +
        KheDrsAUIntervalCost(aui_right, es) -
	KheDrsAUIntervalCost(aui_merged, es);
      KheDrsSignatureAddCost(next_sig, delta_cost);
      if( RERUN_DEBUG(drs) && delta_cost != 0 )
	fprintf(stderr, "  KheDrsExprSequenceEvalSignature: sig->cost += "
	  "(%.5f + %.5f - %.5f) = %.5f from %s\n",
          KheCostShow(KheDrsAUIntervalCost(aui_left, es)),
	  KheCostShow(KheDrsAUIntervalCost(aui_right, es)),
	  KheCostShow(KheDrsAUIntervalCost(aui_merged, es)),
	  KheCostShow(next_sig->cost), KheDrsMonitorId(es->monitor));
      KheDrsMonitorUpdateRerunCost(es->monitor, (KHE_DRS_EXPR) es,
	drs, dsg, KHE_DRS_SEARCH, "search", -1, "++-",
	KheDrsAUIntervalCost(aui_left, es),
	KheDrsAUIntervalCost(aui_right, es),
	KheDrsAUIntervalCost(aui_merged, es));

      /* the a-interval to the right changes its unassigned_precedes */
      ai_right_before = KheDrsAIntervalFindRight(es, index, true);
      ai_right_after  = KheDrsAIntervalFindRight(es, index, false);
      delta_cost = KheDrsAIntervalCost(ai_right_after, es) -
        KheDrsAIntervalCost(ai_right_before, es);
      KheDrsSignatureAddCost(next_sig, delta_cost);
      if( RERUN_DEBUG(drs) && delta_cost != 0 )
	fprintf(stderr, "  KheDrsExprSequenceEvalSignature: sig->cost += "
	  "(%.5f - %.5f) = %.5f from %s\n",
          KheCostShow(KheDrsAIntervalCost(ai_right_after, es)),
          KheCostShow(KheDrsAIntervalCost(ai_right_before, es)),
	  KheCostShow(next_sig->cost), KheDrsMonitorId(es->monitor));
      KheDrsMonitorUpdateRerunCost(es->monitor, (KHE_DRS_EXPR) es,
	drs, dsg, KHE_DRS_SEARCH, "search2", -1, "+-",
	KheDrsAIntervalCost(ai_right_after, es),
        KheDrsAIntervalCost(ai_right_before, es));

      /* set active_len for next iteration (child_e is now inactive) */
      dcs = HaArray(es->closed_seqs, index + 1);
      active_len = dcs->active_at_right;
      KheDrsDebugActiveLen(es, "inactive", index, child_e, prev_sig,
	dcs, active_len, drs);
    }
    else
    {
      /* child_e moves from unassigned to active: update cost */
      /* the enclosing au-interval is unchanged, but its cost may change */
      aui_left = KheDrsAUIntervalMakeLeft(es, index, active_len);
      aui_right = KheDrsAUIntervalFindRight(es, index, drs);
      aui_before = KheDrsAUIntervalMerge(aui_left, aui_right, false);
      aui_after = KheDrsAUIntervalMerge(aui_left, aui_right, true);
      delta_cost = KheDrsAUIntervalCost(aui_after, es)
	- KheDrsAUIntervalCost(aui_before, es);
      KheDrsSignatureAddCost(next_sig, delta_cost);
      if( RERUN_DEBUG(drs) && delta_cost != 0 )
	fprintf(stderr, "  KheDrsExprSequenceEvalSignature: sig->cost += "
	  "(%.5f - %.5f) = %.5f from %s\n",
          KheCostShow(KheDrsAUIntervalCost(aui_after, es)),
          KheCostShow(KheDrsAUIntervalCost(aui_before, es)),
	  KheCostShow(next_sig->cost), KheDrsMonitorId(es->monitor));
      KheDrsMonitorUpdateRerunCost(es->monitor, (KHE_DRS_EXPR) es,
	drs, dsg, KHE_DRS_SEARCH, "search", -1, "+-",
	KheDrsAUIntervalCost(aui_after, es),
	KheDrsAUIntervalCost(aui_before, es));

      /* the a-intervals on each side merge */
      ai_left = KheDrsAIntervalMakeLeft(es, index, active_len);
      ai_right = KheDrsAIntervalFindRight(es, index, true);
      ai_merged = KheDrsAIntervalMerge(ai_left, ai_right);
      delta_cost = KheDrsAIntervalCost(ai_merged, es)
	- KheDrsAIntervalCost(ai_left, es)
	- KheDrsAIntervalCost(ai_right, es);
      KheDrsSignatureAddCost(next_sig, delta_cost);
      if( RERUN_DEBUG(drs) && delta_cost != 0 )
	fprintf(stderr, "  KheDrsExprSequenceEvalSignature: sig->cost += "
	  "(%.5f - %.5f - %.5f) = %.5f from %s\n",
	  KheCostShow(KheDrsAIntervalCost(ai_merged, es)),
	  KheCostShow(KheDrsAIntervalCost(ai_left, es)),
	  KheCostShow(KheDrsAIntervalCost(ai_right, es)),
	  KheCostShow(next_sig->cost), KheDrsMonitorId(es->monitor));
      KheDrsMonitorUpdateRerunCost(es->monitor, (KHE_DRS_EXPR) es,
	drs, dsg, KHE_DRS_SEARCH, "search2", -1, "+--",
	KheDrsAIntervalCost(ai_merged, es),
	KheDrsAIntervalCost(ai_left, es),
	KheDrsAIntervalCost(ai_right, es));

      /* set active_len for next iteration (child_e is now active) */
      dcs = HaArray(es->closed_seqs, index + 1);
      if( KheDrsClosedSeqAllActive(dcs) )
	active_len += 1 + dcs->active_at_right;
      else
        active_len = dcs->active_at_right;
      KheDrsDebugActiveLen(es, "active", index, child_e, prev_sig,
	dcs, active_len, drs);
    }
  }

  if( !KheDrsOpenChildrenIndexIsLast(&es->open_children_by_day, next_di) )
  {
    /* not last day; store active_len (adjusted) in sig */
    val.i = KheDrsAdjustedSigVal(active_len,
      es->adjust_type, es->min_limit, es->max_limit, 0);
    if( DEBUG44_MONITOR(es->monitor->monitor) && val.i >= 4 )
      fprintf(stderr, "  EvalSig(%s): day %d, adjusted_len = %d\n",
	KheDrsMonitorId(es->monitor), next_di, val.i);
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) es);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsExprSequenceShowSeqType(KHE_DRS_EXPR_SEQUENCE es)            */
/*                                                                           */
/*  Show the seq_type and seq_index fields of es.                            */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsExprSequenceShowSeqType(KHE_DRS_EXPR_SEQUENCE es)
{
  static char buff[200];
  switch( es->seq_type )
  {
    case KHE_DRS_SEQ_NONE:

      sprintf(buff, "-");
      break;

    case KHE_DRS_SEQ_DAYS_POSITIVE:

      sprintf(buff, "+ve");
      break;

    case KHE_DRS_SEQ_DAYS_NEGATIVE:

      sprintf(buff, "-ve");
      break;

    case KHE_DRS_SEQ_SINGLE_POSITIVE:

      sprintf(buff, "+%d", es->seq_index);
      break;

    default:

      sprintf(buff, "?");
      break;
  }
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSequenceDoDebug(KHE_DRS_EXPR_SEQUENCE es,                 */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of es which is specific to it.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSequenceDoDebug(KHE_DRS_EXPR_SEQUENCE es,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "SEQUENCE(%s, %s) %s", KheDrsMonitorId(es->monitor),
    KheDrsAdjustTypeShow(es->adjust_type),
    KheDrsExprSequenceShowSeqType(es));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) es, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - deferred functions"                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAddChild(KHE_DRS_EXPR parent, KHE_DRS_EXPR child,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make child a child of parent.  This includes setting parent and child    */
/*  pointers, which is the same for all expressions, but it also includes    */
/*  updating the parent's closed state and value upper bound, hence the      */
/*  big switch.                                                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAddChild(KHE_DRS_EXPR parent, KHE_DRS_EXPR child,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;

  /* link parent and child */
  HnAssert(HaArrayCount(parent->parents) == 0,
    "KheDrsExprAddChild internal error:  too late to add child (1)");
  HnAssert(parent->postorder_index == -1,
    "KheDrsExprAddChild internal error:  too late to add child (2)");
  prnt.expr = parent;
  prnt.index = HaArrayCount(parent->children);
  HaArrayAddLast(child->parents, prnt);
  HaArrayAddLast(parent->children, child);

  /* update state in each parent */
  switch( parent->tag )
  {
    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrAddChild((KHE_DRS_EXPR_OR) parent, child, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndAddChild((KHE_DRS_EXPR_AND) parent, child, drs);
      break;

    /* ***
    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumAddChild((KHE_DRS_EXPR_INT_SUM) parent, child, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumAddChild((KHE_DRS_EXPR_FLOAT_SUM) parent, child, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevAddChild((KHE_DRS_EXPR_INT_DEV) parent, child, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevAddChild((KHE_DRS_EXPR_FLOAT_DEV) parent, child, drs);
      break;
    *** */

    case KHE_DRS_EXPR_COUNTER_TAG:

      KheDrsExprCounterAddChild((KHE_DRS_EXPR_COUNTER) parent,
	child, drs);
      break;

    case KHE_DRS_EXPR_SUM_INT_TAG:

      KheDrsExprSumIntAddChild((KHE_DRS_EXPR_SUM_INT) parent, child, drs);
      break;

    case KHE_DRS_EXPR_SUM_FLOAT_TAG:

      KheDrsExprSumFloatAddChild((KHE_DRS_EXPR_SUM_FLOAT) parent, child, drs);
      break;

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      KheDrsExprSequenceAddChild((KHE_DRS_EXPR_SEQUENCE) parent,
	child, drs);
      break;

    default:

      HnAbort("KheDrsExprAddChild internal error (tag %d)", parent->tag);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KheDrsExprMakeEnd(res, drs)                                              */
/*                                                                           */
/*  Finalize the initialization of e by setting its postorder index and      */
/*  closed value.                                                            */
/*                                                                           */
/*****************************************************************************/

/* *** withdrawn; we're calling the specific versions now
static void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExprMakeEnd(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ** postorder index **
  e->postorder_index = drs->postorder_count++;

  ** closed value **
  KheDrsExprSetClosedValue(e, drs);

  ** type-specific initializations **
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:

      ** nothing to do in these cases **
      break;

    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrMakeEnd((KHE_DRS_EXPR_OR) e, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndMakeEnd((KHE_DRS_EXPR_AND) e, drs);
      break;

    ** ***
    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumMakeEnd((KHE_DRS_EXPR_INT_SUM) e, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumMakeEnd((KHE_DRS_EXPR_FLOAT_SUM) e, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevMakeEnd((KHE_DRS_EXPR_INT_DEV) e, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevMakeEnd((KHE_DRS_EXPR_FLOAT_DEV) e, drs);
      break;
    *** **

    case KHE_DRS_EXPR_COUNTER_TAG:

      KheDrsExprCounterMakeEnd((KHE_DRS_EXPR_COUNTER) e, drs);
      break;

    case KHE_DRS_EXPR_SUM_INT_TAG:

      KheDrsExprSumIntMakeEnd((KHE_DRS_EXPR_SUM_INT) e, drs);
      break;

    case KHE_DRS_EXPR_SUM_FLOAT_TAG:

      KheDrsExprSumFloatMakeEnd((KHE_DRS_EXPR_SUM_FLOAT) e, drs);
      break;

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      KheDrsExprSequenceMakeEnd((KHE_DRS_EXPR_SEQUENCE) e, drs);
      break;

    default:

      HnAbort("KheDrsExprMakeEnd internal error (tag %d)", e->tag);
  }

  ** debug stuff **
  if( KheDrsExprDebugWanted(e) )
  {
    fprintf(stderr, "  KheDrsExprMakeEnd(e), e =\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
  }
  if( DEBUG59 )
  {
    KHE_DRS_EXPR child_e;  int i;
    HaArrayForEach(e->children, child_e, i)
      if( e->postorder_index <= child_e->postorder_index )
      {
	fprintf(stderr, "  KheDrsExprMakeEnd aborting on e:\n");
	KheDrsExprDebug(e, drs, 2, 2, stderr);
	HnAbort("KheDrsExprMakeEnd: e pi %d <= child_e pi %d",
	  e->postorder_index, child_e->postorder_index);
      }
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprChildHasOpened(KHE_DRS_EXPR e, KHE_DRS_EXPR child_e,      */
/*    int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                      */
/*                                                                           */
/*  As part of opening e for solving, inform e that its child_index'th       */
/*  child, child_e, has opened.  When this is called, the child will have    */
/*  the value it had when it was closed.                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprChildHasOpened(KHE_DRS_EXPR e, KHE_DRS_EXPR child_e,
  int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* add child_e to e->open_children */
  /* *** this code moved down into the sub-calls
  if( e->tag == KHE_DRS_EXPR_SEQUENCE_TAG )
    KheDrsOpenChildrenAddChildWithDayUpdate(&e->open_children_by_day, child_e);
  else
    KheDrsOpenChildrenAddChildInDayOrder(&e->open_children_by_day, child_e);
  *** */

  /* alter the state of e to take account of child_e opening */
  switch( e->tag )
  {
    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrChildHasOpened((KHE_DRS_EXPR_OR) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndChildHasOpened((KHE_DRS_EXPR_AND) e,
	child_e, child_index, drs);
      break;

    /* ***
    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumChildHasOpened((KHE_DRS_EXPR_INT_SUM) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumChildHasOpened((KHE_DRS_EXPR_FLOAT_SUM) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevChildHasOpened((KHE_DRS_EXPR_INT_DEV) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevChildHasOpened((KHE_DRS_EXPR_FLOAT_DEV) e,
	child_e, child_index, drs);
      break;
    *** */

    case KHE_DRS_EXPR_COUNTER_TAG:

      KheDrsExprCounterChildHasOpened((KHE_DRS_EXPR_COUNTER) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_SUM_INT_TAG:

      KheDrsExprSumIntChildHasOpened((KHE_DRS_EXPR_SUM_INT) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_SUM_FLOAT_TAG:

      KheDrsExprSumFloatChildHasOpened((KHE_DRS_EXPR_SUM_FLOAT) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      KheDrsExprSequenceChildHasOpened((KHE_DRS_EXPR_SEQUENCE) e,
	child_e, child_index, drs);
      break;

    default:

      HnAbort("KheDrsExprChildHasOpened internal error (tag %d)", e->tag);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprChildHasClosed(KHE_DRS_EXPR e,                            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform internal expression e that its child_index'th child, child_e,     */
/*  has closed.  The child will have its correct closed_state and value      */
/*  fields when this is called.                                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprChildHasClosed(KHE_DRS_EXPR e,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrChildHasClosed((KHE_DRS_EXPR_OR) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndChildHasClosed((KHE_DRS_EXPR_AND) e,
	child_e, child_index, drs);
      break;

    /* ***
    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumChildHasClosed((KHE_DRS_EXPR_INT_SUM) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumChildHasClosed((KHE_DRS_EXPR_FLOAT_SUM) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevChildHasClosed((KHE_DRS_EXPR_INT_DEV) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevChildHasClosed((KHE_DRS_EXPR_FLOAT_DEV) e,
	child_e, child_index, drs);
      break;
    *** */

    case KHE_DRS_EXPR_COUNTER_TAG:

      KheDrsExprCounterChildHasClosed((KHE_DRS_EXPR_COUNTER) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_SUM_INT_TAG:

      KheDrsExprSumIntChildHasClosed((KHE_DRS_EXPR_SUM_INT) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_SUM_FLOAT_TAG:

      KheDrsExprSumFloatChildHasClosed((KHE_DRS_EXPR_SUM_FLOAT) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      KheDrsExprSequenceChildHasClosed((KHE_DRS_EXPR_SEQUENCE) e,
	child_e, child_index, drs);
      break;

    default:

      HnAbort("KheDrsExprChildHasClosed internal error (tag %d)", e->tag);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing e, set its value suitably for its closed state.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      KheDrsExprAssignedTaskSetClosedValue((KHE_DRS_EXPR_ASSIGNED_TASK) e, drs);
      break;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      KheDrsExprBusyTimeSetClosedValue((KHE_DRS_EXPR_BUSY_TIME) e, drs);
      break;

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      KheDrsExprFreeTimeSetClosedValue((KHE_DRS_EXPR_FREE_TIME) e, drs);
      break;

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      KheDrsExprWorkTimeSetClosedValue((KHE_DRS_EXPR_WORK_TIME) e, drs);
      break;

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      KheDrsExprBusyDaySetClosedValue((KHE_DRS_EXPR_BUSY_DAY) e, drs);
      break;

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      KheDrsExprFreeDaySetClosedValue((KHE_DRS_EXPR_FREE_DAY) e, drs);
      break;

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      KheDrsExprWorkDaySetClosedValue((KHE_DRS_EXPR_WORK_DAY) e, drs);
      break;

    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrSetClosedValue((KHE_DRS_EXPR_OR) e, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndSetClosedValue((KHE_DRS_EXPR_AND) e, drs);
      break;

    /* ***
    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumSetClosedValue((KHE_DRS_EXPR_INT_SUM) e, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumSetClosedValue((KHE_DRS_EXPR_FLOAT_SUM) e,drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevSetClosedValue((KHE_DRS_EXPR_INT_DEV) e, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevSetClosedValue((KHE_DRS_EXPR_FLOAT_DEV) e,drs);
      break;
    *** */

    case KHE_DRS_EXPR_COUNTER_TAG:

      KheDrsExprCounterSetClosedValue((KHE_DRS_EXPR_COUNTER) e, drs);
      break;

    case KHE_DRS_EXPR_SUM_INT_TAG:

      KheDrsExprSumIntSetClosedValue((KHE_DRS_EXPR_SUM_INT) e, drs);
      break;

    case KHE_DRS_EXPR_SUM_FLOAT_TAG:

      KheDrsExprSumFloatSetClosedValue((KHE_DRS_EXPR_SUM_FLOAT) e, drs);
      break;

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      KheDrsExprSequenceSetClosedValue((KHE_DRS_EXPR_SEQUENCE) e, drs);
      break;

    default:

      HnAbort("KheDrsExprSetClosedValue internal error (tag %d)", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprDomTest(KHE_DRS_EXPR e,                       */
/*    int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return a dom test suited to e, taking account not just of e itself       */
/*  but also of its context.                                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprDomTest(KHE_DRS_EXPR e,
  int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:

      HnAbort("KheDrsExprDomTest: unexpected type (%d)", e->tag);
      return NULL;  /* keep compiler happy */

    case KHE_DRS_EXPR_OR_TAG:

      return KheDrsExprOrDomTest((KHE_DRS_EXPR_OR) e, open_day_index, drs);

    case KHE_DRS_EXPR_AND_TAG:

      return KheDrsExprAndDomTest((KHE_DRS_EXPR_AND) e, open_day_index, drs);

    case KHE_DRS_EXPR_COUNTER_TAG:

      return KheDrsExprCounterDomTest((KHE_DRS_EXPR_COUNTER) e,
	open_day_index, drs);

    case KHE_DRS_EXPR_SUM_INT_TAG:

      return KheDrsExprSumIntDomTest((KHE_DRS_EXPR_SUM_INT) e,
	open_day_index, drs);

    case KHE_DRS_EXPR_SUM_FLOAT_TAG:

      return KheDrsExprSumFloatDomTest((KHE_DRS_EXPR_SUM_FLOAT) e,
	open_day_index, drs);

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      return KheDrsExprSequenceDomTest((KHE_DRS_EXPR_SEQUENCE) e,
	open_day_index, drs);

    default:

      HnAbort("KheDrsExprDomTest internal error (tag %d)", e->tag);
      return NULL;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprChildDomTest(KHE_DRS_EXPR e,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return a dom test suited to child_e, a child of e.                       */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsExprDomTest
static KHE_DRS_DOM_TEST KheDrsExprChildDomTest(KHE_DRS_EXPR e,
  int open_day_index, KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT parent;
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:

      HnAbort("KheDrsExprChildDomTest: unexpected tag %d", e->tag);
      ** no break, to keep compiler happy **

    case KHE_DRS_EXPR_OR_TAG:
    case KHE_DRS_EXPR_AND_TAG:
    ** ***
    case KHE_DRS_EXPR_INT_SUM_TAG:
    case KHE_DRS_EXPR_FLOAT_SUM_TAG:
    case KHE_DRS_EXPR_INT_DEV_TAG:
    case KHE_DRS_EXPR_FLOAT_DEV_TAG:
    *** **

      HnAssert(HaArrayCount(e->parents) > 0,
	"KheDrsExprChildDomTest internal error");
      parent = HaArrayFirst(e->parents);
      return KheDrsExprChildDomTest(parent.expr, open_day_index, child_e, drs);

    case KHE_DRS_EXPR_COUNTER_TAG:

      return KheDrsExprCounterChildDomTest((KHE_DRS_EXPR_COUNTER) e,
	open_day_index, child_e, drs);

    case KHE_DRS_EXPR_SUM_INT_TAG:

      return KheDrsExprSumIntChildDomTest((KHE_DRS_EXPR_SUM_INT) e,
	open_day_index, child_e, drs);

    case KHE_DRS_EXPR_SUM_FLOAT_TAG:

      return KheDrsExprSumFloatChildDomTest((KHE_DRS_EXPR_SUM_FLOAT) e,
	open_day_index, child_e, drs);

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      return KheDrsExprSequenceChildDomTest((KHE_DRS_EXPR_SEQUENCE) e,
	open_day_index, child_e, drs);

    default:

      HnAbort("KheDrsExprChildDomTest internal error (tag %d)", e->tag);
      return KheDrsExprSequenceChildDomTest((KHE_DRS_EXPR_SEQUENCE) e,
	0, NULL, drs);  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e,                    */
/*    int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return a dom test suited to e on the open day with this open day index.  */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsExprDomTest
static KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e,
  int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:

      HnAbort("KheDrsExprDayDomTest: unexpected tag %d", e->tag);
      ** no break, to keep compiler happy **

    case KHE_DRS_EXPR_OR_TAG:
    case KHE_DRS_EXPR_AND_TAG:
    ** ***
    case KHE_DRS_EXPR_INT_SUM_TAG:
    case KHE_DRS_EXPR_FLOAT_SUM_TAG:
    case KHE_DRS_EXPR_INT_DEV_TAG:
    case KHE_DRS_EXPR_FLOAT_DEV_TAG:
    *** **

      HnAssert(HaArrayCount(e->parents) > 0,
	"KheDrsExprDayDomTest internal error");
      prnt = HaArrayFirst(e->parents);
      return KheDrsExprChildDomTest(prnt.expr, open_day_index, e, drs);

    case KHE_DRS_EXPR_COUNTER_TAG:

      return KheDrsExprCounterDayDomTest((KHE_DRS_EXPR_COUNTER) e,
	open_day_index, drs);

    case KHE_DRS_EXPR_SUM_INT_TAG:

      return KheDrsExprSumIntDayDomTest((KHE_DRS_EXPR_SUM_INT) e,
	open_day_index, drs);

    case KHE_DRS_EXPR_SUM_FLOAT_TAG:

      return KheDrsExprSumFloatDayDomTest((KHE_DRS_EXPR_SUM_FLOAT) e,
	open_day_index, drs);

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      return KheDrsExprSequenceDayDomTest((KHE_DRS_EXPR_SEQUENCE) e,
	open_day_index, drs);

    default:

      HnAbort("KheDrsExprDayDomTest internal error (tag %d)", e->tag);
      return KheDrsExprSequenceDayDomTest((KHE_DRS_EXPR_SEQUENCE) e,
	open_day_index, drs);  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,          */
/*    KHE_DRS_RESOURCE dr)                                                   */
/*                                                                           */
/*  Inform leaf expression e that dtd has been assigned dr.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      KheDrsExprAssignedTaskLeafSet((KHE_DRS_EXPR_ASSIGNED_TASK) e, dtd, dr,
	drs);
      break;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      KheDrsExprBusyTimeLeafSet((KHE_DRS_EXPR_BUSY_TIME) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      KheDrsExprFreeTimeLeafSet((KHE_DRS_EXPR_FREE_TIME) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      KheDrsExprWorkTimeLeafSet((KHE_DRS_EXPR_WORK_TIME) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      KheDrsExprBusyDayLeafSet((KHE_DRS_EXPR_BUSY_DAY) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      KheDrsExprFreeDayLeafSet((KHE_DRS_EXPR_FREE_DAY) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      KheDrsExprWorkDayLeafSet((KHE_DRS_EXPR_WORK_DAY) e, dtd, dr);
      break;

    default:

      HnAbort("KheDrsExprLeafSet internal error: tag %d", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprLeafClear(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform leaf expression e that the assignment of drd to dtd carried out   */
/*  by KheDrsExprLeafSet above has been removed.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprLeafClear(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      KheDrsExprAssignedTaskLeafClear((KHE_DRS_EXPR_ASSIGNED_TASK) e, drs);
      break;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      KheDrsExprBusyTimeLeafClear((KHE_DRS_EXPR_BUSY_TIME) e);
      break;

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      KheDrsExprFreeTimeLeafClear((KHE_DRS_EXPR_FREE_TIME) e);
      break;

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      KheDrsExprWorkTimeLeafClear((KHE_DRS_EXPR_WORK_TIME) e);
      break;

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      KheDrsExprBusyDayLeafClear((KHE_DRS_EXPR_BUSY_DAY) e);
      break;

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      KheDrsExprFreeDayLeafClear((KHE_DRS_EXPR_FREE_DAY) e);
      break;

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      KheDrsExprWorkDayLeafClear((KHE_DRS_EXPR_WORK_DAY) e);
      break;

    default:

      HnAbort("KheDrsExprLeafClear internal error: tag %d");
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,         */
/*    KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE sig,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)                           */
/*                                                                           */
/*  Like KheDrsExprEvalDay except that the results go into ss rather         */
/*  than into next_soln.                                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, /* int next_di, */ KHE_DRS_SIGNATURE next_sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:

      HnAbort("KheDrsExprEvalSignature: external tag (%d)", e->tag);
      break;

    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrEvalSignature((KHE_DRS_EXPR_OR) e, dsg,
	prev_sig, /* next_di, */ next_sig, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndEvalSignature((KHE_DRS_EXPR_AND) e, dsg,
	prev_sig, /* next_di, */ next_sig, drs);
      break;

    /* ***
    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumEvalSignature((KHE_DRS_EXPR_INT_SUM) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumEvalSignature((KHE_DRS_EXPR_FLOAT_SUM) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevEvalSignature((KHE_DRS_EXPR_INT_DEV) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevEvalSignature((KHE_DRS_EXPR_FLOAT_DEV) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;
    *** */

    case KHE_DRS_EXPR_COUNTER_TAG:

      KheDrsExprCounterEvalSignature((KHE_DRS_EXPR_COUNTER) e, dsg,
	prev_sig, /* next_di, */ next_sig, drs, debug);
      break;

    case KHE_DRS_EXPR_SUM_INT_TAG:

      KheDrsExprSumIntEvalSignature((KHE_DRS_EXPR_SUM_INT) e, dsg,
	prev_sig, /* next_di, */ next_sig, drs, debug);
      break;

    case KHE_DRS_EXPR_SUM_FLOAT_TAG:

      KheDrsExprSumFloatEvalSignature((KHE_DRS_EXPR_SUM_FLOAT) e, dsg,
	prev_sig, /* next_di, */ next_sig, drs, debug);
      break;

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      KheDrsExprSequenceEvalSignature((KHE_DRS_EXPR_SEQUENCE) e, dsg,
	prev_sig, /* next_di, */ next_sig, drs);
      break;

    default:

      HnAbort("KheDrsExprEvalSignature internal error");
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)                             */
/*                                                                           */
/*  Unindented debug print of that part of e which is specific to it.        */
/*  If soln != NULL, also print e's signature (if any) in soln.              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      KheDrsExprAssignedTaskDoDebug((KHE_DRS_EXPR_ASSIGNED_TASK) e,
	soln, drs, fp);
      break;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      KheDrsExprBusyTimeDoDebug((KHE_DRS_EXPR_BUSY_TIME) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      KheDrsExprFreeTimeDoDebug((KHE_DRS_EXPR_FREE_TIME) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      KheDrsExprWorkTimeDoDebug((KHE_DRS_EXPR_WORK_TIME) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      KheDrsExprBusyDayDoDebug((KHE_DRS_EXPR_BUSY_DAY) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      KheDrsExprFreeDayDoDebug((KHE_DRS_EXPR_FREE_DAY) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      KheDrsExprWorkDayDoDebug((KHE_DRS_EXPR_WORK_DAY) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrDoDebug((KHE_DRS_EXPR_OR) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndDoDebug((KHE_DRS_EXPR_AND) e, soln, drs, fp);
      break;

    /* ***
    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumDoDebug((KHE_DRS_EXPR_INT_SUM) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumDoDebug((KHE_DRS_EXPR_FLOAT_SUM) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevDoDebug((KHE_DRS_EXPR_INT_DEV) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevDoDebug((KHE_DRS_EXPR_FLOAT_DEV) e, soln, drs, fp);
      break;
    *** */

    case KHE_DRS_EXPR_COUNTER_TAG:

      KheDrsExprCounterDoDebug((KHE_DRS_EXPR_COUNTER) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_SUM_INT_TAG:

      KheDrsExprSumIntDoDebug((KHE_DRS_EXPR_SUM_INT) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_SUM_FLOAT_TAG:

      KheDrsExprSumFloatDoDebug((KHE_DRS_EXPR_SUM_FLOAT) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_SEQUENCE_TAG:

      KheDrsExprSequenceDoDebug((KHE_DRS_EXPR_SEQUENCE) e, soln, drs, fp);
      break;

    default:

      HnAbort("KheDrsExprDoDebug internal error: unknown tag %d", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "solutions"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_ASST_OP" - the type of an operation (unassign, etc.)  */
/*                                                                           */
/*****************************************************************************/

/* *** should use this but currently unused
static char *KheDrsAsstOpShow(KHE_DRS_ASST_OP op)
{
  switch( op )
  {
    case KHE_DRS_ASST_OP_UNASSIGN:	return "-";
    case KHE_DRS_ASST_OP_ASSIGN:	return "+";
    case KHE_DRS_ASST_OP_REPLACE:	return "-+";
    default:				return "?";
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN KheDrsSolnMake(KHE_DRS_SOLN prev_soln,                      */
/*    KHE_COST cost, KHE_DYNAMIC_RESOURCE_SOLVER drs)                        */
/*                                                                           */
/*  Make a new soln with these attributes.                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN KheDrsSolnMake(KHE_DRS_SOLN prev_soln,
  KHE_COST cost, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN res;

  /* get the basic object */
  if( HaArrayCount(drs->soln_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->soln_free_list);
    KheDrsSignatureSetClear(&res->sig_set, cost, drs);  /* just sets cost */
    HaArrayClear(res->prev_tasks);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->prev_tasks, drs->arena);
    KheDrsSignatureSetInit(&res->sig_set, cost, drs->arena);
  }

  /* initialize the fields */
  res->prev_soln = prev_soln;
  res->priqueue_index = 0;  /* means never yet been in priqueue */
  res->durns_squared = 0;
  /* res->total_durn = (prev_soln != NULL ? prev_soln->total_durn : 0); */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnFree(KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Free soln.  Don't worry about whether it is in the priqueue, or          */
/*  indeed in any container.  If it is, the caller will handle it.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnFree(KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* unrefer to all signatures; this may or may not free them */
  KheDrsSignatureSetClear(&soln->sig_set, 0, drs);

  /* put soln on the free list */
  HaArrayAddLast(drs->soln_free_list, soln);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnMarkExpanded(KHE_DRS_SOLN soln)                           */
/*                                                                           */
/*  Mark solution as expanded.                                               */
/*                                                                           */
/*  Note.  All solutions are created unexpanded.  At some point they may     */
/*  be expanded (so this function is called), after which they remain        */
/*  expanded until they are freed.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnMarkExpanded(KHE_DRS_SOLN soln)
{
  soln->priqueue_index = -1;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnNotExpanded(KHE_DRS_SOLN soln)                            */
/*                                                                           */
/*  Return true if soln is not expanded.                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnNotExpanded(KHE_DRS_SOLN soln)
{
  return soln->priqueue_index != -1;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsSolnCost(KHE_DRS_SOLN soln)                               */
/*                                                                           */
/*  Return the cost of soln.                                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsSolnCost(KHE_DRS_SOLN soln)
{
  return soln->sig_set.cost;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnNotForDominanceTesting(KHE_DRS_SOLN soln)                 */
/*                                                                           */
/*  Return true if soln should not be involved in any dominance testing.     */
/*  This is true when there is just one resource, and that resource is       */
/*  assigned to some task on some of the task's days but not all of them.    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnNotForDominanceTesting(KHE_DRS_SOLN soln)
{
  KHE_DRS_TASK_ON_DAY sole_dtd;
  if( HaArrayCount(soln->prev_tasks) != 1 )
    return false;
  sole_dtd = HaArrayFirst(soln->prev_tasks);
  return sole_dtd != NULL && sole_dtd->not_last_day;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DAY KheDrsSolnDay(KHE_DRS_SOLN soln,                             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return the day that soln is for, or NULL if soln is the root solution.   */
/*                                                                           */
/*  Note.  Formerly we stored this as an attribute of soln.  We are now      */
/*  avoiding storing it, to save memory.  But searching for it, as we do     */
/*  here, is expensive and is only done when there is no better way.         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DAY KheDrsSolnDay(KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;  int i;  KHE_DRS_DAY prev_day;

  /* return NULL if this is the root solution, not on any day */
  if( soln->prev_soln == NULL )
    return NULL;

  /* if prev_tasks has a non-NULL task on day, return its day */
  HaArrayForEach(soln->prev_tasks, dtd, i)
    if( dtd != NULL )
      return dtd->day;

  /* else have to recurse back */
  prev_day = KheDrsSolnDay(soln->prev_soln, drs);
  if( prev_day == NULL )
    return HaArray(drs->open_days, 0);
  else
    return HaArray(drs->open_days, prev_day->open_day_index + 1);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNATURE KheDrsSolnEventResourceSignature(KHE_DRS_SOLN soln)    */
/*                                                                           */
/*  Return the event resource signature of soln.  This is the last element   */
/*  of soln's signature set.                                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNATURE KheDrsSolnEventResourceSignature(KHE_DRS_SOLN soln)
{
  return HaArrayLast(soln->sig_set.signatures);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSignatureSetFullHash(KHE_DRS_SOLN soln)                    */
/*                                                                           */
/*  Hash function for hashing the full signature of soln.                    */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSignatureSetFullHash(KHE_DRS_SOLN soln)
{
  return KheDrsSignatureSetFullHash(&soln->sig_set);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSignatureSetFullHashUntyped(void *p)                       */
/*                                                                           */
/*  Untyped version of KheDrsSolnSignatureSetHash.                           */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSignatureSetFullHashUntyped(void *p)
{
  return KheDrsSolnSignatureSetFullHash((KHE_DRS_SOLN) p);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSignatureSetPartialHash(KHE_DRS_SOLN soln)                 */
/*                                                                           */
/*  Hash function for hashing the partial signature of soln, that is, only   */
/*  the parts of it whose dominance tests are equality (or loose equality).  */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSignatureSetPartialHash(KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY day;
  day = KheDrsSolnDay(soln, drs);
  return KheDrsSignerSetPartialHashSignature(day->signer_set, &soln->sig_set);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSignatureSetFullEqual(KHE_DRS_SOLN soln1,                 */
/*    KHE_DRS_SOLN soln2)                                                    */
/*                                                                           */
/*  Return true if the signatures of soln1 and soln2 are equal.  The lengths */
/*  of their signatures must be equal.                                       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSignatureSetFullEqual(KHE_DRS_SOLN soln1,
  KHE_DRS_SOLN soln2)
{
  return KheDrsSignatureSetFullEqual(&soln1->sig_set, &soln2->sig_set);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSignatureSetFullEqualUntyped(void *p1, void *p2)          */
/*                                                                           */
/*  Untyped version of KheDrsSolnAssignmentsEqual.                           */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSignatureSetFullEqualUntyped(void *p1, void *p2)
{
  return KheDrsSolnSignatureSetFullEqual((KHE_DRS_SOLN) p1, (KHE_DRS_SOLN) p2);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSignatureSetPartialEqual(KHE_DRS_SOLN soln1,              */
/*    KHE_DRS_SOLN soln2, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return true if the signatures of soln1 and soln2 are partially equal,    */
/*  that is, the entries whose dominance tests are equality (or loose        */
/*  equality) are equal.                                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSignatureSetPartialEqual(KHE_DRS_SOLN soln1,
  KHE_DRS_SOLN soln2, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY day;
  day = KheDrsSolnDay(soln1, drs);
  return KheDrsSignerSetPartialEqualSignature(day->signer_set,
    &soln1->sig_set, &soln2->sig_set);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSignatureSetDebug(KHE_DRS_SOLN soln, FILE *fp)            */
/*                                                                           */
/*  Debug print of the signature of soln onto fp (no verbosity or indent).   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSignatureSetDebug(KHE_DRS_SOLN soln, FILE *fp)
{
  KheDrsSignatureSetDebug(&soln->sig_set, 1, -1, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSignatureSetDebugUntyped(void *p, FILE *fp)               */
/*                                                                           */
/*  Untyped version of KheDrsSolnSignatureSetDebug.                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSignatureSetDebugUntyped(void *p, FILE *fp)
{
  KheDrsSolnSignatureSetDebug((KHE_DRS_SOLN) p, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsDistanceToAncestor(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2)     */
/*                                                                           */
/*  Return the distance to the first common ancestor.  If soln1 == soln2     */
/*  this will be 0; otherwise, if they have the same parent solution, it     */
/*  will be 1; and so on.  The distance must be the same from each soln.     */
/*                                                                           */
/*****************************************************************************/

#if TESTING
static int KheDrsDistanceToAncestor(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int res;
  KHE_DRS_DAY day1, day2;
  day1 = KheDrsSolnDay(soln1, drs);
  day2 = KheDrsSolnDay(soln2, drs);
  if( DEBUG32 )
    fprintf(stderr, "  KheDrsDistanceToAncestor(%p day %s, %p day %s)\n",
      (void *) soln1, KheDrsDayId(day1), (void *) soln2, KheDrsDayId(day2));
  res = 0;
  while( soln1 != soln2 )
    soln1 = soln1->prev_soln, soln2 = soln2->prev_soln, res++;
  return res;
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnDoDominates(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,       */
/*    KHE_DRS_SIGNER_SET signer_set, KHE_COST trie_extra_cost,               */
/*    int trie_start_depth, int *dom_test_count,                             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Do the actual work of KheDrsSolnDomninates.                              */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnDoDominates(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,
  KHE_DRS_SIGNER_SET signer_set, KHE_COST trie_extra_cost,
  int trie_start_depth, int *dom_test_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  bool res;  KHE_COST avail_cost;
  if( fp != NULL )
    fprintf(fp, "%*s[ KheDrsSolnDoDominates(soln1 %.5f, soln2 %.5f, day %s)\n",
      indent, "", KheCostShow(KheDrsSolnCost(soln1)),
      KheCostShow(KheDrsSolnCost(soln2)),
      KheDrsDayId(KheDrsSolnDay(soln1, drs)));
      /* signer_set->day)); */
  else
    *dom_test_count += 1;
  res = KheDrsSignerSetDominates(signer_set, &soln1->sig_set,
    &soln2->sig_set, trie_extra_cost, trie_start_depth,
    soln1->prev_soln == soln2->prev_soln, drs, &avail_cost,
    verbosity, indent, fp);
  if( avail_cost == 0 )
    res = (soln1->durns_squared >= soln2->durns_squared);
  /* ***
  if( avail_cost == 0 )
    res = soln1->total_durn >= soln2->total_durn;
  *** */
  if( fp != NULL )
  {
    KheDrsDominanceDebugSpacer(indent + 2, fp);
    fprintf(fp, "%*s] KheDrsSolnDoDominates returning %s\n", indent, "",
      bool_show(res));
  }
#if TESTING
  else if( res )
  {
    int dist = KheDrsDistanceToAncestor(soln1, soln2, drs);
    HaArrayFill(drs->ancestor_freq, dist + 1, 0);
    HaArray(drs->ancestor_freq, dist)++;
  }
#endif
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnDominates(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,         */
/*    KHE_DRS_SIGNER_SET signer_set, int *dom_test_count,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return true if soln1 dominates soln2.                                    */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSolnDominatesDebug(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,
  KHE_DRS_SIGNER_SET signer_set, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp);

static bool KheDrsSolnDominates(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,
  KHE_DRS_SIGNER_SET signer_set, int *dom_test_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  return KheDrsSolnDoDominates(soln1, soln2, signer_set, 0, 0,
    dom_test_count, drs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnDominatesDebug(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,    */
/*    KHE_DRS_SIGNER_SET signer_set, KHE_DYNAMIC_RESOURCE_SOLVER drs,        */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of what KheDrsSolnDominates is doing with these arguments.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnDominatesDebug(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,
  KHE_DRS_SIGNER_SET signer_set, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  KHE_COST avail_cost;
  fprintf(fp, "%*s[ KheDrsSolnDominatesDebug(soln1 %.5f, soln2 %.5f, day %s)\n",
    indent, "", KheCostShow(KheDrsSolnCost(soln1)),
    KheCostShow(KheDrsSolnCost(soln2)),
    KheDrsDayId(KheDrsSolnDay(soln2, drs)));
  KheDrsSignerSetDominates(signer_set, &soln1->sig_set,
    &soln2->sig_set, 0, 0, soln1->prev_soln == soln2->prev_soln, drs,
    &avail_cost, verbosity, indent + 2, fp);
  KheDrsDominanceDebugSpacer(indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnOverWrite(KHE_DRS_SOLN dest_soln, KHE_DRS_SOLN src_soln)  */
/*                                                                           */
/*  Overwrite src_soln onto dest_soln.  Their days and signatures are known  */
/*  to be equal, so those don't need to be touched; and priqueue_index has   */
/*  to be left unchanged.                                                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnOverWrite(KHE_DRS_SOLN dest_soln, KHE_DRS_SOLN src_soln)
{
  int i;
  dest_soln->prev_soln = src_soln->prev_soln;
  HaArrayClear(dest_soln->prev_tasks);
  HaArrayAppend(dest_soln->prev_tasks, src_soln->prev_tasks, i);
  dest_soln->sig_set.cost = KheDrsSolnCost(src_soln);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnResourceIsAssigned(KHE_DRS_SOLN soln,                     */
/*    KHE_DRS_RESOURCE dr, KHE_DRS_TASK_ON_DAY *dtd)                         */
/*                                                                           */
/*  If dr is assigned in soln->prev_tasks, return true and set *dtd to what  */
/*  it is assigned to; otherwise return false and set *dtd to NULL.          */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnResourceIsAssigned(KHE_DRS_SOLN soln,
  KHE_DRS_RESOURCE dr, KHE_DRS_TASK_ON_DAY *dtd)
{
  if( soln->prev_soln == NULL )
  {
    /* this is the root solution, so there can be no assignment */
    return *dtd = NULL, false;
  }
  else
  {
    /* non-root solution, get assignment from soln->prev_tasks */
    *dtd = HaArray(soln->prev_tasks, dr->open_resource_index);
    return *dtd != NULL;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePrintCol(char *str, bool rule_at_right, FILE *fp)                */
/*                                                                           */
/*  Print str (truncated if required) onto fp in a column.  If rule_at_right */
/*  is true, include a vertical rule at the right.                           */
/*                                                                           */
/*****************************************************************************/

static void KhePrintCol(char *str, bool rule_at_right, FILE *fp)
{
  int i;  char *p;

  /* print initial margin */
  fprintf(fp, " ");

  /* print exactly COL_WIDTH - 3 characters, starting with str */
  for( i = 0, p = str;  i < COL_WIDTH - 3;  i++ )
  {
    if( *p != '\0' )
      fprintf(fp, "%c", *p), p++;
    else
      fprintf(fp, " ");
  }

  /* print final margin, optionally including a rule */
  fprintf(fp, rule_at_right ? " |" : "  ");
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePrintRule(int col_count, int indent, FILE *fp)                   */
/*                                                                           */
/*  Print a rule, wide enough for col_count columns.                         */
/*                                                                           */
/*****************************************************************************/

static void KhePrintRule(int col_count, int indent, FILE *fp)
{
  int i, j;

  /* indent */
  fprintf(fp, "%*s", indent, "");

  /* header column */
  fprintf(fp, "%*s+", COL_WIDTH - 1, "");

  /* ordinary columns */
  for( i = 0;  i < col_count;  i++ )
  {
    for( j = 1;  j < COL_WIDTH;  j++ )
      fprintf(fp, "-");
    fprintf(fp, "+");
  }

  /* final newline */
  fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  char *ShortName(char *name, char *prefix)                                */
/*                                                                           */
/*  Return name with prefix removed from the front, if present.              */
/*                                                                           */
/*****************************************************************************/

static char *ShortName(char *name, char *prefix)
{
  if( strstr(name, prefix) != name )
    return name;
  else
  {
    name = &name[strlen(prefix)];
    if( name[0] == ':' )
      return &name[1];
    else
      return name;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnDebugResourceTimetable(KHE_DRS_RESOURCE dr,               */
/*    KHE_DRS_SOLN soln, int first_day_index, int last_day_index,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of the full timetable of dr from first_day_index to          */
/*  last_day_index inclusive.                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnDebugResourceTimetable(KHE_DRS_RESOURCE dr,
  KHE_DRS_SOLN soln, int first_day_index, int last_day_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SOLN soln2;  KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_RESOURCE_ON_DAY drd;
  int i;  KHE_DRS_DAY soln_day, day;

  /* indent */
  fprintf(fp, "%*s", indent, "");

  /* header column */
  KhePrintCol(KheDrsResourceId(dr), true, fp);

  /* ordinary columns */
  soln_day = KheDrsSolnDay(soln, drs);
  for( i = first_day_index;  i <= last_day_index;  i++ )
  {
    day = HaArray(drs->all_days, i);
    if( day->open_day_index == -1 )
    {
      /* a closed day, get timetable from dr */
      drd = HaArray(dr->days, i);
      dtd = drd->closed_dtd;
      if( dtd == NULL )
        KhePrintCol("", true, fp);
      else
	KhePrintCol(ShortName(KheTaskId(dtd->task), KheDrsDayId(day)),
	  KheDrsTaskOnDayIsLast(dtd), fp);
    }
    else if( soln_day==NULL || day->open_day_index > soln_day->open_day_index )
    {
      /* an open day whose assignment is not yet known */
      KhePrintCol("?", true, fp);
    }
    else
    {
      /* an open day whose assignment is known */
      for( soln2 = soln;  soln2 != NULL;  soln2 = soln2->prev_soln )
	if( KheDrsSolnDay(soln2, drs) == day )
	  break;
      HnAssert(soln2 != NULL,
	"KheDrsSolnDebugResourceTimetable internal error 1");
      HnAssert(0 <= dr->open_resource_index &&
	dr->open_resource_index < HaArrayCount(soln2->prev_tasks),
	"KheDrsSolnDebugResourceTimetable internal error 2");
      dtd = HaArray(soln2->prev_tasks, dr->open_resource_index);
      if( dtd == NULL )
        KhePrintCol("", true, fp);
      else
	KhePrintCol(ShortName(KheTaskId(dtd->task), KheDrsDayId(day)),
	  KheDrsTaskOnDayIsLast(dtd), fp);
    }
  }

  /* final newline */
  fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnDebugTimetable(KHE_DRS_SOLN soln, int first_day_index,    */
/*    int last_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of the timetables of the open resources from first_day_index */
/*  to last_day_index inclusive.                                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnDebugTimetable(KHE_DRS_SOLN soln, int first_day_index,
  int last_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  int i;  KHE_DRS_DAY day;  KHE_DRS_RESOURCE dr;  char buff[200];

  /* header row and first rule */
  fprintf(fp, "%*s", indent + COL_WIDTH, "");
  for( i = first_day_index;  i <= last_day_index;  i++ )
  {
    day = HaArray(drs->all_days, i);
    if( day->open_day_index != -1 )
    {
      snprintf(buff, 200, "*%s*", KheDrsDayId(day));
      KhePrintCol(buff, false, fp);
    }
    else
      KhePrintCol(KheDrsDayId(day), false, fp);
  }
  fprintf(fp, "\n");
  KhePrintRule(last_day_index - first_day_index + 1, indent, fp);

  /* resource rows and other rules */
  KheDrsResourceSetForEach(drs->open_resources, dr, i)
  {
    KheDrsSolnDebugResourceTimetable(dr, soln, first_day_index, last_day_index,
      drs, verbosity, indent, fp);
    KhePrintRule(last_day_index - first_day_index + 1, indent, fp);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnDebug(KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of soln onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnDebug(KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  int i, j;
  for( i = 0;  i < HaArrayCount(drs->all_days) - 1;  i += COL_COUNT )
  {
    fprintf(fp, "\n");  /* even the first block has a preceding blank line */
    j = min(i + COL_COUNT - 1, HaArrayCount(drs->all_days) - 1);
    KheDrsSolnDebugTimetable(soln, i, j, drs, verbosity, indent, fp);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_MTASK_SOLN"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_MTASK_SOLN KheDrsMTaskSolnMake(KHE_DRS_SIGNATURE sig,            */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_MTASK dmt,                        */
/*    KHE_DRS_TASK_ON_DAY fixed_dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)        */
/*                                                                           */
/*  Make a new, empty mtask soln object with these attributes.               */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_MTASK_SOLN KheDrsMTaskSolnMake(KHE_DRS_SIGNATURE sig,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_MTASK dmt,
  KHE_DRS_TASK_ON_DAY fixed_dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_MTASK_SOLN res;
  if( HaArrayCount(drs->mtask_soln_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->mtask_soln_free_list);
    HaArrayClear(res->skip_assts);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->skip_assts, drs->arena);
  }
  HnAssert(sig != NULL, "KheDrsMTaskSolnMake internal error");
  res->sig = sig;
  KheDrsSignatureRefer(sig);
  res->resource_on_day = drd;
  res->mtask = dmt;
  res->fixed_task_on_day = fixed_dtd;
  res->skip_count = 0;
  res->random = (23 * KheResourceInstanceIndex(drd->encl_dr->resource) +
    89 * KheSolnDiversifier(drs->soln)) / 511;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMTaskSolnFree(KHE_DRS_MTASK_SOLN dms,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free asst.                                                               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMTaskSolnFree(KHE_DRS_MTASK_SOLN dms,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSignatureUnRefer(dms->sig, drs);
  dms->sig = NULL;
  HaArrayAddLast(drs->mtask_soln_free_list, dms);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT KheDrsMTaskSolnShift(KHE_DRS_MTASK_SOLN dms)               */
/*                                                                           */
/*  Return the shift that dms assigns, or NULL if dms is for a free day.     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SHIFT KheDrsMTaskSolnShift(KHE_DRS_MTASK_SOLN dms)
{
  if( dms->mtask != NULL )
    return dms->mtask->encl_shift;
  else if( dms->fixed_task_on_day != NULL )
    return dms->fixed_task_on_day->encl_dt->encl_dmt->encl_shift;
  else
    return NULL;
}


/*****************************************************************************/
/*                                                                           */ 
/*  KHE_DRS_SIGNATURE KheDrsMTaskSolnSignature(KHE_DRS_MTASK_SOLN dms)       */
/*                                                                           */
/*  Return the signature of dms.                                             */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNATURE KheDrsMTaskSolnSignature(KHE_DRS_MTASK_SOLN dms)
{
  HnAssert(dms->sig != NULL, "KheDrsMTaskSolnSignature internal error");
  return dms->sig;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsMTaskSolnCmp(const void *t1, const void *t2)                   */
/*                                                                           */
/*  Comparison function for sorting an array of mtask soln objects by        */
/*  increasing cost.                                                         */
/*                                                                           */
/*****************************************************************************/

static int KheDrsMTaskSolnCmp(const void *t1, const void *t2)
{
  KHE_DRS_MTASK_SOLN dms1 = * (KHE_DRS_MTASK_SOLN *) t1;
  KHE_DRS_MTASK_SOLN dms2 = * (KHE_DRS_MTASK_SOLN *) t2;
  KHE_DRS_SIGNATURE sig1, sig2;  int cmp;
  sig1 = KheDrsMTaskSolnSignature(dms1);
  sig2 = KheDrsMTaskSolnSignature(dms2);
  cmp = KheCostCmp(sig1->cost, sig2->cost);
  if( cmp != 0 )
    return cmp;
  else
    return dms1->random - dms2->random;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMTaskSolnDebug(KHE_DRS_MTASK_SOLN dms, int verbosity,         */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dms with the given verbosity and indent.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMTaskSolnDebug(KHE_DRS_MTASK_SOLN dms, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_SIGNATURE sig;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "<%s on %s: ",
    KheDrsResourceId(dms->resource_on_day->encl_dr),
    KheDrsDayId(dms->resource_on_day->day));
  if( dms->mtask != NULL )
    KheDrsMTaskDebug(dms->mtask, NULL, 1, -1, fp);
  else if( dms->fixed_task_on_day != NULL )
    KheDrsTaskOnDayDebug(dms->fixed_task_on_day, 1, -1, fp);
  else
    fprintf(fp, "(free)");
  if( HaArrayCount(dms->skip_assts) > 0 )
    fprintf(fp, ", skip_assts %d", HaArrayCount(dms->skip_assts));
  if( dms->skip_count > 0 )
    fprintf(fp, ", skip_count %d", dms->skip_count);
  if( verbosity >= 3 )
  {
    fprintf(fp, ", ");
    sig = KheDrsMTaskSolnSignature(dms);
    KheDrsSignatureDebug(sig, verbosity, -1, fp);
  }
  fprintf(fp, ">");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_MTASK_SOLN - dominance"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsMTaskSolnDominates(KHE_DRS_MTASK_SOLN dms_r_c1,               */
/*    KHE_DRS_MTASK_SOLN dms_r_c2, KHE_DYNAMIC_RESOURCE_SOLVER drs,          */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Return true if S + dms_r_c1 dominates S + dms_r_c2.                      */
/*                                                                           */
/*  If fp != NULL, produce debug output while doing this.                    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsMTaskSolnDominates(KHE_DRS_MTASK_SOLN dms_r_c1,
  KHE_DRS_MTASK_SOLN dms_r_c2, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  KHE_COST avail_cost;  int m;  bool res;
  KHE_DRS_RESOURCE dr;  KHE_DRS_MTASK dmt_r_c1, dmt_r_c2;
  dr = dms_r_c1->resource_on_day->encl_dr;
  m = KheDrsResourceSetCount(drs->open_resources);
  avail_cost = 0;
  if( fp != NULL )
  {
    fprintf(fp, "%*s[ KheDrsMTaskSolnDominates(", indent, "");
    KheDrsMTaskSolnDebug(dms_r_c1, 2, -1, fp);
    fprintf(fp, ", ");
    KheDrsMTaskSolnDebug(dms_r_c2, 2, -1, fp);
    fprintf(fp, ", drs)\n");
  }
  dmt_r_c1 = dms_r_c1->mtask;
  dmt_r_c2 = dms_r_c2->mtask;
  res = KheDrsMTaskOneExtraAvailable(dmt_r_c1, m) &&
    KheDrsMTaskMinCost(dmt_r_c1, KHE_DRS_ASST_OP_UNASSIGN, dr,
      NULL, m, &avail_cost, verbosity, indent, fp) &&
    KheDrsMTaskMinCost(dmt_r_c2, KHE_DRS_ASST_OP_ASSIGN, dr,
      NULL, m, &avail_cost, verbosity, indent, fp) &&
    KheDrsSignerDominates(dms_r_c1->resource_on_day->signer,
      KheDrsMTaskSolnSignature(dms_r_c1),
      KheDrsMTaskSolnSignature(dms_r_c2),
      &avail_cost, verbosity, indent, fp);
  if( DEBUG40 )
    fprintf(stderr, "] KheDrsMTaskSolnDominates returning %s "
      "(avail_cost %.5f)\n", bool_show(res), KheCostShow(avail_cost));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMTaskSolnDominanceInit(KHE_DYNAMIC_RESOURCE_SOLVER drs)     